package com.threerings.protobuf.io;

import java.io.IOException;
import java.lang.reflect.Method;
import java.util.Map;

import org.apache.commons.collections4.EnumerationUtils;

import com.google.common.collect.Maps;
import com.google.protobuf.Any;
import com.google.protobuf.ByteString;
import com.google.protobuf.Int32Value;
import com.google.protobuf.Int64Value;
import com.google.protobuf.Message;
import com.google.protobuf.ProtocolMessageEnum;
import com.google.protobuf.StringValue;
import com.hexnova.narya.util.AnyMessage;
import com.samskivert.util.Logger;
import com.threerings.io.Streamable;

public class ProtobufRegistry {
	static Logger logger = Logger.getLogger(ProtobufRegistry.class);
	private final static Map<Integer,Class<? extends Streamable>> streamableMap = Maps.newHashMap();
	private final static Map<Integer,Class<? extends Message>> messageMap = Maps.newHashMap();
	private final static Map<Class<? extends Message>,Integer> messageType = Maps.newHashMap();
	private final static Map<Class<? extends Streamable>,Integer> streamableType = Maps.newHashMap();
	
	
	private final static Map<Integer,Class<? extends ProtocolMessageEnum>> messageEnumMap = Maps.newHashMap();
	private final static Map<Class<? extends ProtocolMessageEnum>,Integer> messageEnumType = Maps.newHashMap();
	private final static Map<Integer,Class<? extends Enum>> enumMap = Maps.newHashMap();
	private final static Map<Class<? extends Enum>,Integer> enumType = Maps.newHashMap();
	
	
	private final static Map<String, Class<? extends Message>> stringMessage= Maps.newHashMap();
	private final static Map<Class,Method> builderMethodMap = Maps.newHashMap();
	
	private final static Map<Class,ConstructorHelper> constructorHelperMap = Maps.newHashMap();
	
	static {
		ProtobufHelper.registType();
	}
	
	private ProtobufRegistry(){}
	
	public static void registEnum(int key ,Class<? extends Enum> value,Class<? extends ProtocolMessageEnum> msg) {
		messageEnumMap.put(key, msg);
		messageEnumType.put(msg,key);
		enumMap.put(key, value);
		enumType.put(value,key);
	}
	
	public static void regist(int key ,Class<? extends Streamable> value,Class<? extends Message> msg) {
		
		if(streamableMap.containsKey(key)) {
			logger.warning("duplicated key", "key",key);
		}
		streamableMap.put(key, value);
		streamableType.put(value, key);
		
		messageMap.put(key, msg);
		messageType.put(msg, key);
		
		try {
			Class mclazz = msg;
			Method method = null;
			if((method = builderMethodMap.get(mclazz)) == null) {
				method = mclazz.getMethod("newBuilder");
				builderMethodMap.put(mclazz, method);
			}
			
			Message.Builder builder = (Message.Builder)method.invoke(null);
			Message messageV3 = builder.build();
		     stringMessage.put(messageV3.getDescriptorForType().getFullName(), msg);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	public static <T extends Message> T newMessage(String suffix,byte[] data) {
		try {
			Class<T> t = (Class<T>)getMessageType(suffix);
			Message.Builder builder = (Message.Builder)t.getMethod("newBuilder").invoke(null);
			builder.mergeFrom(data);
			return (T)builder.build();
		} catch (Exception e) {
			e.printStackTrace();
		}
		return null;
	}
	
	
	public static <T> T createObject(Class target) throws Exception {
		ConstructorHelper helper = constructorHelperMap.get(target);
		if(helper == null) {
			helper = new ConstructorHelper(target);
			constructorHelperMap.put(target, helper);
		}
		
		return helper.createObject();
	}
	
	public static Message.Builder getBuilder(Class streamable) throws Exception {
		Class<? extends Message> mclazz = getProtoMessageByStreamable(streamable);
		Method method = null;
		if((method = builderMethodMap.get(mclazz)) == null) {
			method = mclazz.getMethod("newBuilder");
			builderMethodMap.put(mclazz, method);
		}
		Message.Builder builder = (Message.Builder)method.invoke(null);
		
		return builder;
		
	}
	
	
	/**
	 * 	将anyMessage 转化成 Streamable
	 * @param <T>
	 * @param any
	 * @return
	 */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	public static <T> T fromAnyMessage(com.hexnova.narya.util.AnyMessage any) {
		try {
			//复杂类型为>100
			if(any.getType() >= 50) {
				Class sclazz = getStreamableType(any.getType());
				if(sclazz != null) {
					Class<? extends Message> mclazz = (Class<? extends Message>)getMessageType(any.getType());
					Method method = null;
					if(mclazz != null && (method = builderMethodMap.get(mclazz)) == null) {
						method = mclazz.getMethod("newBuilder");
						builderMethodMap.put(mclazz, method);
					}
					Message.Builder builder = (Message.Builder)method.invoke(null);
					
					builder = builder.mergeFrom(any.getValue());
					
					
					Object stream = createObject(sclazz);
					
					if(stream instanceof ProtobufProvider) {
						((ProtobufProvider) stream).populate(builder.build());
					}else {
						//TODO
					}
					return (T)stream;
				}else {
					Class<? extends Enum> e = enumMap.get(any.getType());
					
					if(e != null) {
						ByteArrayDataInputStream bis = new ByteArrayDataInputStream(any.getValue().toByteArray());
						int value = Integer.valueOf(bis.readInt());
						if(value >= e.getEnumConstants().length || value < 0) {
							return null;
						}
						return (T)e.getEnumConstants()[value];
					}
				}
				
			}else {
				//基础数据类型
				if(any.getType() == 0) {
					return null;
				}
				
				ByteArrayDataInputStream bis = new ByteArrayDataInputStream(any.getValue().toByteArray());
				switch (any.getType()) {
				case 1:
					return (T)any.getValue().toStringUtf8();
				case 2:
					return (T)Byte.valueOf(any.toByteArray()[0]);
				case 3:
					return (T)Integer.valueOf(bis.readInt());
				case 4:
					return (T)Long.valueOf(bis.readLong());
				case 5:
					return (T)Float.valueOf(bis.readFloat());
				case 6:
					return (T)Double.valueOf(bis.readDouble());
				case 7:
					return (T)Boolean.valueOf(bis.readBoolean());
				case 8:
					return (T)any.toByteArray();
				
				default:
					break;
				}
				
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		return null;
	}
	
	@SuppressWarnings("rawtypes")
	public static AnyMessage toAnyMessage(Object obj) {
		if(obj == null) {
			return AnyMessage.newBuilder().build();
		}

		if(obj instanceof ProtobufProvider) {
			try {
				Message message = ((ProtobufProvider) obj).transform();
				if(message == null) {
					return null;
				}
				byte[] bts = message.toByteArray();
				
				int type = getTypeByStreamable(obj.getClass());
				
				AnyMessage.Builder builder = AnyMessage.newBuilder();
				builder.setType(type);
				builder.setValue(ByteString.copyFrom(bts));
				
				return builder.build();
			}catch(Exception e) {
				if(e instanceof RuntimeException) {
					logger.error("toAnyMessage error,class="+obj.getClass(), e);
					throw (RuntimeException)e;
				}else {
					throw new RuntimeException(e);
				}
			}
		}else {
			Class clazz = obj.getClass();
			AnyMessage.Builder builder = AnyMessage.newBuilder();
			
			if(clazz == String.class) {
				builder.setType(1);
				builder.setValue(ByteString.copyFromUtf8((String)obj));
			}else if (clazz == byte.class || clazz == Byte.class) {
				builder.setType(2);
				builder.setValue(ByteString.copyFrom(new byte[] {(byte)obj}));
			}else if (clazz == int.class || clazz == Integer.class) {
				builder.setType(3);
				byte writeBuffer[] = new byte[4];
				int v = (int)obj;
				writeBuffer[0] = (byte)((v >>> 24) & 0xFF);
				writeBuffer[1] = (byte)((v >>> 16) & 0xFF);
				writeBuffer[2] = (byte)((v >>>  8) & 0xFF);
				writeBuffer[3] = (byte)((v >>>  0) & 0xFF);
				builder.setValue(ByteString.copyFrom(writeBuffer));
			}else if (clazz == long.class || clazz == Long.class) {
				builder.setType(4);
				byte writeBuffer[] = new byte[8];
				long v = (long)obj;
				writeBuffer[0] = (byte)(v >>> 56);
		        writeBuffer[1] = (byte)(v >>> 48);
		        writeBuffer[2] = (byte)(v >>> 40);
		        writeBuffer[3] = (byte)(v >>> 32);
		        writeBuffer[4] = (byte)(v >>> 24);
		        writeBuffer[5] = (byte)(v >>> 16);
		        writeBuffer[6] = (byte)(v >>>  8);
		        writeBuffer[7] = (byte)(v >>>  0);
				builder.setValue(ByteString.copyFrom(writeBuffer));
			}else if (clazz == float.class || clazz == Float.class) {
				builder.setType(5);
				byte writeBuffer[] = new byte[4];
				int v = (int)Float.floatToIntBits((float)obj);
				writeBuffer[0] = (byte)((v >>> 24) & 0xFF);
				writeBuffer[1] = (byte)((v >>> 16) & 0xFF);
				writeBuffer[2] = (byte)((v >>>  8) & 0xFF);
				writeBuffer[3] = (byte)((v >>>  0) & 0xFF);
				builder.setValue(ByteString.copyFrom(writeBuffer));
			}else if (clazz == double.class || clazz == Double.class) {
				builder.setType(6);
				byte writeBuffer[] = new byte[8];
				long v = (int)Double.doubleToLongBits((double)obj);
				writeBuffer[0] = (byte)(v >>> 56);
		        writeBuffer[1] = (byte)(v >>> 48);
		        writeBuffer[2] = (byte)(v >>> 40);
		        writeBuffer[3] = (byte)(v >>> 32);
		        writeBuffer[4] = (byte)(v >>> 24);
		        writeBuffer[5] = (byte)(v >>> 16);
		        writeBuffer[6] = (byte)(v >>>  8);
		        writeBuffer[7] = (byte)(v >>>  0);
				builder.setValue(ByteString.copyFrom(writeBuffer));
			}else if (clazz == boolean.class || clazz == Boolean.class) {
				builder.setType(7);
				boolean v = (boolean)obj;
				builder.setValue(ByteString.copyFrom(new byte[] {v ? (byte)1 : 0}));
			}else if (clazz == byte[].class || clazz == Byte[].class) {
				builder.setType(8);
				builder.setValue(ByteString.copyFrom((byte[])obj));
			}else if (clazz.isEnum()) {
				Integer type = enumType.get(clazz);
				if(type == null) {
					return null;
				}
				builder.setType(type.intValue());
				byte writeBuffer[] = new byte[4];
				int v = ((Enum)obj).ordinal();
				writeBuffer[0] = (byte)((v >>> 24) & 0xFF);
				writeBuffer[1] = (byte)((v >>> 16) & 0xFF);
				writeBuffer[2] = (byte)((v >>>  8) & 0xFF);
				writeBuffer[3] = (byte)((v >>>  0) & 0xFF);
				builder.setValue(ByteString.copyFrom(writeBuffer));
			}else {
				throw new IllegalArgumentException(obj.getClass()+" is not registed");
			}
			
			return builder.build();
		}
	}
	
	/**
	 * 	将anyMessage 转化成 Streamable
	 * @param <T>
	 * @param any
	 * @return
	 */
	public static Object fromAny(com.google.protobuf.Any any) {
		try {
			if(any.is(StringValue.class)) {
				return any.unpack(StringValue.class).getValue();
			}else if(any.is(Int32Value.class)) {
				return any.unpack(Int32Value.class).getValue();
			}else if(any.is(Int64Value.class)) {
				return any.unpack(Int64Value.class).getValue();
			}
			return null;
		} catch (Exception e) {
			e.printStackTrace();
		}
		return null;
	}
	
	public static com.google.protobuf.Any toAny(Object value) {
		try {
			if(value instanceof String) {
				return Any.pack(StringValue.of((String)value));
			}else if(value instanceof Integer) {
				return Any.pack(Int32Value.of((Integer)value));
			}else if(value instanceof Long) {
				return Any.pack(Int64Value.of((Long)value));
			}
			return null;
		} catch (Exception e) {
			e.printStackTrace();
		}
		
		return null;
	}
	
	public static <T extends Message> T newMessage(int type,byte[] data) {
		try {
			Class<T> t = (Class<T>)getMessageType(type);
			Message.Builder builder = (Message.Builder)t.getMethod("newBuilder").invoke(null);
			builder.mergeFrom(data);
			return (T)builder.build();
		} catch (Exception e) {
			e.printStackTrace();
		}
		return null;
	}
	
	
	public static <T> T transform(Message message) throws IOException {
		Class t = getStreamableByMessage(message.getClass());
		T obj;
		
		try {
			obj = createObject(t);
			if(obj instanceof ProtobufProvider) {
				((ProtobufProvider) obj).populate(message);
			}
			
			return obj;
		} catch (Exception e) {
			new IOException(e);
		}
		
		return null;
	}
	
	public static Class<? extends Streamable> getStreamableType(int key) {
		return streamableMap.get(key);
	}
	
	public static Class<? extends Message> getMessageType(int key) {
		return messageMap.get(key);
	}
	
	public static Class<? extends Message> getMessageType(String key) {
		return stringMessage.get(key);
	}
	
	public static int getTypeByStreamable(Class type) {
		Integer value = null;
		
		if((value = streamableType.get(type)) == null) {
			throw new RuntimeException("type="+type.getName() +",not registed");
		}
		return value;
	}
	
	public static Class<? extends Message> getProtoMessageByStreamable(Class type) {
		int key =  streamableType.get(type);
		return getMessageType(key);
	}
	
	public static Class<? extends Streamable> getStreamableByMessage(Class type) {
		
		Integer value = null;
		if((value = messageType.get(type)) == null) {
			throw new RuntimeException("type="+type.getName() +",not registed");
		}
		return getStreamableType(value);
	}
	
	public static int getTypeByMessage(Class type) {
		return messageType.get(type);
	}
}

