//
// $Id$
//
// Narya library - tools for developing networked games
// Copyright (C) 2002-2012 Three Rings Design, Inc., All Rights Reserved
// http://code.google.com/p/narya/
//
// This library is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License as published
// by the Free Software Foundation; either version 2.1 of the License, or
// (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package com.threerings.protobuf.io;

import static com.threerings.NaryaLog.log;

import java.io.DataInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Map;

import com.google.common.collect.Maps;
import com.samskivert.util.StringUtil;
import com.threerings.io.ObjectInputStream;
import com.threerings.io.ObjectOutputStream;
import com.threerings.io.Streamable;

/**
 * Used to read {@link Streamable} objects from an {@link InputStream}.  Other common object types
 * are supported as well (@see {@link ObjectOutputStream}).
 *
 * @see Streamable
 */
public class ProtobufInputStream extends ObjectInputStream
{
    /**
     * Constructs an object input stream which will read its data from the supplied source stream.
     */
    public ProtobufInputStream (InputStream source)
    {
        super(source);
    }

    /**
     * Reads a {@link Streamable} instance or one of the supported object types from the input
     * stream.
     */
    public Object readObject ()
        throws IOException, ClassNotFoundException
    {
        try {
            // read the class mapping
        	Class<? extends Streamable> clazz = readClass();
            if (clazz == null) {
                if (STREAM_DEBUG) {
                    log.info(hashCode() + ": Read null.");
                }
                return null;
            }

            if (STREAM_DEBUG) {
                log.info(hashCode() + ": Reading with " + clazz.getName() + ".");
            }

            Streamable target = null;
			try {
				target = clazz.newInstance();
			} catch (Exception e) {
				throw new IOException(e.getCause());
			}
            
            readBareObject(target);
            if (STREAM_DEBUG) {
                log.info(hashCode() + ": Reading with " + clazz.getName() + ".obj="+target);
            }
            // create an instance of the appropriate object
            return target;

        } catch (OutOfMemoryError oome) {
            throw (IOException)new IOException("Malformed object data").initCause(oome);
        }
    }
    
    /**
     * Reads a class mapping from the stream.
     *
     * @return the class mapping, or <code>null</code> to represent a null value.
     */
    protected Class<? extends Streamable> readClass ()
        throws IOException, ClassNotFoundException
    {
        // read in the class code for this instance
        int code = readInt();
        // a zero code indicates a null value
        if (code == 0) {
            return null;

        // if the code is negative, that means that we've never seen it before and class
        // metadata follows
        } else if (code < 0) {
            // first swap the code into positive-land
            code *= -1;
        } 
        
        Class<? extends Streamable> clazz = ProtobufRegistry.getStreamableType(code);

            // sanity check
        if (clazz == null) {
            // this will help with debugging
            log.warning("Internal stream error, no class metadata", "code", code,
                        "ois", this, new Exception());
            String errmsg = "Read object code for which we have no Message class " +
                "metadata [code=" + code + "]";
            //throw new RuntimeException(errmsg);
        }
        return clazz;
    }

   
    /**
     * Reads an object from the input stream that was previously written with {@link
     * ObjectOutputStream#writeBareObject(Object)}.
     *
     * @param object the object to be populated from data on the stream.  It cannot be
     * <code>null</code>.
     */
    public void readBareObject (Object object)
        throws IOException, ClassNotFoundException
    {
    	_current = object;
        // read the instance data
    	try {
	        if(_current instanceof ProtobufProvider) {
	        	com.google.protobuf.Message.Builder builder = ((ProtobufProvider) _current).getBuilder();
	        	//int type = this.readInt();
	        	builder.mergeFrom(this);
	        	
	        	com.google.protobuf.Message message = builder.build();
	        	
	        	((ProtobufProvider) _current).populate(message);
	        }
    	}finally {
    		_current = null;
    	}
    }

	/**
     * Reads the fields of the specified {@link Streamable} instance from the input stream using
     * the default object streaming mechanisms (a call is not made to <code>readObject()</code>,
     * even if such a method exists).
     */
    public void defaultReadObject ()
        throws IOException, ClassNotFoundException
    {
        // sanity check
        if (_current == null) {
            throw new RuntimeException("defaultReadObject() called illegally.");
        }

        // read the instance data
        if(_current instanceof ProtobufProvider) {
        	com.google.protobuf.Message.Builder builder = ((ProtobufProvider) _current).getBuilder();
        	builder.mergeFrom(this);
        	
        	com.google.protobuf.Message message = builder.build();
        	
        	((ProtobufProvider) _current).populate(message);
        }
    }

    @Override
    public String toString ()
    {
        return "[hash=" + hashCode() +
            ", current=" + StringUtil.safeToString(_current)+ "]";
    }
}
