package com.threerings.protobuf.io;

import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;

import com.google.common.base.Defaults;
import com.samskivert.util.ArrayUtil;

/**
 * A streamer that streams the fields of a class.
 */
public class ConstructorHelper {
	/** Constructor. */
	protected ConstructorHelper(Class<?> target) {
		_target = target;
		initConstructor();
	}

	public <T> T createObject() throws IOException, ClassNotFoundException {
		try {
			return (T) _ctor.newInstance(_ctorArgs);

		} catch (InvocationTargetException ite) {
			String errmsg = "Error instantiating object [type=" + _target.getName() + "]";
			throw (IOException) new IOException(errmsg).initCause(ite.getCause());

		} catch (InstantiationException ie) {
			String errmsg = "Error instantiating object [type=" + _target.getName() + "]";
			throw (IOException) new IOException(errmsg).initCause(ie);

		} catch (IllegalAccessException iae) {
			String errmsg = "Error instantiating object [type=" + _target.getName() + "]";
			throw (IOException) new IOException(errmsg).initCause(iae);
		}
	}

	/**
	 * Locates the appropriate constructor for creating instances.
	 */
	protected void initConstructor() {
		// if we have a zero argument constructor, we have to use that one
		for (Constructor<?> ctor : _target.getDeclaredConstructors()) {
			if (ctor.getParameterTypes().length == 0) {
				_ctor = ctor;
				_ctorArgs = ArrayUtil.EMPTY_OBJECT;
				return;
			}
		}

		// otherwise there should be a single non-zero-argument constructor, which we'll
		// call
		// with zero-valued arguments at unstreaming time, which will then be
		// overwritten by
		// Constructor()
		Constructor<?>[] ctors = _target.getDeclaredConstructors();
		if (ctors.length > 1) {
			throw new RuntimeException("Streamable closure classes must have either a zero-argument constructor "
					+ "or a single argument-taking constructor; multiple argument-taking "
					+ "constructors are not allowed [class=" + _target.getName() + "]");
		}
		_ctor = ctors[0];
		_ctor.setAccessible(true);

		// we pass bogus arguments to it (because unstreaming will overwrite our bogus
		// values with the real values)
		Class<?>[] ptypes = _ctor.getParameterTypes();
		_ctorArgs = new Object[ptypes.length];
		for (int ii = 0; ii < ptypes.length; ii++) {
			// this will be the appropriately typed zero, or null
			_ctorArgs[ii] = Defaults.defaultValue(ptypes[ii]);
		}
	}

	/** The class for which this streamer instance is configured. */
	protected Class<?> _target;

	/** The constructor we use to create instances. */
	protected Constructor<?> _ctor;

	/** The arguments we pass to said constructor (empty or all null/zero). */
	protected Object[] _ctorArgs;

}