//
// $Id$
//
// Narya library - tools for developing networked games
// Copyright (C) 2002-2012 Three Rings Design, Inc., All Rights Reserved
// http://code.google.com/p/narya/
//
// This library is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License as published
// by the Free Software Foundation; either version 2.1 of the License, or
// (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package com.threerings.protobuf.io;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.OutputStream;

import com.google.protobuf.Message;
import com.threerings.io.ObjectOutputStream;
import com.threerings.io.Streamable;

/**
 * Used to write {@link Streamable} objects to an {@link OutputStream}.  Other common object types
 * are supported as well: <code>Boolean, Byte, Character, Short, Integer, Long, Float, Double,
 * boolean[], byte[], char[], short[], int[], long[], float[], double[], Object[]</code>.
 *
 * @see Streamable
 */
public class ProtobufOutputStream extends ObjectOutputStream
{
    /**
     * Constructs an object output stream which will write its data to the supplied target stream.
     */
    public ProtobufOutputStream (OutputStream target)
    {
        super(target);
    }

    /**
     * Writes a {@link Streamable} instance or one of the support object types to the output
     * stream.
     */
    public void writeObject (Object object)
        throws IOException
    {
        // if the object to be written is null, simply write a zero
        if (object == null) {
            writeShort(0);
            return;
        }

        // otherwise, write the class mapping, then the bare object
        if(object instanceof ProtobufProvider) {
        	
	        int type = ProtobufRegistry.getTypeByStreamable(object.getClass());
	        this.writeInt(type);
	        writeBareObject(object);
        }else {
        	super.writeObject(object);
        }
    }

    /**
     * Writes a {@link Streamable} instance or one of the support object types <em>without
     * associated class metadata</em> to the output stream. The caller is responsible for knowing
     * the exact class of the written object, creating an instance of such and calling {@link
     * ObjectInputStream#readBareObject(Object)} to read its data from the stream.
     *
     * @param object the object to be written. It cannot be <code>null</code>.
     */
    public void writeBareObject (Object object)
        throws IOException
    {
    	_current = object;
    	try {
    		defaultWriteObject();
    	}finally {
    		_current = null;
    	}
    }

 
    /**
     * Uses the default streamable mechanism to write the contents of the object currently being
     * streamed. This can only be called from within a <code>writeObject</code> implementation in a
     * {@link Streamable} object.
     */
    public void defaultWriteObject ()
        throws IOException
    {
        // sanity check
        if (_current == null) {
            throw new RuntimeException("defaultWriteObject() called illegally.");
        }

//         log.info("Writing default", "cmap", _streamer, "current", _current);

        // write the instance data
        if(_current instanceof ProtobufProvider) {
        	ProtobufProvider provider = (ProtobufProvider) _current;
        	byte[] bts = provider.transform().toByteArray();
        	//this.writeInt(bts.length);
        	this.write(bts);
        }
    }

}
