//
// $Id$
//
// Narya library - tools for developing networked games
// Copyright (C) 2002-2012 Three Rings Design, Inc., All Rights Reserved
// http://code.google.com/p/narya/
//
// This library is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License as published
// by the Free Software Foundation; either version 2.1 of the License, or
// (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package com.threerings.presents.dobj;

import java.io.IOException;

import com.threerings.presents.annotation.Transient;
import com.threerings.protobuf.io.ProtobufProvider;

/**
 * An object removed event is dispatched when an object is removed from an <code>OidList</code>
 * attribute of a distributed object. It can also be constructed to request the removal of an oid
 * from an <code>OidList</code> attribute of an object and posted to the dobjmgr.
 *
 * @see DObjectManager#postEvent
 */
public class ObjectRemovedEvent extends NamedEvent implements ProtobufProvider<com.hexnova.narya.event.data.ObjectRemovedEvent>
{
    /**
     * Constructs a new object removed event on the specified target object with the supplied oid
     * list attribute name and object id to remove.
     *
     * @param targetOid the object id of the object from whose oid list we will remove an oid.
     * @param name the name of the attribute (data member) from which to remove the specified oid.
     * @param oid the oid to remove from the oid list attribute.
     */
    public ObjectRemovedEvent (int targetOid, String name, int oid)
    {
        super(targetOid, name);
        _oid = oid;
    }

    /**
     * Returns the oid that has been removed.
     */
    public int getOid ()
    {
        return _oid;
    }

    @Override
    public boolean alreadyApplied ()
    {
        return _alreadyApplied;
    }

    @Override
    public boolean applyToObject (DObject target)
        throws ObjectAccessException
    {
        if (!_alreadyApplied) {
            OidList list = (OidList)target.getAttribute(_name);
            list.remove(_oid);
        }
        return true;
    }

    @Override
    protected void notifyListener (Object listener)
    {
        if (listener instanceof OidListListener) {
            ((OidListListener)listener).objectRemoved(this);
        }
    }

    @Override
    protected void toString (StringBuilder buf)
    {
        buf.append("OBJREM:");
        super.toString(buf);
        buf.append(", oid=").append(_oid);
    }

    /** Used by {@link DObject} to note if this event has already been applied locally. */
    protected ObjectRemovedEvent setAlreadyApplied (boolean alreadyApplied)
    {
        _alreadyApplied = alreadyApplied;
        return this;
    }

    protected int _oid;
    protected transient boolean _alreadyApplied;
    

	@Override
	public com.hexnova.narya.event.data.ObjectRemovedEvent.Builder getBuilder() {
		if(builder == null) {
			builder = com.hexnova.narya.event.data.ObjectRemovedEvent.newBuilder(); 
		}else {
			builder.clear();
		}
		return builder;
	}

	@Override
	public void populate(com.hexnova.narya.event.data.ObjectRemovedEvent message) throws IOException {
		this._toid = message.getToid();
		this._name = message.getName();
	}

	@Override
	public com.hexnova.narya.event.data.ObjectRemovedEvent transform() throws IOException {
		com.hexnova.narya.event.data.ObjectRemovedEvent.Builder builder = getBuilder();
		builder.setToid(_toid);
		builder.setName(_name);
		return builder.build();
	}
	
	@Transient
	private com.hexnova.narya.event.data.ObjectRemovedEvent.Builder builder = null;
}
