//
// $Id$
//
// Clyde library - tools for developing networked games
// Copyright (C) 2005-2012 Three Rings Design, Inc.
// http://code.google.com/p/clyde/
//
// Redistribution and use in source and binary forms, with or without modification, are permitted
// provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice, this list of
//    conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright notice, this list of
//    conditions and the following disclaimer in the documentation and/or other materials provided
//    with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
// INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
// PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
// INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
// TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

package com.threerings.tudey.server.logic;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.samskivert.util.ArrayUtil;
import com.samskivert.util.Randoms;
import com.threerings.config.ConfigReference;
import com.threerings.math.FloatMath;
import com.threerings.math.Vector2f;
import com.threerings.probs.FloatVariable;
import com.threerings.tudey.config.BehaviorConfig;
import com.threerings.tudey.config.BehaviorConfig.DirectionType;
import com.threerings.tudey.config.BehaviorConfig.WeightedBehavior;
import com.threerings.tudey.data.EntityKey;
import com.threerings.tudey.data.TudeySceneModel;
import com.threerings.tudey.data.actor.Actor;
import com.threerings.tudey.data.actor.Mobile;
import com.threerings.tudey.server.TudeySceneManager;
import com.threerings.tudey.server.TudeySceneManager.ActorObserver;
import com.threerings.tudey.shape.Segment;
import com.threerings.tudey.util.DirectionUtil;

/**
 * Handles the server-side processing for agent behavior.
 */
public abstract class BehaviorLogic extends Logic
{
    /** The maximum path length for following. */
    public static final float MAX_FOLLOW_PATH_LENGTH = 128;

    /**
     * Handles the idle behavior.
     */
    public static class Idle extends BehaviorLogic
    {
        @Override
        public void startup ()
        {
            _agent.stopMoving();
            _agent.clearTargetRotation();
        }
    }

    /**
     * Superclass of the evaluating behaviors.
     */
    public static abstract class Evaluating extends BehaviorLogic
    {
        @Override
        public void startup ()
        {
            advanceEvaluation();
        }

        @Override
        public void tick (int timestamp)
        {
            // if scheduled to do so, evaluate
            if (_agent.canThink() && timestamp >= _nextEvaluation) {
                evaluate();
            }
        }

        @Override
        public void transfer (Logic source, Map<Object, Object> refs)
        {
            super.transfer(source, refs);
            _nextEvaluation = ((Evaluating)source)._nextEvaluation;
        }

        /**
         * Performs an evaluation.  Default implementation simply schedules the next evaluation.
         */
        protected void evaluate ()
        {
            scheduleNextEvaluation();
        }

        /**
         * Schedules the next evaluation.
         */
        protected void scheduleNextEvaluation ()
        {
            _nextEvaluation = _scenemgr.getTimestamp() +
                (int)(((BehaviorConfig.Evaluating)_config).evaluationInterval.getValue() * 1000f);
        }

        /**
         * Postpones the next evaluation until the next rescheduling.
         */
        protected void postponeNextEvaluation ()
        {
            _nextEvaluation = Integer.MAX_VALUE;
        }

        /**
         * Ensures that we will evaluate on the next tick.
         */
        protected void advanceEvaluation ()
        {
            _nextEvaluation = _scenemgr.getTimestamp();
        }

        /** The time for which the next evaluation is scheduled. */
        protected int _nextEvaluation;
    }
    
    public static class Action extends BehaviorLogic {
    	 
    	@Override
         public void startup ()
         {
    		 _action.execute(_scenemgr.getTimestamp(), _agent);
         }
    	
    	@Override
        protected void didInit ()
        {
            super.didInit();
            BehaviorConfig.Action config = (BehaviorConfig.Action)_config;
            _action = _agent.createAction(config.action, _agent);
            if (_action != null) {
            	_action.didInit();
            }
        }
    	
    	/** Our first behavior. */
        protected ActionLogic _action;
    }
    
    /**
     * Handles the base wander behavior.
     */
    public static class ConditionalAction extends Evaluating
    {
    	
    	 @Override
         protected void evaluate ()
         {
             super.evaluate();
             if(_condition.isSatisfied(_agent)) {
            	 _action.execute(_scenemgr.getTimestamp(), _agent);
             }
         }
    	 
    	@Override
        protected void didInit ()
        {
            super.didInit();
            BehaviorConfig.ConditionalAction config = (BehaviorConfig.ConditionalAction)_config;
            _action = _agent.createAction(config.action, _agent);
            if (_action != null) {
            	_action.didInit();
            }
            _condition = _agent.createCondition(config.condition, _agent);
            if (_condition != null) {
            	_condition.didInit();
            }
        }

        /** Our first behavior. */
        protected ActionLogic _action;

        /** Our second behavior. */
        protected ConditionLogic _condition;
    }

    /**
     * Handles the base wander behavior.
     */
    public static class BaseWander extends Evaluating
    {
        @Override
        public void startup ()
        {
            super.startup();
            _startRotating = Integer.MAX_VALUE;
            _startMoving = Integer.MAX_VALUE;
        }

        @Override
        public void tick (int timestamp)
        {
            super.tick(timestamp);
            // start rotating if the time to do so has come
            if (_agent.canRotate() && timestamp >= _startRotating) {
                _startRotating = _startMoving = Integer.MAX_VALUE;
                _agent.setTargetRotation(_rotation);
            }

            // likewise with moving
            if (_agent.canMove() && timestamp >= _startMoving) {
                scheduleNextEvaluation();
                _agent.startMoving();
                _startMoving = Integer.MAX_VALUE;
            }
        }

        @Override
        public void reachedTargetRotation ()
        {
            BehaviorConfig.BaseWander config = (BehaviorConfig.BaseWander)_config;
            int pause = (int)(config.postRotationPause.getValue() * 1000f);
            if (pause == 0) {
                scheduleNextEvaluation();
                _agent.startMoving();
                _startRotating = _startMoving = Integer.MAX_VALUE;
            } else {
                _startMoving = _scenemgr.getTimestamp() + pause;
                _startRotating = Integer.MAX_VALUE;
            }
        }

        @Override
        public void transfer (Logic source, Map<Object, Object> refs)
        {
            super.transfer(source, refs);

            BaseWander wsource = (BaseWander)source;
            _startRotating = wsource._startRotating;
            _rotation = wsource._rotation;
            _startMoving = wsource._startMoving;
        }

        /**
         * Sets the new direction on the agent.
         */
        protected void setDirectionChange (float rotation)
        {
            int pause = (int)
                (((BehaviorConfig.BaseWander)_config).preRotationPause.getValue() * 1000f);
            postponeNextEvaluation();
            _rotation = rotation;
            if (pause == 0) {
                _startRotating = _startMoving = Integer.MAX_VALUE;
                _agent.setTargetRotation(_rotation);
            } else {
                _startRotating = _scenemgr.getTimestamp() + pause;
                _startMoving = Integer.MAX_VALUE;
            }
        }

        /** The time at which we should start rotating. */
        protected int _startRotating = Integer.MAX_VALUE;

        /** The rotation that we will face when we stop pausing. */
        protected float _rotation;

        /** The time at which we should start moving. */
        protected int _startMoving = Integer.MAX_VALUE;
    }

    /**
     * Handles the wander behavior.
     */
    public static class Wander extends BaseWander
    {
        @Override
        public void tick (int timestamp)
        {
            super.tick(timestamp);

            // if we have exceeded the radius and are moving away from the origin, change direction
            Actor actor = _agent.getActor();
            Vector2f trans = _agent.getTranslation();
            if (actor.isSet(Mobile.MOVING) && ((BehaviorConfig.Wander)_config).radius >0 &&
                    trans.distance(_origin) > ((BehaviorConfig.Wander)_config).radius) {
                float angle = FloatMath.atan2(_origin.y - trans.y, _origin.x - trans.x);
                float rotation = _agent.getActor().getRotation();
                if (FloatMath.getAngularDistance(angle, rotation) > FloatMath.HALF_PI) {
                    changeDirection(angle);
                }
            }
        }

        @Override
        public void penetratedEnvironment (Vector2f penetration)
        {

        	// change the direction, using the reflected direction as a base
            float rotation = FloatMath.normalizeAngle(
                _agent.getActor().getRotation() + FloatMath.PI);
            if (penetration.length() > FloatMath.EPSILON) {
                float angle = FloatMath.atan2(penetration.y, penetration.x);
                rotation = FloatMath.normalizeAngle(
                    angle - FloatMath.getAngularDifference(rotation, angle));
            }
            changeDirection(rotation);
        }

        @Override
        public void transfer (Logic source, Map<Object, Object> refs)
        {
            super.transfer(source, refs);

            Wander wsource = (Wander)source;
            _origin.set(wsource._origin);
        }

        @Override
        protected void didInit ()
        {
            super.didInit();
            _origin.set(_agent.getTranslation());
        }

        @Override
        protected void evaluate ()
        {
            super.evaluate();
            changeDirection();
        }

        /**
         * Changes the direction of the agent.
         */
        protected void changeDirection ()
        {
            changeDirection(_agent.getActor().getRotation());
        }

        /**
         * Changes the direction of the agent.
         *
         * @param rotation the rotation to use as a base.
         */
        protected void changeDirection (float rotation)
        {
            //_agent.stopMoving();
            BehaviorConfig.Wander config = (BehaviorConfig.Wander)_config;
            setDirectionChange(FloatMath.normalizeAngle(rotation + config.directionChange.getValue()));
        }

        /** The translation of the actor when initialized. */
        protected Vector2f _origin = new Vector2f();
    }

    /**
     * Handles the wander collision behavior.
     */
    public static class WanderCollision extends BaseWander
    {
        @Override
        public void penetratedEnvironment (Vector2f penetration)
        {
            _action.execute(_scenemgr.getTimestamp(), _agent);
        }

        @Override
        public void transfer (Logic source, Map<Object, Object> refs)
        {
            super.transfer(source, refs);

            WanderCollision wsource = (WanderCollision)source;
            _action = (ActionLogic)refs.get(wsource._action);
        }

        @Override
        protected void didInit ()
        {
            super.didInit();
            _action = createAction(((BehaviorConfig.WanderCollision)_config).action, _agent);
        }

        @Override
        public void shutdown ()
        {
            super.shutdown();
            _action.removed();
        }

        protected ActionLogic _action;
    }

    /**
     * Handles the grid wander behavior.
     */
    public static class GridWander extends BaseWander
    {
        @Override
        public void startup ()
        {
            super.startup();
            if (!((BehaviorConfig.GridWander)_config).evaluationRotate) {
                setDirectionChange(_rotation);
            }
        }

        @Override
        public void penetratedEnvironment (Vector2f penetration)
        {
            BehaviorConfig.GridWander config = (BehaviorConfig.GridWander)_config;

            changeDirection();
        }

        @Override
        protected void evaluate ()
        {
            super.evaluate();
            if (((BehaviorConfig.GridWander)_config).evaluationRotate) {
                changeDirection();
            }
        }

        @Override
        protected void didInit ()
        {
            super.didInit();
            _rotation = _agent.getActor().getRotation();
        }

        /**
         * Changes the direction of the agent.
         */
        protected void changeDirection ()
        {
            _agent.stopMoving();
            float rotation = _rotation;
            switch(((BehaviorConfig.GridWander)_config).gridTurn) {
            case REVERSE:
                rotation += FloatMath.PI;
                break;
            case LEFT:
                rotation += FloatMath.HALF_PI;
                break;
            case RIGHT:
                rotation -= FloatMath.HALF_PI;
                break;
            case RANDOM:
                rotation += Randoms.threadLocal().getBoolean()
                    ? FloatMath.HALF_PI : -FloatMath.HALF_PI;
                break;
            }
            rotation = FloatMath.normalizeAnglePositive(rotation);
            if (rotation > FloatMath.TWO_PI - FloatMath.QUARTER_PI) {
                rotation = 0;
            } else if (rotation > FloatMath.PI + FloatMath.QUARTER_PI) {
                rotation = -FloatMath.HALF_PI;
            } else if (rotation > FloatMath.PI - FloatMath.QUARTER_PI) {
                rotation = FloatMath.PI;
            } else if (rotation > FloatMath.QUARTER_PI) {
                rotation = FloatMath.HALF_PI;
            } else {
                rotation = 0;
            }
            setDirectionChange(rotation);
        }
    }

    /**
     * Base class for behaviors that involve following paths.
     */
    public static abstract class Pathing extends Evaluating
    {
    	protected Vector2f _lastPenetration = new Vector2f();
    	protected Vector2f _tmp = new Vector2f();
    	protected int _lastBedetainedCheckTime = 0;
    	//private Wander wander = null;
    	protected int duringPenetration = 0;
    	protected boolean _penetration = false;
    	
		public void startup() {
			clearPath();
			super.startup();
		}
    	 
		 @Override
        protected void evaluate ()
        {
        	//if(duringPenetration >  _scenemgr.getTimestamp()){
        		//wander.evaluate();
        	//	return;
        	//}
            super.evaluate();
        }
		 
		 protected boolean reachTarget() {
			return false;
		 }
		 
        @Override
        public void tick (int timestamp)
        {
        	super.tick(timestamp);
        	
            if (_path == null || _path.length == 0) {
                return; // nothing to do
            }
            
            // see if we've reached the current node (looping around in case the notification
            // sets us on a new path)
            Vector2f trans = _agent.getTranslation();
            boolean completedPath = false;
            while (_path[_pidx].distance(trans) <= getReachRadius() ) {
            	//System.out.println("current node = "+_pidx+",node="+_path[_pidx]+",path="+Arrays.toString(_path)+",current="+_agent.getRotation());
                if (++_pidx == _path.length) {
                    _agent.stopMoving();
                    _path = null;
                    // If we've already completed a path then just exit to prevent an infinite loop
                    if (completedPath) {
                        return;
                    }
                    completedPath();
                    completedPath = true;
                } else {
                	
                    reachedPathIndex(_pidx - 1);
                }
                if (_path == null || _path.length == 0 ) {
                    return;
                }
            }
            // make sure we're facing the right direction
            Vector2f node = _path[_pidx];
            float rot = FloatMath.atan2(node.y - trans.y, node.x - trans.x);
            
            
            //System.out.println("reachedPathIndex="+_pidx+",node="+_path[_pidx]+", location="+_agent.getTranslation()+", setTargetRotation="+rot+",current="+_agent.getRotation());
            
            BehaviorConfig.Pathing config = ((BehaviorConfig.Pathing)_config);
            if(config.directionType == DirectionType.FREE) {
            	float dist = Math.abs(FloatMath.getAngularDistance(_agent.getRotation(), rot));
	            if (dist > FloatMath.EPSILON) {
	                _agent.setTargetRotation(rot);
	            }
	            
	            if (dist > _moveFaceRange) {
	                _agent.stopMoving();
	            } else {
	                _agent.startMoving();
	            }
            }else if(config.directionType == DirectionType.CARDINAL) {
            	float targetRotation = DirectionUtil.normalize4Direction(rot);
            	float dist = Math.abs(FloatMath.getAngularDistance(_agent.getRotation(), targetRotation));
            	
            	if (dist > FloatMath.EPSILON) {
 	                _agent.setTargetRotation(targetRotation);
 	            }
            	 
            	if (dist > _moveFaceRange) {
	                _agent.stopMoving();
	            } else {
	                _agent.startMoving();
	            }
            }
            
            if(_penetration) {
        		if(duringPenetration<_scenemgr.getTimestamp()) {
        			if(_agent.getTranslation().distance(_lastPenetration) <0.01) {
        				bedetained();
        			}else {
        				_penetration = false;
        			}
            	}
            }
        }

		protected void bedetained() {
			//wander._origin = new Vector2f(_agent.getTranslation());
			_penetration = false;
		}
        
        /**
         * Notifies the behavior that the agent has penetrated its environment during advancement.
         *
         * @param penetration the sum penetration vector.
         */
        public void penetratedEnvironment (Vector2f penetration)
        {
        	super.penetratedEnvironment(penetration);
    		_lastPenetration.set(_agent.getTranslation());
    		if(!_penetration) {
	    		duringPenetration = _scenemgr.getTimestamp() + 1000;
	        	//wander.penetratedEnvironment(penetration);
	        	//this._agent.startMoving();
	        	_penetration = true;
    		}
        }

        
        public void reachedTargetRotation ()
        {
        	if(_path!= null && _pidx <= _path.length -1){
        		_agent.startMoving();
        	}
        }
        
        @Override
        public void transfer (Logic source, Map<Object, Object> refs)
        {
            super.transfer(source, refs);

            Pathing psource = (Pathing)source;
            _path = psource._path;
            _pidx = psource._pidx;
        }
        
        /**
         * Notifies the behavior that the agent has reached its target.
         */
        public void reachedTarget (int timestamp)
        {
        	super.reachedTarget(timestamp);
        	clearPath();
        }

        protected void switchPath() {
        	if(_nextPath != null) {
        		int cidx = -1;
                float cdist = Float.MAX_VALUE;
                for (int jj = 0, mm = _nextPath.length; jj < mm; jj++) {
                    float dist = _nextPath[jj].distanceSquared(_agent.getTranslation());
                    if (dist < cdist) {
                        cidx = jj;
                        cdist = dist;
                    }
                }
                _path = _nextPath;
                _nextPath = null;
                _pidx = cidx;
                _penetration = false;
        	}
        }
        
        /**
         * Sets the path to follow.
         */
        protected void setPath (Vector2f[] path)
        {
			/*
			 * if(path != null && path.length == 0) { this._agent.stopMoving(); }
			 */
        	if (_path == path || Arrays.equals(_path,path)) {
        		return;
        	}
        	
        	if(_path != null) {
        		_nextPath =  path;
        	}else {
        		_path = path;
        		_pidx = 0;
        	}
        	if(true) {
            	return;
            }
        	
            if (_path != path && !Arrays.equals(_path,path)) {
                _path = path;
                _pidx = 0;
                _penetration = false;
                
                
                if(path != null && _path.length > 0) {
	                int cidx = -1;
	                float cdist = Float.MAX_VALUE;
	                for (int jj = 0, mm = path.length; jj < mm; jj++) {
	                    float dist = path[jj].distanceSquared(_agent.getTranslation());
	                    if (dist < cdist) {
	                        cidx = jj;
	                        cdist = dist;
	                    }
	                }
	                
	                _pidx = cidx;
	               
	                /**
	                while(_path != null && _pidx < _path.length - 2) {
	            		Segment segment = new Segment(_path[_pidx],_path[_pidx+1]);
	            		if(segment.contains(_agent.getTranslation())) {
	            			_pidx++;
	            		}else {
	            			break;
	            		}
	                }
	                
	                **/
	                
	                //if(this._agent.getActor().getOriginal().type == 3) {
	                //	System.out.println("_agent="+this.getTranslation().adjustCenter()+",idx="+_pidx+","+Arrays.toString(path));
	                //}
	               
	            }
            }
        }

        /**
         * Clears the path.
         */
        protected void clearPath ()
        {
            if (_path != null) {
                _agent.stopMoving();
                _path = null;
                _penetration = false;
            }
        }

        /**
         * Returns the radius within which we can be consider ourselves to have reached a node
         * (which depends on the actor's speed, since it's possible to overshoot).
         */
        protected float getReachRadius ()
        {
            // radius is the distance we can travel in a single tick
            float speed = ((Mobile)_agent.getActor()).getSpeed();
            return speed / _scenemgr.getTicksPerSecond();
        }

        /**
         * Called when we reach each node in the path (except for the last one, for which we call
         * {@link #completedPath}.
         */
        protected void reachedPathIndex (int idx)
        {
        	switchPath();
            // nothing by default
        }

        /**
         * Called when we complete the set path.
         */
        protected void completedPath ()
        {
            // nothing by default
        }

        @Override
        protected void didInit ()
        {
            super.didInit();
            _moveFaceRange = ((BehaviorConfig.Pathing)_config).moveFaceRange;
            if (_moveFaceRange == 0f) {
                _moveFaceRange = 0.001f;
            }
            
            BehaviorConfig.Wander wanderConfig = new BehaviorConfig.Wander();
            wanderConfig.radius = 0;
            wanderConfig.evaluationInterval = new FloatVariable.Constant(0.5f);

            // create the logic instance
            BehaviorConfig.Original original = wanderConfig.getOriginal(_scenemgr.getConfigManager());
            if (original == null) {
                original = new BehaviorConfig.Original();
            }
            
           // wander = (Wander)_scenemgr.createLogic(original.getLogicClassName());

            // initialize, return the logic
            //wander.init(_scenemgr, original, this._agent);
        }
        
        public void shutdown ()
        {
           super.shutdown();
          // wander.shutdown();
        }
        /** The waypoints of the path being followed. */
        protected Vector2f[] _path,_nextPath;

        /** The index of the next point on the path. */
        protected int _pidx;

        /** The angular range we need to be within before starting to move. */
        protected float _moveFaceRange;
    }

    /**
     * Handles the patrol behavior.
     */
    public static class Patrol extends Pathing implements ActorObserver
    {
    	
        @Override
        public Logic getCurrentTarget ()
        {
            return _currentTarget;
        }

        @Override
        public void transfer (Logic source, Map<Object, Object> refs)
        {
            super.transfer(source, refs);

            Patrol psource = (Patrol)source;
            _target.transfer(psource._target, refs);
            _currentTarget = (Logic)refs.get(psource._currentTarget);
        }

        @Override
        protected void didInit ()
        {
            super.didInit();
            _target = createTarget(((BehaviorConfig.Patrol)_config).target, _agent);
            _scenemgr.addActorObserver(this);
        }
        
        public void shutdown ()
        {
           super.shutdown();
           _scenemgr.removeActorObserver(this);
        }
        
        
        protected PathCandidate findPath(float branchRadius,boolean partial,boolean shortcut) {
        	float nearest = Float.MAX_VALUE;
        	PathCandidate candidate = null;
            for (int ii = 0, nn = _targets.size(); ii < nn; ii++) {
                Logic target = _targets.get(ii);
                Vector2f[] path = target.getPatrolPath();
                if (path == null) {
                	path = _scenemgr.getPathfinder().getPath(
                            _agent, ((BehaviorConfig.Patrol)_config).longest, _agent.getTranslation().x,_agent.getTranslation().y,target.getTranslation().x, target.getTranslation().y, partial, shortcut);
                	//System.out.println("agent translation="+_agent.getTranslation()+",Find path="+Arrays.toString(path));
                	
                	if(path == null) {
                		continue;
                	}
                	PathCandidate p = new PathCandidate(target, path, 0,((BehaviorConfig.Patrol)_config).unidirection);
                	float d = p.getDistance();
                	if(d<nearest) {
                		candidate = p;
                		nearest = d;
                	}
                	_candidates.add(p);
                }else {
                	// find the index of the closest node on the path
                    float cdist = Float.MAX_VALUE;
                    int cidx = -1;
                    for (int jj = 0, mm = path.length; jj < mm; jj++) {
                        float dist = path[jj].distanceSquared(_agent.getTranslation());
                        if (dist < cdist) {
                            cidx = jj;
                            cdist = dist;
                        }
                    }
                    
                    if (cdist <= branchRadius) {
                    	if(((BehaviorConfig.Patrol)_config).nearest) {
                    		PathCandidate p = new PathCandidate(target, path, cidx,((BehaviorConfig.Patrol)_config).unidirection);
                    		float d = p.getDistance();
                    		
                    		if(d<nearest) {
                        		candidate = p;
                        		nearest = d;
                        	}
                    		_candidates.add(p);
                        }else {
                        	_candidates.add(new PathCandidate(target, path, cidx,((BehaviorConfig.Patrol)_config).unidirection));
                        }
                    }
                }
                
            }
            
            return candidate;
        }

        @Override
        protected void evaluate ()
        {
            super.evaluate();

            // determine the square of the branch radius
            float br2;
            BehaviorConfig.Patrol config = ((BehaviorConfig.Patrol)_config);
            if (_path == null) {
                br2 = Float.MAX_VALUE;
            } else {
                float radius = ((BehaviorConfig.Patrol)_config).branchRadius;
                if (radius < 0f) {
                    return; // no possibility of branching
                }
                br2 = radius*radius;
            }
            PathCandidate candidate = null;
            // resolve the target paths
            _target.resolve(_agent, _targets);
            try {
	            candidate = findPath(br2,false,config.shortcut);
	            // pick a candidate at random
	            if (_candidates.isEmpty()) {
	            	if(config.partial && _currentTarget == null) {
	            		candidate = findPath(br2,true,config.shortcut);
	            		if (_candidates.isEmpty()) {
	            			return;
	            		}
	            	}
	            }
            
	            if(!((BehaviorConfig.Patrol)_config).nearest) {
	            	candidate = Randoms.threadLocal().pick(_candidates, null);
	            }
            
            }finally {
            	_candidates.clear();
            	_targets.clear();
            }
            
            // set off on that path
            if (candidate != null) {
                setPath(candidate.getRemainingPath(_agent.getRotation()), candidate.getTarget());
            }
        }


		 protected boolean reachTarget() {
			 Logic logic = getCurrentTarget();
			 if(logic instanceof EntryLogic) {
				 EntryLogic entry = (EntryLogic)logic;
				 if(entry.getEntry() instanceof TudeySceneModel.PathEntry) {
					 return false;
				 }else if ((_agent.getActor().getCollisionMask() & entry.getEntry().getFlags()) == 0) {
					 return entry.getShape().contains(_agent.getTranslation());
				 }else {
					 return false;
				 }
			 }else if(logic instanceof ActorLogic){
				 ActorLogic actor = (ActorLogic)logic;
				 if(!_agent.getActor().canCollide(actor.getActor(), _agent)) {
					 return actor.getShape().contains(_agent.getTranslation());
				 }else {
					 return false;
				 }
			 } else {
				 return false;
			 }
		 }
		 
		 protected void bedetained() {
			super.bedetained();
			clearPath();
		}
        
        /**
         * Returns the radius within which we can be consider ourselves to have reached a node
         * (which depends on the actor's speed, since it's possible to overshoot).
         */
        protected float getReachRadius ()
        {
        	float speed = ((Mobile)_agent.getActor()).getSpeed();
            float radius =  speed / _scenemgr.getTicksPerSecond();
        	return radius;
        }
        
        @Override
        protected void reachedPathIndex (int idx)
        {
            evaluate();
        }

        @Override
		public void actorAdded(ActorLogic logic) {
        	if(logic == _agent) {
				return;
			}
        	if((logic.getCollisionFlags() & _agent.getActor().getCollisionMask()) != 0 || logic.getActor().getOriginal().traversalCosts != null) {
        		clearPath();
        	}
		}

		@Override
		public void actorRemoved(ActorLogic logic) {
			if(logic == _agent) {
				return;
			}
			if((logic.getCollisionFlags() & _agent.getActor().getCollisionMask()) != 0 || logic.getActor().getOriginal().traversalCosts != null) {
				clearPath();
        	}
		}
		
        @Override
        protected void completedPath ()
        {
            _currentTarget = null;
            evaluate();
        }

        protected void clearPath ()
        {
        	super.clearPath();
        	_currentTarget = null;
        }
        
        /**
         * Sets the path to traverse.
         */
        protected void setPath (Vector2f[] path, Logic currentTarget)
        {
            super.setPath(path);
            _currentTarget = currentTarget;
        }

        /** The target to patrol. */
        protected TargetLogic _target;

        /** Holds targets during processing. */
        protected ArrayList<Logic> _targets = Lists.newArrayList();

        /** Holds candidate paths during processing. */
        protected ArrayList<PathCandidate> _candidates = Lists.newArrayList();

        /** The logic corresponding to the current path, if any. */
        protected Logic _currentTarget;

    }

    /**
     * Handles the follow behavior.
     */
    public static class Follow extends Pathing
    {
        @Override
        public Logic getCurrentTarget ()
        {
            return _currentTarget;
        }

        @Override
        public void transfer (Logic source, Map<Object, Object> refs)
        {
            super.transfer(source, refs);
            _target.transfer(((Follow)source)._target, refs);
        }

        @Override
        protected void didInit ()
        {
            super.didInit();
            _target = createTarget(((BehaviorConfig.Follow)_config).target, _agent);
        }
        
        @Override
        protected void evaluate ()
        {
            super.evaluate();

            // find the closest target
            _target.resolve(_agent, _targets);
            Vector2f trans = _agent.getTranslation();
            _currentTarget = null;
            float cdist = Float.MAX_VALUE;
            for (int ii = 0, nn = _targets.size(); ii < nn; ii++) {
                Logic target = _targets.get(ii);
                float dist = target.getTranslation().distanceSquared(trans);
                if (dist < cdist) {
                    _currentTarget = target;
                    cdist = dist;
                }
            }
            _targets.clear();

            // if we're within our distance bounds, stop and face the target
            if (_currentTarget == null) {
                return;
            }
            BehaviorConfig.Follow config = (BehaviorConfig.Follow)_config;
            float min2 = config.minimumDistance*config.minimumDistance;
            float max2 = config.maximumDistance*config.maximumDistance;
            if (FloatMath.isWithin(cdist, min2, max2)) {
                clearPath();
                _agent.face(_currentTarget);
                return;
            }

            // compute a path to the target
            Vector2f[] path = _scenemgr.getPathfinder().getPath(
                _agent, MAX_FOLLOW_PATH_LENGTH, _currentTarget, false, false);
            if (path == null) {
                clearPath();
                _agent.face(_currentTarget);
                return;
            }

            // start out on the path
            setPath(path);
        }

        /** The target to follow. */
        protected TargetLogic _target;

        /** Holds targets during processing. */
        protected ArrayList<Logic> _targets = Lists.newArrayList();

        /** The current target. */
        protected Logic _currentTarget;
    }

    /**
     * Handles the random behavior.
     */
    public static class Random extends Evaluating
    {
        @Override
        public void tick (int timestamp)
        {
            super.tick(timestamp);
            if (_active != null) {
                _active.tick(timestamp);
            }
        }

        @Override
        public void reachedTargetRotation ()
        {
            if (_active != null) {
                _active.reachedTargetRotation();
            }
        }

        @Override
        public void penetratedEnvironment (Vector2f penetration)
        {
            if (_active != null) {
                _active.penetratedEnvironment(penetration);
            }
        }

        @Override
        public Logic getCurrentTarget ()
        {
            return _active == null ? null : _active.getCurrentTarget();
        }

        @Override
        public void transfer (Logic source, Map<Object, Object> refs)
        {
            super.transfer(source, refs);

            BehaviorLogic[] sbehaviors = ((Random)source)._behaviors;
            for (int ii = 0; ii < _behaviors.length; ii++) {
                if (_behaviors[ii] != null) {
                    _behaviors[ii].transfer(sbehaviors[ii], refs);
                }
            }
        }

        /**
         * when entered stasis
         */
        public void enteredStasis ()
        {
        	super.enteredStasis();
        	for (int ii = 0; ii < _behaviors.length; ii++) {
                if (_behaviors[ii] != null) {
                    _behaviors[ii].enteredStasis();
                }
            }
        }
        
        @Override
        public void shutdown ()
        {
            super.shutdown();
            for (int ii = 0; ii < _behaviors.length; ii++) {
                if (_behaviors[ii] != null) {
                    _behaviors[ii].shutdown();
                }
            }
        }

        @Override
        protected void didInit ()
        {
            super.didInit();
            WeightedBehavior[] wbehaviors = ((BehaviorConfig.Random)_config).behaviors;
            _behaviors = new BehaviorLogic[wbehaviors.length];
            _behaviorWeights = Maps.newHashMap();
            for (int ii = 0; ii < wbehaviors.length; ii++) {
                WeightedBehavior wbehavior = wbehaviors[ii];
                _behaviors[ii] = _agent.createBehavior(wbehavior.behavior);
                _behaviorWeights.put(_behaviors[ii], wbehavior.weight);
            }
        }

        @Override
        protected void evaluate ()
        {
            super.evaluate();
            BehaviorLogic nactive = Randoms.threadLocal().pick(_behaviorWeights, null);
            if (nactive == _active) {
                return;
            }
            if ((_active = nactive) != null) {
                _active.startup();
            }
        }

        /** The component behaviors. */
        protected BehaviorLogic[] _behaviors;

        /** The active behavior. */
        protected BehaviorLogic _active;

        /** The behavior weight map. */
        protected Map<BehaviorLogic, Float> _behaviorWeights;
    }

    /**
     * Handles the scripted behavior.
     */
    public static class Scripted extends BehaviorLogic implements Scriptable
    {
        /**
         * Sets the current step.
         */
        public void setCurrentStep (int step, int timestamp)
        {
            _currentStep = Math.min(_steps.length, step);
            _steps[_currentStep].start(timestamp);
        }

        @Override
        public void tick (int timestamp)
        {
            if (_currentStep < _steps.length) {
                if (_start) {
                    _steps[_currentStep].start(timestamp);
                }
                if (_start = _steps[_currentStep].tick(timestamp)) {
                    _currentStep++;
                }
            }
        }

        @Override
        public void reachedTargetRotation ()
        {
            if (_currentStep < _steps.length) {
                _steps[_currentStep].reachedTargetRotation();
            }
        }

        @Override
        public void startup ()
        {
            _currentStep = 0;
            _start = true;
        }

        @Override
        public void suspend ()
        {
            if (_currentStep < _steps.length && !_start) {
                _steps[_currentStep].suspend();
            }
        }

        @Override
        public void shutdown ()
        {
            super.shutdown();
            for (ScriptLogic logic : _steps) {
                logic.shutdown();
            }
        }

        @Override
        public void transfer (Logic source, Map<Object, Object> refs)
        {
            super.transfer(source, refs);

            Scripted ssource = (Scripted)source;
            for (int ii = 0; ii < _steps.length; ii++) {
                _steps[ii].transfer(ssource._steps[ii], refs);
            }
            _currentStep = ssource._currentStep;
            _start = ssource._start;
        }

        @Override
        protected void didInit ()
        {
            super.didInit();

            BehaviorConfig.Scripted config = (BehaviorConfig.Scripted)_config;
            _steps = new ScriptLogic[config.steps.length];
            for (int ii = 0; ii < _steps.length; ii++) {
                _steps[ii] = ScriptLogic.createScriptLogic(
                        _scenemgr, config.steps[ii], _agent, this);
            }
        }

        /** The script logics. */
        protected ScriptLogic[] _steps;

        /** The current step. */
        protected int _currentStep;

        /** If we need to start the next step. */
        protected boolean _start = true;
    }

    /**
     * Handles the combined behavior.
     */
    public static class Combined extends BehaviorLogic
    {
        @Override
        public void startup ()
        {
            if (_first != null) {
                _first.startup();
            }
            if (_second != null) {
                _second.startup();
            }
        }

        @Override
        public void shutdown ()
        {
            super.shutdown();
            if (_first != null) {
                _first.shutdown();
            }
            if (_second != null) {
                _second.shutdown();
            }
        }

        @Override
        public void tick (int timestamp)
        {
            if (_first != null) {
                _first.tick(timestamp);
            }
            if (_second != null) {
                _second.tick(timestamp);
            }
        }

        @Override
        public void reachedTargetRotation ()
        {
            if (_first != null) {
                _first.reachedTargetRotation();
            }
            if (_second != null) {
                _second.reachedTargetRotation();
            }
        }

        @Override
        public void penetratedEnvironment (Vector2f penetration)
        {
            if (_first != null) {
                _first.penetratedEnvironment(penetration);
            }
            if (_second != null) {
                _second.penetratedEnvironment(penetration);
            }
        }

        @Override
        public Logic getCurrentTarget ()
        {
            Logic target = null;
            if (_first != null) {
                target = _first.getCurrentTarget();
            }
            if (_second != null && target == null) {
                target = _second.getCurrentTarget();
            }
            return target;
        }

        @Override
        public void transfer (Logic source, Map<Object, Object> refs)
        {
            super.transfer(source, refs);

            Combined csource = (Combined)source;
            if (_first != null) {
                _first.transfer(csource._first, refs);
            }
            if (_second != null) {
                _second.transfer(csource._second, refs);
            }
        }
        
        /**
         * when entered stasis
         */
        public void enteredStasis ()
        {
        	super.enteredStasis();
        	if (_first != null) {
                _first.enteredStasis();
            }
            if (_second != null) {
                _second.enteredStasis();
            }
        }

        @Override
        protected void didInit ()
        {
            super.didInit();
            BehaviorConfig.Combined config = (BehaviorConfig.Combined)_config;
            _first = _agent.createBehavior(config.first);
            if (_first != null) {
                _first.didInit();
            }
            _second = _agent.createBehavior(config.second);
            if (_second != null) {
                _second.didInit();
            }
        }

        /** Our first behavior. */
        protected BehaviorLogic _first;

        /** Our second behavior. */
        protected BehaviorLogic _second;
    }
    
    /**
     * Handles the Compound behavior.
     */
    public static class Compound extends BehaviorLogic
    {
        @Override
        public void startup ()
        {
        	for(BehaviorLogic logic : _behaviors) {
         		logic.startup();
         	}
        }

        @Override
        public void shutdown ()
        {
            super.shutdown();
            for(BehaviorLogic logic : _behaviors) {
        		logic.shutdown();
        	}
        }

        @Override
        public void tick (int timestamp)
        {
        	for(BehaviorLogic logic : _behaviors) {
        		logic.tick(timestamp);
        	}
        }

        @Override
        public void reachedTargetRotation ()
        {
        	for(BehaviorLogic logic : _behaviors) {
        		logic.reachedTargetRotation();
        	}
        }

        @Override
        public void penetratedEnvironment (Vector2f penetration)
        {
        	for(BehaviorLogic logic : _behaviors) {
        		logic.penetratedEnvironment(penetration);
        	}
        }

        @Override
        public Logic getCurrentTarget ()
        {
            Logic target = null;
            for(BehaviorLogic logic : _behaviors) {
        		target = logic.getCurrentTarget();
				if(target != null) {
					return target;
				}
        	}
            return target;
        }

        @Override
        public void transfer (Logic source, Map<Object, Object> refs)
        {
            super.transfer(source, refs);

            Compound csource = (Compound)source;
            for(int i=0;i<_behaviors.size();i++) {
            	_behaviors.get(i).transfer(csource._behaviors.get(i), refs);
            }
        }
        
        /**
         * when entered stasis
         */
        public void enteredStasis ()
        {
        	super.enteredStasis();
        	for(BehaviorLogic logic : _behaviors) {
        		logic.enteredStasis();
        	}
        }

        @Override
        protected void didInit ()
        {
            super.didInit();
            BehaviorConfig.Compound config = (BehaviorConfig.Compound)_config;
            for(ConfigReference<BehaviorConfig> ref : config.behaviors) {
            	BehaviorLogic logic = _agent.createBehavior(ref);
            	if (logic != null) {
            		logic.didInit();
                }
            	_behaviors.add(logic);
            }
            
        }

        /** Our second behavior. */
        protected List<BehaviorLogic> _behaviors = Lists.newArrayList();
    }

    /**
     * Initializes the logic.
     */
    public void init (TudeySceneManager scenemgr, BehaviorConfig.Original config, AgentLogic agent)
    {
        super.init(scenemgr);
        _config = config;
        _agent = agent;

        // give subclasses a chance to initialize
        didInit();
    }

    /**
     * Starts up the behavior after initialization or suspension.
     */
    public void startup ()
    {
        // nothing by default
    }

    /**
     * Suspends the behavior.
     */
    public void suspend ()
    {
        // nothing by default
    }
    
    /**
     * when entered stasis
     */
    public void enteredStasis ()
    {
    	
    }

    /**
     * Shuts down the behavior when the agent is destroyed.
     */
    public void shutdown ()
    {
        // nothing by default
    }

    /**
     * Ticks the behavior.
     */
    public void tick (int timestamp)
    {
        // nothing by default
    }
    
    /**
     * Notifies the behavior that the agent has reached its target.
     */
    public void reachedTarget (int timestamp)
    {
    	// nothing by default
    }

    /**
     * Notifies the behavior that the agent has reached its target rotation.
     */
    public void reachedTargetRotation ()
    {
        // nothing by default
    }
    
    /**
     * Notifies the behavior that the agent has penetrated its environment during advancement.
     *
     * @param penetration the sum penetration vector.
     */
    public void penetratedEnvironment (Vector2f penetration)
    {
        // nothing by default
    }

    /**
     * Returns the currently targeted logic, if any.
     */
    public Logic getCurrentTarget ()
    {
        return null;
    }

    /**
     * check the behavior is ready
     * @return
     */
    public boolean isReady(int timestamp) {
    	return true;
    }
    
    public boolean isDoing(int timestamp) {
    	return false;
    }
    
    @Override
    public boolean isActive ()
    {
        return _agent.isActive();
    }

    @Override
    public EntityKey getEntityKey ()
    {
        return _agent.getEntityKey();
    }

    @Override
    public Vector2f getTranslation ()
    {
        return _agent.getTranslation();
    }

    @Override
    public float getRotation ()
    {
        return _agent.getRotation();
    }
    
    public Logic getSource() {
    	return _agent;
    }

    /**
     * Override to perform custom initialization.
     */
    protected void didInit ()
    {
        // nothing by default
    }

    /** The behavior configuration. */
    protected BehaviorConfig.Original _config;

    /** The controlled agent. */
    protected AgentLogic _agent;
}