//
// $Id$
//
// Clyde library - tools for developing networked games
// Copyright (C) 2005-2012 Three Rings Design, Inc.
// http://code.google.com/p/clyde/
//
// Redistribution and use in source and binary forms, with or without modification, are permitted
// provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice, this list of
//    conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright notice, this list of
//    conditions and the following disclaimer in the documentation and/or other materials provided
//    with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
// INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
// PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
// INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
// TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

package com.threerings.swing.filetree;

import static com.threerings.ClydeLog.log;

import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.io.File;
import java.io.FileFilter;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.Set;
import java.util.prefs.Preferences;

import javax.swing.JComponent;
import javax.swing.JTree;
import javax.swing.TransferHandler;
import javax.swing.event.TreeExpansionEvent;
import javax.swing.event.TreeExpansionListener;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.MutableTreeNode;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Ordering;
import com.google.common.collect.Sets;
import com.samskivert.util.ArrayUtil;
import com.samskivert.util.ListUtil;
import com.samskivert.util.StringUtil;
import com.threerings.export.util.SerializableWrapper;
import com.threerings.util.ToolUtil;

/**
 * Displays a tree of configurations.
 */
public class FileTree extends JTree
{
	private static final long serialVersionUID = 1L;

	private String _treeName;

	/**
     * Creates a new config tree to display the configurations in the specified groups.
     */
    public FileTree (String tree,FileFilter filter,File... groups)
    {
        this(tree,filter,groups, false);
    }

    /**
     * Creates a new config tree to display the configurations in the specified group.
     *
     * @param editable if true, the tree will allow editing the configurations.
     */
    public FileTree (String tree,FileFilter filter,File group, boolean editable)
    {
        this(tree,filter,new File[] { group }, editable);
    }

    /**
     * Set the filter to use.
     */
    public void setSearchFilter (FileFilter filter)
    {
        if (_searchFilter != filter) {
        	_searchFilter = filter;
            updateFiltered();
        }
    }

    

    /**
     * Creates a {@link Transferable} containing the selected node for the clipboard.
     */
    public Transferable createClipboardTransferable ()
    {
        FileTreeNode node = getSelectedNode();
        return (node == null) ? null : new NodeTransfer(node, true);
    }

    /**
     * Returns the selected node, or <code>null</code> for none.
     */
    public FileTreeNode getSelectedNode ()
    {
        TreePath path = getSelectionPath();
        return (path == null) ? null : (FileTreeNode)path.getLastPathComponent();
    }

    /**
     * Selects a node by name (if it exists).
     */
    public void setSelectedNode (String name)
    {
        if (name == null) {
            clearSelection();
            return;
        }
        FileTreeNode node = ((FileTreeNode)getModel().getRoot()).getNode(name);
        if (node != null) {
            TreePath path = new TreePath(node.getPath());
            setSelectionPath(path);
            scrollPathToVisible(path);
        }
    }


    /**
     * Creates a new config tree to display the configurations in the specified group.
     *
     * @param editable if true, the tree will allow editing the configurations (only allowed for
     * trees depicting a single group).
     */
    protected FileTree (String treeName,FileFilter filter,File[] roots, boolean editable)
    {
    	this._treeName = treeName;
    	this._groups = roots;
    	this._filter = filter;
        setModel(new DefaultTreeModel(new FileTreeNode(roots, _searchFilter,filter), true) {
            public void valueForPathChanged (TreePath path, Object newValue) {
                // save selection
                TreePath selection = getSelectionPath();

                // remove and reinsert with a unique name in sorted order
                FileTreeNode node = (FileTreeNode)path.getLastPathComponent();
                FileTreeNode parent = (FileTreeNode)node.getParent();
                removeNodeFromParent(node);
                node.setUserObject(newValue);
                insertNodeInto(node, parent, parent.getInsertionIndex(node));

                // re-expand paths, reapply the selection
                node.expandPaths(FileTree.this);
                setSelectionPath(selection);
            }
            public void insertNodeInto (MutableTreeNode child, MutableTreeNode parent, int index) {
                super.insertNodeInto(child, parent, index);
            }
            
            public void removeNodeFromParent (MutableTreeNode node) {
                FileTreeNode ctnode = (FileTreeNode)node;
                if (removeExpanded(ctnode)) {
                    writeExpanded();
                }
                super.removeNodeFromParent(node);
            }
        });

        // start with some basics
        setRootVisible(false);
        setEditable(editable);
        getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);

        // the expansion listener notes expansion in the node state and preferences
        addTreeExpansionListener(new TreeExpansionListener() {
            public void treeExpanded (TreeExpansionEvent event) {
                FileTreeNode node = (FileTreeNode)event.getPath().getLastPathComponent();
                node.setExpanded(true);
                if (!isSearched()) {
                    addExpanded(node.getName());
                }
                node.expandPaths(FileTree.this);
            }
            public void treeCollapsed (TreeExpansionEvent event) {
                FileTreeNode node = (FileTreeNode)event.getPath().getLastPathComponent();
                node.setExpanded(false);
                if (!isSearched()) {
                    removeExpanded(node.getName());
                }
            }
        });

        // the transfer handler handles dragging and dropping (both within the tree and
        // between applications)
        setDragEnabled(true);
        setTransferHandler(new TransferHandler() {
            public int getSourceActions (JComponent comp) {
                return isEditable() ? MOVE : COPY;
            }
            public boolean canImport (JComponent comp, DataFlavor[] flavors) {
                return isEditable() &&
                    ListUtil.contains(flavors, ToolUtil.SERIALIZED_WRAPPED_FLAVOR);
            }
            public boolean importData (JComponent comp, Transferable t) {
                if (!canImport(comp, t.getTransferDataFlavors())) {
                    return false; // this isn't checked automatically for paste
                }
                boolean local = t.isDataFlavorSupported(LOCAL_NODE_TRANSFER_FLAVOR);
                Object data;
                try {
                    data = t.getTransferData(local ?
                        LOCAL_NODE_TRANSFER_FLAVOR : ToolUtil.SERIALIZED_WRAPPED_FLAVOR);
                } catch (Exception e) {
                    log.warning("Failure importing data.", e);
                    return false;
                }
                FileTreeNode node, onode = null;
                if (local) {
                    NodeTransfer transfer = (NodeTransfer)data;
                    node = transfer.cnode;
                    onode = transfer.onode;
                } else {
                    data = ((SerializableWrapper)data).getObject();
                    if (!(data instanceof FileTreeNode)) {
                        return false; // some other kind of wrapped transfer
                    }
                    node = (FileTreeNode)data;
                }
               
                FileTreeNode snode = getSelectedNode();
                FileTreeNode parent = (FileTreeNode)getModel().getRoot();
                if (snode != null && snode.getParent() != null) {
                    parent = snode.getAllowsChildren() ?
                        snode : (FileTreeNode)snode.getParent();
                }
                if (onode == parent || (onode != null && onode.getParent() == parent)) {
                    return false; // can't move to self or to the same folder
                }
                // have to clone it in case we are going to paste it multiple times
                node = (FileTreeNode)node.clone();

                // if we're moving within the tree, remove the original node here so that we
                // can reuse our identifiers
                if (onode != null && onode.getRoot() == parent.getRoot()) {
                    ((DefaultTreeModel)getModel()).removeNodeFromParent(onode);
                }

                // insert, re-expand, select
                ((DefaultTreeModel)getModel()).insertNodeInto(
                    node, parent, parent.getInsertionIndex(node));
                node.expandPaths(FileTree.this);
                setSelectionPath(new TreePath(node.getPath()));
                return true;
            }
            protected Transferable createTransferable (JComponent c) {
                FileTreeNode node = getSelectedNode();
                return (node == null) ? null : new NodeTransfer(node, false);
            }
            protected void exportDone (JComponent source, Transferable data, int action) {
                if (action == MOVE) {
                    FileTreeNode onode = ((NodeTransfer)data).onode;
                    if (onode.getParent() != null) {
                        ((DefaultTreeModel)getModel()).removeNodeFromParent(onode);
                    }
                }
            }
        });

        // read; show all; prune. Pruning must happen when all are showing.
        //readExpanded();
        setSearchFilter(null);
        pruneExpanded();
    }

    /**
     * Update the filtered view of the tree. Remakes the entire tree.
     */
    protected void updateFiltered ()
    {
        // if something was selected, remember its path; otherwise don't overwrite our last
        // selected path
        FileTreeNode selected = getSelectedNode();
        if (selected != null) {
            _lastFilterSelectedPath = selected.getName();
        }

        DefaultTreeModel model = (DefaultTreeModel)getModel();
        /*ChainFileFilter filter = new ChainFileFilter(_filter);
        
        if(_searchFilter != null) {
        	filter.add(_searchFilter);
        }*/
        // build the tree model and listen for updates
        FileTreeNode root = new FileTreeNode(_groups, _filter,_searchFilter);
        model.setRoot(root);
        model.reload();

        // the root is always expanded
        //root.setExpanded(false);

        // expand any paths needing it
        if (_expanded.isEmpty()) {
            // just expand to a default level
            root.expandPaths(this, 1);

        }else if (isSearched()) {
            // expand all the filtered findings
           root.expandPaths(this,Integer.MAX_VALUE);

        } else {
            for (String name : _expanded) {
                FileTreeNode node = root.getNode(name);
                if (node != null) {
                    node.setExpanded(true);
                    this.expandPath(new TreePath(node.getPath()));
                }
            }
            //root.expandPaths(this,0);
        }

        // if we have a last-selected path, see if it's available in our new filtered view
        if (_lastFilterSelectedPath != null) {
            FileTreeNode newSelect = root.getNode(_lastFilterSelectedPath);
            if (newSelect != null) {
                TreePath path = new TreePath(newSelect.getPath());
                setSelectionPath(path);
                scrollPathToVisible(path);
            }
        }
    }

    /**
     * Are we displaying a filtered view?
     */
    protected boolean isSearched ()
    {
        return (_searchFilter != null);
    }

    /**
     * Called when the selected configuration has been modified by a source <em>other</em> than
     * {@link #selectedConfigChanged}.
     */
    protected void selectedConfigUpdated ()
    {
        // nothing by default
    }

    /**
     * Adds the named node to the expanded set and writes out the set if it has changed.
     */
    protected void addExpanded (String name)
    {
        if (_expanded.add(name)) {
            writeExpanded();
        }
    }

    /**
     * Removes the specified node and all of its descendents from the expanded set.
     *
     * @return whether or not any names were actually removed.
     */
    protected boolean removeExpanded (FileTreeNode node)
    {
        boolean changed = _expanded.remove(node.getName());
        for (int ii = 0, nn = node.getChildCount(); ii < nn; ii++) {
            changed |= removeExpanded((FileTreeNode)node.getChildAt(ii));
        }
        return changed;
    }

    /**
     * Removes the named node from the expanded set and writes out the set if it has changed.
     */
    protected void removeExpanded (String name)
    {
        if (_expanded.remove(name)) {
            writeExpanded();
        }
    }

    /**
     * Read in the set of expanded nodes from the preferences.
     */
    protected void readExpanded ()
    {
        String names = _prefs.get(_treeName + ".expanded", null);
        if (names != null) {
            _expanded.addAll(Arrays.asList(StringUtil.parseStringArray(names)));
        }
    }

    /**
     * Prune unused config names out of our expanded set.
     */
    protected void pruneExpanded ()
    {
        FileTreeNode root = (FileTreeNode)getModel().getRoot();
        for (Iterator<String> it = _expanded.iterator(); it.hasNext(); ) {
            if (null == root.getNode(it.next())) {
                it.remove();
            }
        }
    }

    /**
     * Writes the set of expanded nodes out to the preferences.
     */
    protected void writeExpanded ()
    {

        // It's fucking ridiculous to work with arrays, but I don't have a replacement for
        // StringUtil.joinEscaped at the moment, so....
        String[] names = Iterables.toArray(_expanded, String.class);
        String value = StringUtil.joinEscaped(names);
        if (value.length() > Preferences.MAX_VALUE_LENGTH) {
            log.warning("Too many expanded paths to store in preferences, trimming.",
                "group", _treeName, "length", value.length());
            // sort the array so that the deepest paths are at the end
            Arrays.sort(names, Ordering.natural().onResultOf(new Function<String, Integer>() {
                public Integer apply (String s) {
                    return Collections.frequency(Lists.charactersOf(s), '/');
                }
            }));
            do {
                names = ArrayUtil.splice(names, names.length - 1);
                value = StringUtil.joinEscaped(names);
            } while (value.length() > Preferences.MAX_VALUE_LENGTH);
        }

        // write it
        _prefs.put(_treeName + ".expanded", value);
    }

    /**
     * Contains a node for transfer.
     */
    protected static class NodeTransfer
        implements Transferable
    {
        /** The original node (to delete when the transfer completes). */
        public FileTreeNode onode;

        /** The cloned node. */
        public FileTreeNode cnode;

        public NodeTransfer (FileTreeNode onode, boolean clipboard)
        {
            this.onode = clipboard ? null : onode;
            cnode = (FileTreeNode) onode.clone();
        }

        // documentation inherited from interface Transferable
        public DataFlavor[] getTransferDataFlavors ()
        {
            return NODE_TRANSFER_FLAVORS;
        }

        // documentation inherited from interface Transferable
        public boolean isDataFlavorSupported (DataFlavor flavor)
        {
            return ListUtil.contains(NODE_TRANSFER_FLAVORS, flavor);
        }

        // documentation inherited from interface Transferable
        public Object getTransferData (DataFlavor flavor)
        {
            return flavor.equals(LOCAL_NODE_TRANSFER_FLAVOR) ?
                this : new SerializableWrapper(cnode);
        }
    }

    /** The configuration groups. */
    protected File[] _groups;

    /** The set of paths currently expanded. */
    protected Set<String> _expanded = Sets.newHashSet();

    /** The current config filter. */
    protected FileFilter _filter,_searchFilter;

    /** The last non-null selected path when the filter changed. */
    protected String _lastFilterSelectedPath;

    /** The package preferences. */
    protected static Preferences _prefs = Preferences.userNodeForPackage(FileTree.class);

    /** A data flavor that provides access to the actual transfer object. */
    protected static final DataFlavor LOCAL_NODE_TRANSFER_FLAVOR =
        ToolUtil.createLocalFlavor(NodeTransfer.class);

    /** The flavors available for node transfer. */
    protected static final DataFlavor[] NODE_TRANSFER_FLAVORS =
        { LOCAL_NODE_TRANSFER_FLAVOR, ToolUtil.SERIALIZED_WRAPPED_FLAVOR };
}
