//
// $Id$
//
// Clyde library - tools for developing networked games
// Copyright (C) 2005-2012 Three Rings Design, Inc.
// http://code.google.com/p/clyde/
//
// Redistribution and use in source and binary forms, with or without modification, are permitted
// provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice, this list of
//    conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright notice, this list of
//    conditions and the following disclaimer in the documentation and/or other materials provided
//    with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
// INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
// PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
// INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
// TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

package com.threerings.delta;

import java.io.IOException;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.protobuf.CodedInputStream;
import com.google.protobuf.CodedOutputStream;
import com.google.protobuf.Message;
import com.google.protobuf.WireFormat;
import com.hexnova.narya.util.AnyMessage;
import com.samskivert.io.ByteArrayOutInputStream;
import com.samskivert.util.HashIntMap;
import com.samskivert.util.IntMap;
import com.samskivert.util.StringUtil;
import com.samskivert.util.Tuple;
import com.threerings.export.Encodable;
import com.threerings.expr.MutableInteger;
import com.threerings.io.ArrayMask;
import com.threerings.io.ObjectInputStream;
import com.threerings.io.ObjectOutputStream;
import com.threerings.protobuf.io.ProtobufInputStream;
import com.threerings.protobuf.io.ProtobufOutputStream;
import com.threerings.protobuf.io.ProtobufProvider;
import com.threerings.protobuf.io.ProtobufRegistry;
import com.threerings.tudey.util.ObjectMap;
import com.threerings.tudey.util.Pool.Poolable;
import com.threerings.util.DeepObject;

/**
 * A delta object that uses reflection to compare and modify the objects' fields.  Note that
 * unchanged object fields will be preserved by reference.
 */
public class ReflectiveDelta extends Delta implements Poolable
{
    /**
     * Creates a new reflective delta that transforms the original object into the revised object
     * (both of which must be instances of the same class).
     */
    public ReflectiveDelta (Object original, Object revised)
    {
    	initProtobuf(original,revised);
    }
    
    public void init(Object original, Object revised) {
    	_clazz = original.getClass();
    	_cmap = null;
    	if(original instanceof ClassMappingProvider){
    		_cmap = ((ClassMappingProvider) original).getClassMapping();
    		if(_cmap == null){
    			_cmap = getInnerClassMapping(_clazz);
    		}
    	}else{
    		// compare the fields
    		_cmap = getInnerClassMapping(_clazz);
    	}
        _mask = new BareArrayMask(_cmap.getMaskLength());
        Field[] fields = _cmap.getFields();
        FieldHandler[] handlers = _cmap.getHandlers();
        
        List<Object> values = Lists.newArrayList();
        MutableInteger midx = new MutableInteger();
        for (int ii = 0; ii < fields.length; ii++) {
            try {
                handlers[ii].populate(fields[ii], original, revised, _mask, midx, values);
            } catch (IllegalAccessException e) {
                throw new RuntimeException("Failed to access " + fields[ii] +
                    " for delta computation", e);
            }
        }
        _values = values.toArray();
    }
    
    
    public void init(Class clazz) {
    	this._clazz = clazz;
    	_cmap = ReflectiveDelta.getClassMapping(clazz);
        Field[] fields = _cmap.getFields();
        FieldHandler[] handlers = _cmap.getHandlers();
        IntMap<Tuple<Field,FieldHandler>> map =  CLASS_HANDLER_MAP.get(_clazz);
        if(map == null) {
        	map = new HashIntMap<Tuple<Field,FieldHandler>>();
        	CLASS_HANDLER_MAP.put(_clazz, map);
	        for(int ii=0;ii<fields.length;ii++) {
	        	ProtobufField pfield = fields[ii].getAnnotation(ProtobufField.class);
	        	if(pfield != null) {
	        		Tuple<Field,FieldHandler> tuple = new Tuple<Field,FieldHandler>(fields[ii],handlers[ii]);
	        		map.put(pfield.fieldNumber(), tuple);
	        	}
	        }
        }
        
        _protobufHandlers = map;
    }
    
    public void initProtobuf(Object original, Object revised) {
    	_clazz = original.getClass();
    	_cmap = null;
    	if(original instanceof ClassMappingProvider){
    		_cmap = ((ClassMappingProvider) original).getClassMapping();
    		if(_cmap == null){
    			_cmap = getInnerClassMapping(_clazz);
    		}
    	}else{
    		// compare the fields
    		_cmap = getInnerClassMapping(_clazz);
    	}
        Field[] fields = _cmap.getFields();
        FieldHandler[] handlers = _cmap.getHandlers();
        IntMap<Tuple<Field,FieldHandler>> map =  CLASS_HANDLER_MAP.get(_clazz);
        if(map == null) {
        	map = new HashIntMap<Tuple<Field,FieldHandler>>();
        	CLASS_HANDLER_MAP.put(_clazz, map);
	        for(int ii=0;ii<fields.length;ii++) {
	        	ProtobufField pfield = fields[ii].getAnnotation(ProtobufField.class);
	        	if(pfield != null) {
	        		Tuple<Field,FieldHandler> tuple = new Tuple<Field,FieldHandler>(fields[ii],handlers[ii]);
	        		map.put(pfield.fieldNumber(), tuple);
	        	}
	        }
        }
    	_protobufHandlers = map;
        
        
        for (int ii = 0; ii < fields.length; ii++) {
            try {
                handlers[ii].populateProtobuf(fields[ii], original, revised, _protobufFields);
            } catch (IllegalAccessException e) {
                throw new RuntimeException("Failed to access " + fields[ii] +
                    " for delta computation", e);
            }
        }
    }

    /**
     * No-arg constructor for deserialization.
     */
    public ReflectiveDelta ()
    {
    }

    /**
     * Checks whether the delta is empty.
     */
    public boolean isEmpty ()
    {
    	return this._protobufFields == null || _protobufFields.size() == 0;
        //return (_values.length == 0);
    }

    /**
     * Custom write method.
     */
    public void writeObject (ObjectOutputStream out)
        throws IOException
    {
    	
    	if(out instanceof ProtobufOutputStream) {
    		ByteArrayOutInputStream bos = new ByteArrayOutInputStream();
    		CodedOutputStream cout = CodedOutputStream.newInstance(bos);
    		for(Map.Entry<Integer, Object> entry : _protobufFields.entrySet()) {
    			Tuple<Field,FieldHandler> tuple = _protobufHandlers.get(entry.getKey());
    			tuple.right.write(cout, entry.getKey(), entry.getValue());
    		}
    		cout.flush();
    		byte[] bts = bos.toByteArray();
    		out.writeInt(bts.length);
    		out.writeInt(_protobufFields.size());
    		out.write(bts, 0, bts.length);
    	}else {
	        // write the class reference
	        _classStreamer.writeObject(_clazz, out, true);
	
	        // write the bitmask indicating which fields are changed
	        _mask.writeTo(out);
	
	        // write the changed fields
	        MutableInteger midx = new MutableInteger(), vidx = new MutableInteger();
	        for (FieldHandler handler : getInnerClassMapping(_clazz).getHandlers()) {
	            handler.write(_mask, midx, _values, vidx, out);
	        }
    	}
    }

    /**
     * Custom read method.
     */
    public void readObject (ObjectInputStream in)
        throws IOException, ClassNotFoundException
    {
    	if(in instanceof ProtobufInputStream) {
    		int length = in.readInt();
    		int size = in.readInt();
    		byte[] bts = new byte[length];
    		if(length > 0 ) {
    			in.read(bts);
    			CodedInputStream cint = CodedInputStream.newInstance(bts);
    			
    			for(int i=0;i<size ;i++) {
	    			int number = WireFormat.getTagFieldNumber(cint.readTag());
	    			Tuple<Field,FieldHandler> tuple = _protobufHandlers.get(number);
	    			if(tuple != null) {
	    				tuple.right.read(cint,tuple.left, number, _protobufFields);
	    			}
    			}
    		}
    	}else {
	        // read the class reference
	        _clazz = (Class<?>)_classStreamer.createObject(in);
	
	        // read the bitmask
	        ClassMapping cmap = getInnerClassMapping(_clazz);
	        _mask = new BareArrayMask(cmap.getMaskLength());
	        _mask.readFrom(in);
	
	        // read the changed fields
	        List<Object> values = Lists.newArrayList();
	        MutableInteger midx = new MutableInteger();
	        for (FieldHandler handler : cmap.getHandlers()) {
	            handler.read(_mask, midx, values, in);
	        }
	        _values = values.toArray();
    	}
    }

    @Override
    public Object apply (Object original)
    {
        // make sure it's the right class
        if (original.getClass() != _clazz) {
            throw new IllegalArgumentException("Delta class mismatch: original is " +
                original.getClass() + ", expected " + _clazz);
        }

        // create a new instance
        Object revised;
        try {
            revised = _clazz.newInstance();
        } catch (Exception e) { // InstantiationException, IllegalAccessException
            throw new RuntimeException("Failed to instantiate " + _clazz +
                " for delta application", e);
        }

        // set the fields
        ClassMapping cmap = getInnerClassMapping(_clazz);
        Field[] fields = cmap.getFields();
        FieldHandler[] handlers = cmap.getHandlers();
        MutableInteger midx = new MutableInteger(), vidx = new MutableInteger();
        for (int ii = 0; ii < fields.length; ii++) {
            try {
                handlers[ii].apply(fields[ii], original, revised, _mask, midx, _values, vidx);
            } catch (IllegalAccessException e) {
                throw new RuntimeException("Failed to access " + fields[ii] +
                    " for delta application", e);
            }
        }
        return revised;
    }
    
    public Object applyWithProtobuf (DeepObject original)
    {
        // make sure it's the right class
        if (original.getClass() != _clazz) {
            throw new IllegalArgumentException("Delta class mismatch: original is " +
                original.getClass() + ", expected " + _clazz);
        }
        
        // set the fields
        for(Map.Entry<Integer, Tuple<Field,FieldHandler>> entry : _protobufHandlers.entrySet()) {
        	Tuple<Field,FieldHandler> tuple = entry.getValue();
        	try {
        		tuple.right.applyProtobuf(tuple.left, original, original, _protobufFields);
        	} catch (IllegalAccessException e) {
                throw new RuntimeException("Failed to access " + tuple.left +
                    " for delta application", e);
            }
        }
        return original;
    }

    @Override
    public Delta merge (Delta other)
    {
        if (!(other instanceof ReflectiveDelta)) {
            throw new IllegalArgumentException("Cannot merge delta " + other);
        }
        ReflectiveDelta merged = new ReflectiveDelta();
        populateMerged((ReflectiveDelta)other, merged);
        return merged;
    }

    @Override
    public String toString ()
    {
        StringBuilder buf = new StringBuilder();
        buf.append("[class=").append(_clazz.getName());
        ClassMapping cmap = getInnerClassMapping(_clazz);
        Field[] fields = cmap.getFields();
        if(_mask != null) {
	        FieldHandler[] handlers = cmap.getHandlers();
	        MutableInteger midx = new MutableInteger(), vidx = new MutableInteger();
	        for (int ii = 0; ii < fields.length; ii++) {
	            handlers[ii].toString(fields[ii], _mask, midx, _values, vidx, buf);
	        }
        }else {
        	if(_protobufFields != null && _protobufFields.size() >0) {
        		buf.append("{");
        		for(Map.Entry<Integer, Object> entry : _protobufFields.entrySet()) {
        			buf.append(entry.getKey()).append("=").append(StringUtil.toString(entry.getValue())).append(";");
        		}
        		buf.append("}");
        	}
        }
        return buf.append("]").toString();
    }
   
    
    @Override
	public void reset() {
		_clazz = null;
    	_cmap = null;
    	_mask = null;
    	_values = null;
	}

    /**
     * Populates the merged delta.
     */
    protected void populateMerged (ReflectiveDelta other, ReflectiveDelta merged)
    {
        if (_clazz != other._clazz) {
            throw new IllegalArgumentException("Merge class mismatch: other is " +
                other._clazz + ", expected " + _clazz);
        }
        merged._clazz = _clazz;
        int mlength = getInnerClassMapping(_clazz).getMaskLength();
        merged._mask = new BareArrayMask(mlength);
        List<Object> values = Lists.newArrayList();
        for (int ii = 0, oidx = 0, nidx = 0; ii < mlength; ii++) {
            Object value;
            if (_mask.isSet(ii)) {
                Object ovalue = _values[oidx++];
                if (other._mask.isSet(ii)) {
                    Object nvalue = other._values[nidx++];
                    if (nvalue instanceof Delta) {
                        Delta ndelta = (Delta)nvalue;
                        value = (ovalue instanceof Delta) ?
                            ((Delta)ovalue).merge(ndelta) : ndelta.apply(ovalue);
                    } else {
                        value = nvalue;
                    }
                } else {
                    value = ovalue;
                }
            } else {
                if (other._mask.isSet(ii)) {
                    value = other._values[nidx++];
                } else {
                    continue;
                }
            }
            merged._mask.set(ii);
            values.add(value);
        }
        merged._values = values.toArray();
    }

    /**
     * Populates the merged delta.
     */
    protected void populateMergedProtobuf (ReflectiveDelta other, ReflectiveDelta merged)
    {
        if (_clazz != other._clazz) {
            throw new IllegalArgumentException("Merge class mismatch: other is " +
                other._clazz + ", expected " + _clazz);
        }
        
        for(Map.Entry<Integer,Object> entry : _protobufFields.entrySet()) {
        	 Object nvalue = entry.getValue();
        	 if (nvalue instanceof Delta) {
        		 Object ovalue = merged._protobufFields.get(entry.getKey());
        		 if(ovalue == null) {
        			 merged._protobufFields.put(entry.getKey(), nvalue);
        		 }else {
        			 Delta ndelta = (Delta)nvalue;
                     Object value = (ovalue instanceof Delta) ?
                         ((Delta)ovalue).merge(ndelta) : ndelta.apply(ovalue);
                     merged._protobufFields.put(entry.getKey(), value);
        		 }
                
             } else {
            	 merged._protobufFields.put(entry.getKey(), nvalue);
             }
        }
    }
    
    /**
     * Returns the class mapping for the specified class.
     */
    protected ClassMapping getInnerClassMapping (Class<?> clazz)
    {
    	if(_cmap == null){
	    		_cmap = getClassMapping(clazz);
	        return _cmap;
    	}else{
    		return _cmap;
    	}
    }
    
    /**
     * Returns the class mapping for the specified class.
     */
    public static ClassMapping getClassMapping (Class<?> clazz)
    {
    	ClassMapping _cmap = _classes.get(clazz);
        if (_cmap == null) {
            _classes.put(clazz, _cmap = new ClassMapping(clazz));
        }
        return _cmap;
    }
    
    /**
     * Collects all appropriate fields of the specified class (and its superclasses) and places
     * them in the provided results object.
     */
    protected static void collectFields (Class<?> clazz, List<Field> fields)
    {
        // add those of the superclass, if any
        Class<?> sclazz = clazz.getSuperclass();
        if (sclazz != Object.class) {
            collectFields(sclazz, fields);
        }

        // add any non-static, non-synthetic, non-transient fields
        for (Field field : clazz.getDeclaredFields()) {
            int mods = field.getModifiers();
            if (!(Modifier.isStatic(mods) || Modifier.isTransient(mods) || field.isSynthetic())) {
                field.setAccessible(true);
                fields.add(field);
            }
        }
    }

    /**
     * Contains cached information about a class.
     */
    public static class ClassMapping
    {
        /**
         * Creates a new mapping for the specified class.
         */
        public ClassMapping (Class<?> clazz)
        {
            List<Field> fields = Lists.newArrayList();
            collectFields(clazz, fields);
            _fields = fields.toArray(new Field[fields.size()]);
            _handlers = new FieldHandler[_fields.length];

            // get the handlers and count the non-final fields
            for (int ii = 0; ii < _fields.length; ii++) {
                Field field = _fields[ii];
                Class<?> type = field.getType();
                if (Modifier.isFinal(field.getModifiers()) ||
                        field.isAnnotationPresent(DeltaFinal.class)) {
                    _handlers[ii] = type.isPrimitive() ?
                        FINAL_PRIMITIVE_FIELD_HANDLERS.get(type) : FINAL_OBJECT_FIELD_HANDLER;
                } else {
                	ProtobufField protoField = field.getAnnotation(ProtobufField.class);
                	if(protoField != null ) {
                		_maxFieldNumber = Math.max(_maxFieldNumber, protoField.fieldNumber());
                	}
                	
                    _maskLength++;
                    _handlers[ii] = type.isPrimitive() ?
                        PRIMITIVE_FIELD_HANDLERS.get(type) : OBJECT_FIELD_HANDLER;
                }
            }
        }

        
        
        /**
         * Returns a reference to the array of non-transient fields.
         */
        public Field[] getFields ()
        {
            return _fields;
        }

        /**
         * Returns a reference to the array of field handlers.
         */
        public FieldHandler[] getHandlers ()
        {
            return _handlers;
        }

        /**
         * Returns the number of elements in the field mask (the number of non-transient, non-final
         * fields).
         */
        public int getMaskLength ()
        {
            return _maskLength;
        }
        
        

        /** The array of non-transient fields. */
        protected Field[] _fields;

        /** Handlers for each field. */
        protected FieldHandler[] _handlers;

        /** The number of elements in the field mask. */
        protected int _maskLength;
        
        protected int _maxFieldNumber;
    }

    /**
     * Handles a particular field.
     */
    protected static abstract class FieldHandler
    {
        /**
         * Compares the field in the original and revised objects and, if they differ, populates
         * the supplied mask and values list with the delta values.
         *
         * @param midx an in/out parameter representing the index in the mask.
         */
        public abstract void populate (
            Field field, Object original, Object revised,
            ArrayMask mask, MutableInteger midx, List<Object> values)
                throws IllegalAccessException;

        public abstract void populateProtobuf (Field field, Object original, Object revised, IntMap<Object> values)
                    throws IllegalAccessException;

        
        /**
         * Writes the delta value for the field (if any) to the stream.
         *
         * @param midx an in/out parameter representing the index in the mask.
         * @param vidx an in/out parameter representing the index in the value array.
         */
        public abstract void write (
            ArrayMask mask, MutableInteger midx, Object[] values,
            MutableInteger vidx, ObjectOutputStream out)
                throws IOException;

        /**
         * 
         * @param output
         * @param index
         * @param value
         */
        public abstract void write (com.google.protobuf.CodedOutputStream output, int index,Object value) throws IOException;
        
        public abstract void read (com.google.protobuf.CodedInputStream intput, Field field, int index,IntMap<Object> value) throws IOException;
        
        /**
         * Reads the delta value for the field (if any) from the stream.
         *
         * @param midx an in/out parameter representing the index in the mask.
         */
        public abstract void read (
            ArrayMask mask, MutableInteger midx, List<Object> values, ObjectInputStream in)
                throws IOException, ClassNotFoundException;

        
        
        /**
         * Applies the delta value (if any) to the provided objects.
         *
         * @param midx an in/out parameter representing the index in the mask.
         * @param vidx an in/out parameter representing the index in the value array.
         */
        public abstract void apply (
            Field field, Object original, Object revised, ArrayMask mask,
            MutableInteger midx, Object[] values, MutableInteger vidx)
                throws IllegalAccessException;
        

        public abstract void applyProtobuf (Field field, Object original, Object revised, IntMap<Object> map)
                throws IllegalAccessException;

        /**
         * Writes the delta value (if any) to the specified string.
         *
         * @param midx an in/out parameter representing the index in the mask.
         * @param vidx an in/out parameter representing the index in the value array.
         */
        public void toString (
            Field field, ArrayMask mask, MutableInteger midx,
            Object[] values, MutableInteger vidx, StringBuilder buf)
        {
            if (mask.isSet(midx.value++)) {
                buf.append(", " + field.getName() + "=" + values[vidx.value++]);
            }
        }
    }

    /**
     * Base class for final field handlers.
     */
    protected static abstract class FinalFieldHandler extends FieldHandler
    {
        @Override
        public void populate (
            Field field, Object original, Object revised,
            ArrayMask mask, MutableInteger midx, List<Object> values)
        {
            // no-op
        }
        
        public void populateProtobuf (Field field, Object original, Object revised, IntMap<Object> values)
                throws IllegalAccessException{
        	
        }
        
        @Override 
        public void write(CodedOutputStream output, int index, Object value) throws IOException{
        	
		}

        @Override
        public void write (
            ArrayMask mask, MutableInteger midx, Object[] values,
            MutableInteger vidx, ObjectOutputStream out)
        {
            // no-op
        }

        @Override
        public void read (
            ArrayMask mask, MutableInteger midx, List<Object> values, ObjectInputStream in)
        {
            // no-op
        }
        
        @Override
        public void read (com.google.protobuf.CodedInputStream intput,Field field, int index,IntMap<Object> map) throws IOException{
		}

        @Override
        public void toString (
            Field field, ArrayMask mask, MutableInteger midx,
            Object[] values, MutableInteger vidx, StringBuilder buf)
        {
            // no-op
        }
    }

    protected transient ClassMapping _cmap;
    
    /** The object class. */
    protected Class<?> _clazz;

    
    protected transient IntMap<Object> _protobufFields = new HashIntMap<Object>();
    
    protected transient IntMap<Tuple<Field,FieldHandler>> _protobufHandlers = null;
    protected static Map<Class<?>,IntMap<Tuple<Field,FieldHandler>>> CLASS_HANDLER_MAP = Maps.newHashMap();
    
    /** The mask indicating which fields have changed. */
    protected BareArrayMask _mask;

    /** The values for each of the object's changed fields (either a new value or a {@link Delta}
     * object). */
    protected Object[] _values;

    /** Cached mappings for deltable classes. */
    protected static ObjectMap<Class<?>, ClassMapping> _classes = new ObjectMap<Class<?>, ClassMapping>();

    /** Field handlers for primitive fields mapped by class. */
    protected static final Map<Class<?>, FieldHandler> PRIMITIVE_FIELD_HANDLERS =
        ImmutableMap.<Class<?>, FieldHandler>builder()
            .put(Boolean.TYPE, new FieldHandler() {
                @Override public void populate (
                    Field field, Object original, Object revised,
                    ArrayMask mask, MutableInteger midx, List<Object> values)
                        throws IllegalAccessException {
                    int idx = midx.value++;
                    boolean nvalue = field.getBoolean(revised);
                    if (field.getBoolean(original) != nvalue) {
                        mask.set(idx);
                        values.add(nvalue);
                    }
                }

                public void populateProtobuf (Field field, Object original, Object revised, IntMap<Object> values)
                            throws IllegalAccessException{
                	ProtobufField protoField = field.getAnnotation(ProtobufField.class);
                	if(protoField != null ) {
                		boolean nvalue = field.getBoolean(revised);
                        if (field.getBoolean(original) != nvalue) {
                        	values.put(protoField.fieldNumber(), Boolean.valueOf(nvalue));
                        }
                	}
                }
                
                @Override public void write (
                   ArrayMask mask, MutableInteger midx, Object[] values,
                   MutableInteger vidx, ObjectOutputStream out)
                       throws IOException {
                   if (mask.isSet(midx.value++)) {
                       out.writeBoolean((Boolean)values[vidx.value++]);
                   }
                }
                @Override public void read (
                    ArrayMask mask, MutableInteger midx, List<Object> values, ObjectInputStream in)
                        throws IOException, ClassNotFoundException {
                    if (mask.isSet(midx.value++)) {
                        values.add(in.readBoolean());
                    }
                }
                @Override public void apply (
                    Field field, Object original, Object revised, ArrayMask mask,
                    MutableInteger midx, Object[] values, MutableInteger vidx)
                        throws IllegalAccessException {
                    boolean value;
                    if (mask.isSet(midx.value++)) {
                        value = (Boolean)values[vidx.value++];
                    } else {
                        value = field.getBoolean(original);
                    }
                    field.setBoolean(revised, value);
                }
                

                @Override
                public void applyProtobuf (Field field, Object original, Object revised, IntMap<Object> map)
                        throws IllegalAccessException{
                	 boolean value;
                	 ProtobufField protoField = field.getAnnotation(ProtobufField.class);
                	 
                	if (protoField != null && map.containsKey(protoField.fieldNumber())) {
                        value = (Boolean)map.get(protoField.fieldNumber());
                    } else {
                        value = field.getBoolean(original);
                    }
                	
                	field.setBoolean(revised, value);
                }
                
				@Override
				public void write(CodedOutputStream output, int index, Object value) throws IOException{
					output.writeBool(index,(boolean)value);
				}
				
				@Override
				public void read (com.google.protobuf.CodedInputStream intput,Field field, int index,IntMap<Object> map) throws IOException{
					map.put(index,Boolean.valueOf(intput.readBool()));
				}
				
            })
            .put(Byte.TYPE, new FieldHandler() {
                @Override public void populate (
                    Field field, Object original, Object revised,
                    ArrayMask mask, MutableInteger midx, List<Object> values)
                        throws IllegalAccessException {
                    int idx = midx.value++;
                    byte nvalue = field.getByte(revised);
                    if (field.getByte(original) != nvalue) {
                        mask.set(idx);
                        values.add(nvalue);
                    }
                }
                
                public void populateProtobuf (Field field, Object original, Object revised, IntMap<Object> values)
                        throws IllegalAccessException{
            	ProtobufField protoField = field.getAnnotation(ProtobufField.class);
            	if(protoField != null ) {
            		byte nvalue = field.getByte(revised);
                    if (field.getByte(original) != nvalue) {
                    	values.put(protoField.fieldNumber(), Byte.valueOf(nvalue));
                    }
            	}
            }
                
                @Override public void write (
                   ArrayMask mask, MutableInteger midx, Object[] values,
                   MutableInteger vidx, ObjectOutputStream out)
                       throws IOException {
                   if (mask.isSet(midx.value++)) {
                       out.writeByte((Byte)values[vidx.value++]);
                   }
                }
                
                @Override 
                public void write(CodedOutputStream output, int index, Object value) throws IOException{
                	output.writeFixed32(index,(int)value);
				}
                
                @Override
                public void read (com.google.protobuf.CodedInputStream intput,Field field, int index,IntMap<Object> map) throws IOException{
					map.put(index,Byte.valueOf((byte)intput.readFixed32()));
				}
                
                @Override public void read (
                    ArrayMask mask, MutableInteger midx, List<Object> values, ObjectInputStream in)
                        throws IOException, ClassNotFoundException {
                    if (mask.isSet(midx.value++)) {
                        values.add(in.readByte());
                    }
                }
                @Override public void apply (
                    Field field, Object original, Object revised, ArrayMask mask,
                    MutableInteger midx, Object[] values, MutableInteger vidx)
                        throws IllegalAccessException {
                    byte value;
                    if (mask.isSet(midx.value++)) {
                        value = (Byte)values[vidx.value++];
                    } else {
                        value = field.getByte(original);
                    }
                    field.setByte(revised, value);
                }
                
                @Override
                public void applyProtobuf (Field field, Object original, Object revised, IntMap<Object> map)
                        throws IllegalAccessException{
                	byte value;
                	 ProtobufField protoField = field.getAnnotation(ProtobufField.class);
                	 
                	if (protoField != null && map.containsKey(protoField.fieldNumber())) {
                        value = (Byte)map.get(protoField.fieldNumber());
                    } else {
                        value = field.getByte(original);
                    }
                	
                	field.setByte(revised, value);
                }
            })
            .put(Character.TYPE, new FieldHandler() {
                @Override public void populate (
                    Field field, Object original, Object revised,
                    ArrayMask mask, MutableInteger midx, List<Object> values)
                        throws IllegalAccessException {
                    int idx = midx.value++;
                    char nvalue = field.getChar(revised);
                    if (field.getChar(original) != nvalue) {
                        mask.set(idx);
                        values.add(nvalue);
                    }
                }
                
                public void populateProtobuf (Field field, Object original, Object revised, IntMap<Object> values)
                        throws IllegalAccessException{
                	ProtobufField protoField = field.getAnnotation(ProtobufField.class);
	            	if(protoField != null ) {
	            		 char nvalue = field.getChar(revised);
	            		 if (field.getChar(original) != nvalue) {
	                    	values.put(protoField.fieldNumber(), Character.valueOf(nvalue));
	                    }
	            	}
                }
                
                @Override public void write (
                   ArrayMask mask, MutableInteger midx, Object[] values,
                   MutableInteger vidx, ObjectOutputStream out)
                       throws IOException {
                   if (mask.isSet(midx.value++)) {
                       out.writeChar((Character)values[vidx.value++]);
                   }
                }
                
                @Override 
                public void write(CodedOutputStream output, int index, Object value) throws IOException{
                	output.writeFixed32(index,((Character)value));
				}

                @Override
                public void read (com.google.protobuf.CodedInputStream intput,Field field, int index,IntMap<Object> map) throws IOException{
					map.put(index,Character.valueOf((char)intput.readFixed32()));
				}
                
                @Override public void read (
                    ArrayMask mask, MutableInteger midx, List<Object> values, ObjectInputStream in)
                        throws IOException, ClassNotFoundException {
                    if (mask.isSet(midx.value++)) {
                        values.add(in.readChar());
                    }
                }
                @Override public void apply (
                    Field field, Object original, Object revised, ArrayMask mask,
                    MutableInteger midx, Object[] values, MutableInteger vidx)
                        throws IllegalAccessException {
                    char value;
                    if (mask.isSet(midx.value++)) {
                        value = (Character)values[vidx.value++];
                    } else {
                        value = field.getChar(original);
                    }
                    field.setChar(revised, value);
                }
                
                @Override
                public void applyProtobuf (Field field, Object original, Object revised, IntMap<Object> map)
                        throws IllegalAccessException{
                	char value;
                	ProtobufField protoField = field.getAnnotation(ProtobufField.class);
                	 
                	if (protoField != null && map.containsKey(protoField.fieldNumber())) {
                        value = (Character)map.get(protoField.fieldNumber());
                    } else {
                        value = field.getChar(original);
                    }
                	
                	field.setChar(revised, value);
                }
            })
            .put(Double.TYPE, new FieldHandler() {
                @Override public void populate (
                    Field field, Object original, Object revised,
                    ArrayMask mask, MutableInteger midx, List<Object> values)
                        throws IllegalAccessException {
                    int idx = midx.value++;
                    double nvalue = field.getDouble(revised);
                    if (field.getDouble(original) != nvalue) {
                        mask.set(idx);
                        values.add(nvalue);
                    }
                }
                
                public void populateProtobuf (Field field, Object original, Object revised, IntMap<Object> values)
                        throws IllegalAccessException{
                	ProtobufField protoField = field.getAnnotation(ProtobufField.class);
	            	if(protoField != null ) {
	            		double nvalue = field.getDouble(revised);
	            		if (field.getDouble(original) != nvalue) {
	                    	values.put(protoField.fieldNumber(), Double.valueOf(nvalue));
	                    }
	            	}
                }
                
                @Override public void write (
                   ArrayMask mask, MutableInteger midx, Object[] values,
                   MutableInteger vidx, ObjectOutputStream out)
                       throws IOException {
                   if (mask.isSet(midx.value++)) {
                       out.writeDouble((Double)values[vidx.value++]);
                   }
                }
                
                
                @Override 
                public void write(CodedOutputStream output, int index, Object value) throws IOException{
                	output.writeDouble(index,(Double)value);
				}
                
                @Override
                public void read (com.google.protobuf.CodedInputStream intput, Field field,int index,IntMap<Object> map) throws IOException{
					map.put(index,Double.valueOf(intput.readDouble()));
				}
                
                
                @Override public void read (
                    ArrayMask mask, MutableInteger midx, List<Object> values, ObjectInputStream in)
                        throws IOException, ClassNotFoundException {
                    if (mask.isSet(midx.value++)) {
                        values.add(in.readDouble());
                    }
                }
                @Override public void apply (
                    Field field, Object original, Object revised, ArrayMask mask,
                    MutableInteger midx, Object[] values, MutableInteger vidx)
                        throws IllegalAccessException {
                    double value;
                    if (mask.isSet(midx.value++)) {
                        value = (Double)values[vidx.value++];
                    } else {
                        value = field.getDouble(original);
                    }
                    field.setDouble(revised, value);
                }
                
                @Override
                public void applyProtobuf (Field field, Object original, Object revised, IntMap<Object> map)
                        throws IllegalAccessException{
                	double value;
                	ProtobufField protoField = field.getAnnotation(ProtobufField.class);
                	 
                	if (protoField != null && map.containsKey(protoField.fieldNumber())) {
                        value = (Double)map.get(protoField.fieldNumber());
                    } else {
                        value = field.getDouble(original);
                    }
                	
                	field.setDouble(revised, value);
                }
            })
            .put(Float.TYPE, new FieldHandler() {
                @Override public void populate (
                    Field field, Object original, Object revised,
                    ArrayMask mask, MutableInteger midx, List<Object> values)
                        throws IllegalAccessException {
                    int idx = midx.value++;
                    float nvalue = field.getFloat(revised);
                    if (field.getFloat(original) != nvalue) {
                        mask.set(idx);
                        values.add(nvalue);
                    }
                }
                
                public void populateProtobuf (Field field, Object original, Object revised, IntMap<Object> values)
                        throws IllegalAccessException{
                	ProtobufField protoField = field.getAnnotation(ProtobufField.class);
	            	if(protoField != null ) {
	            		float nvalue = field.getFloat(revised);
	            		if (field.getFloat(original) != nvalue) {
	                    	values.put(protoField.fieldNumber(), Float.valueOf(nvalue));
	                    }
	            	}
                }
                
                @Override 
                public void write(CodedOutputStream output, int index, Object value) throws IOException{
                	output.writeFloat(index,(Float)value);
				}
                
                @Override
                public void read (com.google.protobuf.CodedInputStream intput, Field field,int index,IntMap<Object> map) throws IOException{
					map.put(index,Float.valueOf(intput.readFloat()));
				}
                
                @Override public void write (
                   ArrayMask mask, MutableInteger midx, Object[] values,
                   MutableInteger vidx, ObjectOutputStream out)
                       throws IOException {
                   if (mask.isSet(midx.value++)) {
                       out.writeFloat((Float)values[vidx.value++]);
                   }
                }
                @Override public void read (
                    ArrayMask mask, MutableInteger midx, List<Object> values, ObjectInputStream in)
                        throws IOException, ClassNotFoundException {
                    if (mask.isSet(midx.value++)) {
                        values.add(in.readFloat());
                    }
                }
                @Override public void apply (
                    Field field, Object original, Object revised, ArrayMask mask,
                    MutableInteger midx, Object[] values, MutableInteger vidx)
                        throws IllegalAccessException {
                    float value;
                    if (mask.isSet(midx.value++)) {
                        value = (Float)values[vidx.value++];
                    } else {
                        value = field.getFloat(original);
                    }
                    field.setFloat(revised, value);
                }
                
                @Override
                public void applyProtobuf (Field field, Object original, Object revised, IntMap<Object> map)
                        throws IllegalAccessException{
                	float value;
                	ProtobufField protoField = field.getAnnotation(ProtobufField.class);
                	 
                	if (protoField != null && map.containsKey(protoField.fieldNumber())) {
                        value = (Float)map.get(protoField.fieldNumber());
                    } else {
                        value = field.getFloat(original);
                    }
                	
                	field.setFloat(revised, value);
                }
            })
            .put(Integer.TYPE, new FieldHandler() {
                @Override public void populate (
                    Field field, Object original, Object revised,
                    ArrayMask mask, MutableInteger midx, List<Object> values)
                        throws IllegalAccessException {
                    int idx = midx.value++;
                    int nvalue = field.getInt(revised);
                    if (field.getInt(original) != nvalue) {
                        mask.set(idx);
                        values.add(nvalue);
                    }
                }
                

                public void populateProtobuf (Field field, Object original, Object revised, IntMap<Object> values)
                        throws IllegalAccessException{
                	ProtobufField protoField = field.getAnnotation(ProtobufField.class);
	            	if(protoField != null ) {
	            		int nvalue = field.getInt(revised);
	            		if (field.getInt(original) != nvalue) {
	                    	values.put(protoField.fieldNumber(), Integer.valueOf(nvalue));
	                    }
	            	}
                }
                
                @Override 
                public void write(CodedOutputStream output, int index, Object value) throws IOException{
                	output.writeInt32(index,(Integer)value);
				}
                
                @Override
                public void read (com.google.protobuf.CodedInputStream intput,Field field, int index,IntMap<Object> map) throws IOException{
					map.put(index,Integer.valueOf(intput.readInt32()));
				}
                
                @Override public void write (
                   ArrayMask mask, MutableInteger midx, Object[] values,
                   MutableInteger vidx, ObjectOutputStream out)
                       throws IOException {
                   if (mask.isSet(midx.value++)) {
                       out.writeInt((Integer)values[vidx.value++]);
                   }
                }
                @Override public void read (
                    ArrayMask mask, MutableInteger midx, List<Object> values, ObjectInputStream in)
                        throws IOException, ClassNotFoundException {
                    if (mask.isSet(midx.value++)) {
                        values.add(in.readInt());
                    }
                }
                @Override public void apply (
                    Field field, Object original, Object revised, ArrayMask mask,
                    MutableInteger midx, Object[] values, MutableInteger vidx)
                        throws IllegalAccessException {
                    int value;
                    if (mask.isSet(midx.value++)) {
                        value = (Integer)values[vidx.value++];
                    } else {
                        value = field.getInt(original);
                    }
                    field.setInt(revised, value);
                }
                
                
                @Override
                public void applyProtobuf (Field field, Object original, Object revised, IntMap<Object> map)
                        throws IllegalAccessException{
                	int value;
                	ProtobufField protoField = field.getAnnotation(ProtobufField.class);
                	 
                	if (protoField != null && map.containsKey(protoField.fieldNumber())) {
                        value = (Integer)map.get(protoField.fieldNumber());
                    } else {
                        value = field.getInt(original);
                    }
                	
                	field.setInt(revised, value);
                }
            })
            .put(Long.TYPE, new FieldHandler() {
                @Override public void populate (
                    Field field, Object original, Object revised,
                    ArrayMask mask, MutableInteger midx, List<Object> values)
                        throws IllegalAccessException {
                    int idx = midx.value++;
                    long nvalue = field.getLong(revised);
                    if (field.getLong(original) != nvalue) {
                        mask.set(idx);
                        values.add(nvalue);
                    }
                }
                

                public void populateProtobuf (Field field, Object original, Object revised, IntMap<Object> values)
                        throws IllegalAccessException{
                	ProtobufField protoField = field.getAnnotation(ProtobufField.class);
	            	if(protoField != null ) {
	            		long nvalue = field.getLong(revised);
	            		if (field.getLong(original) != nvalue) {
	                    	values.put(protoField.fieldNumber(), Long.valueOf(nvalue));
	                    }
	            	}
                }
                
                @Override 
                public void write(CodedOutputStream output, int index, Object value) throws IOException{
                	output.writeInt64(index,(Long)value);
				}
                
                @Override
                public void read (com.google.protobuf.CodedInputStream intput, Field field,int index,IntMap<Object> map) throws IOException{
					map.put(index,Long.valueOf(intput.readInt64()));
				}
                
                @Override public void write (
                   ArrayMask mask, MutableInteger midx, Object[] values,
                   MutableInteger vidx, ObjectOutputStream out)
                       throws IOException {
                   if (mask.isSet(midx.value++)) {
                       out.writeLong((Long)values[vidx.value++]);
                   }
                }
                @Override public void read (
                    ArrayMask mask, MutableInteger midx, List<Object> values, ObjectInputStream in)
                        throws IOException, ClassNotFoundException {
                    if (mask.isSet(midx.value++)) {
                        values.add(in.readLong());
                    }
                }
                @Override public void apply (
                    Field field, Object original, Object revised, ArrayMask mask,
                    MutableInteger midx, Object[] values, MutableInteger vidx)
                        throws IllegalAccessException {
                    long value;
                    if (mask.isSet(midx.value++)) {
                        value = (Long)values[vidx.value++];
                    } else {
                        value = field.getLong(original);
                    }
                    field.setLong(revised, value);
                }
                

                @Override
                public void applyProtobuf (Field field, Object original, Object revised, IntMap<Object> map)
                        throws IllegalAccessException{
                	long value;
                	ProtobufField protoField = field.getAnnotation(ProtobufField.class);
                	 
                	if (protoField != null && map.containsKey(protoField.fieldNumber())) {
                        value = (Long)map.get(protoField.fieldNumber());
                    } else {
                        value = field.getLong(original);
                    }
                	
                	field.setLong(revised, value);
                }
            })
            .put(Short.TYPE, new FieldHandler() {
                @Override public void populate (
                    Field field, Object original, Object revised,
                    ArrayMask mask, MutableInteger midx, List<Object> values)
                        throws IllegalAccessException {
                    int idx = midx.value++;
                    short nvalue = field.getShort(revised);
                    if (field.getShort(original) != nvalue) {
                        mask.set(idx);
                        values.add(nvalue);
                    }
                }
                
                public void populateProtobuf (Field field, Object original, Object revised, IntMap<Object> values)
                        throws IllegalAccessException{
                	ProtobufField protoField = field.getAnnotation(ProtobufField.class);
	            	if(protoField != null ) {
	            		short nvalue = field.getShort(revised);
	            		if (field.getShort(original) != nvalue) {
	                    	values.put(protoField.fieldNumber(), Short.valueOf(nvalue));
	                    }
	            	}
                }
                
                @Override 
                public void write(CodedOutputStream output, int index, Object value) throws IOException{
                	output.writeInt32(index,(Short)value);
				}
                
                @Override
                public void read (com.google.protobuf.CodedInputStream intput, Field field,int index,IntMap<Object> map) throws IOException{
					map.put(index,Integer.valueOf(intput.readInt32()));
				}
                
                @Override public void write (
                   ArrayMask mask, MutableInteger midx, Object[] values,
                   MutableInteger vidx, ObjectOutputStream out)
                       throws IOException {
                   if (mask.isSet(midx.value++)) {
                       out.writeShort((Short)values[vidx.value++]);
                   }
                }
                @Override public void read (
                    ArrayMask mask, MutableInteger midx, List<Object> values, ObjectInputStream in)
                        throws IOException, ClassNotFoundException {
                    if (mask.isSet(midx.value++)) {
                        values.add(in.readShort());
                    }
                }
                @Override public void apply (
                    Field field, Object original, Object revised, ArrayMask mask,
                    MutableInteger midx, Object[] values, MutableInteger vidx)
                        throws IllegalAccessException {
                    short value;
                    if (mask.isSet(midx.value++)) {
                        value = (Short)values[vidx.value++];
                    } else {
                        value = field.getShort(original);
                    }
                    field.setShort(revised, value);
                }
                
                @Override
                public void applyProtobuf (Field field, Object original, Object revised, IntMap<Object> map)
                        throws IllegalAccessException{
                	short value;
                	ProtobufField protoField = field.getAnnotation(ProtobufField.class);
                	 
                	if (protoField != null && map.containsKey(protoField.fieldNumber())) {
                        value = (Short)map.get(protoField.fieldNumber());
                    } else {
                        value = field.getShort(original);
                    }
                	
                	field.setShort(revised, value);
                }
            })
            .build();

    /** Field handlers for final primitive fields mapped by class. */
    protected static final Map<Class<?>, FieldHandler> FINAL_PRIMITIVE_FIELD_HANDLERS =
        ImmutableMap.<Class<?>, FieldHandler>builder()
            .put(Boolean.TYPE, new FinalFieldHandler() {
                @Override public void apply (
                    Field field, Object original, Object revised, ArrayMask mask,
                    MutableInteger midx, Object[] values, MutableInteger vidx)
                        throws IllegalAccessException {
                    field.setBoolean(revised, field.getBoolean(original));
                }
                
                @Override
                public void applyProtobuf (Field field, Object original, Object revised, IntMap<Object> map)
                        throws IllegalAccessException{
                	field.setBoolean(revised, field.getBoolean(original));
                }
            })
            .put(Byte.TYPE, new FinalFieldHandler() {
                @Override public void apply (
                    Field field, Object original, Object revised, ArrayMask mask,
                    MutableInteger midx, Object[] values, MutableInteger vidx)
                        throws IllegalAccessException {
                    field.setByte(revised, field.getByte(original));
                }
                
                @Override
                public void applyProtobuf (Field field, Object original, Object revised, IntMap<Object> map)
                        throws IllegalAccessException{
                	field.setByte(revised, field.getByte(original));
                }
            })
            .put(Character.TYPE, new FinalFieldHandler() {
                @Override public void apply (
                    Field field, Object original, Object revised, ArrayMask mask,
                    MutableInteger midx, Object[] values, MutableInteger vidx)
                        throws IllegalAccessException {
                    field.setChar(revised, field.getChar(original));
                }
                
                @Override
                public void applyProtobuf (Field field, Object original, Object revised, IntMap<Object> map)
                        throws IllegalAccessException{
                	field.setChar(revised, field.getChar(original));
                }
            })
            .put(Double.TYPE, new FinalFieldHandler() {
                @Override public void apply (
                    Field field, Object original, Object revised, ArrayMask mask,
                    MutableInteger midx, Object[] values, MutableInteger vidx)
                        throws IllegalAccessException {
                    field.setDouble(revised, field.getDouble(original));
                }
                
                @Override
                public void applyProtobuf (Field field, Object original, Object revised, IntMap<Object> map)
                        throws IllegalAccessException{
                	field.setDouble(revised, field.getDouble(original));
                }
            })
            .put(Float.TYPE, new FinalFieldHandler() {
                @Override public void apply (
                    Field field, Object original, Object revised, ArrayMask mask,
                    MutableInteger midx, Object[] values, MutableInteger vidx)
                        throws IllegalAccessException {
                    field.setFloat(revised, field.getFloat(original));
                }
                
                @Override
                public void applyProtobuf (Field field, Object original, Object revised, IntMap<Object> map)
                        throws IllegalAccessException{
                	field.setFloat(revised, field.getFloat(original));
                }
            })
            .put(Integer.TYPE, new FinalFieldHandler() {
                @Override public void apply (
                    Field field, Object original, Object revised, ArrayMask mask,
                    MutableInteger midx, Object[] values, MutableInteger vidx)
                        throws IllegalAccessException {
                    field.setInt(revised, field.getInt(original));
                }
                
                @Override
                public void applyProtobuf (Field field, Object original, Object revised, IntMap<Object> map)
                        throws IllegalAccessException{
                	field.setInt(revised, field.getInt(original));
                }
            })
            .put(Long.TYPE, new FinalFieldHandler() {
                @Override public void apply (
                    Field field, Object original, Object revised, ArrayMask mask,
                    MutableInteger midx, Object[] values, MutableInteger vidx)
                        throws IllegalAccessException {
                    field.setLong(revised, field.getLong(original));
                }
                
                @Override
                public void applyProtobuf (Field field, Object original, Object revised, IntMap<Object> map)
                        throws IllegalAccessException{
                	 field.setLong(revised, field.getLong(original));
                }
            })
            .put(Short.TYPE, new FinalFieldHandler() {
                @Override public void apply (
                    Field field, Object original, Object revised, ArrayMask mask,
                    MutableInteger midx, Object[] values, MutableInteger vidx)
                        throws IllegalAccessException {
                    field.setShort(revised, field.getShort(original));
                }
                
                @Override
                public void applyProtobuf (Field field, Object original, Object revised, IntMap<Object> map)
                        throws IllegalAccessException{
                	field.setShort(revised, field.getShort(original));
                }
            })
            .build();

    /** Handler for object fields. */
    protected static final FieldHandler OBJECT_FIELD_HANDLER = new FieldHandler() {
        @Override public void populate (
            Field field, Object original, Object revised,
            ArrayMask mask, MutableInteger midx, List<Object> values)
                throws IllegalAccessException {
            int idx = midx.value++;
            Object ovalue = _oarray[0] = field.get(original);
            Object nvalue = _narray[0] = field.get(revised);
            if (!Arrays.deepEquals(_oarray, _narray)) {
                if (Delta.checkDeltable(ovalue, nvalue)) {
                    nvalue = Delta.createDelta(ovalue, nvalue);
                }
                mask.set(idx);
                values.add(nvalue);
            }
        }
        
        public void populateProtobuf (Field field, Object original, Object revised, IntMap<Object> values)
                throws IllegalAccessException{
        	ProtobufField protoField = field.getAnnotation(ProtobufField.class);
        	if(protoField != null ) {
        		Object ovalue = _oarray[0] = field.get(original);
                Object nvalue = _narray[0] = field.get(revised);
                if (!Arrays.deepEquals(_oarray, _narray)) {
                	 if (Delta.checkDeltable(ovalue, nvalue)) {
                         nvalue = Delta.createDelta(ovalue, nvalue);
                     }
                	values.put(protoField.fieldNumber(), nvalue);
                }
        	}
        }
        
        @Override 
        public void write(CodedOutputStream output, int index, Object value) throws IOException{
        	if(value instanceof ProtobufProvider) {
        		Message messsage = ((ProtobufProvider) value).transform();
        		output.writeMessage(index,messsage);
        	}
		}
        
        @Override
        public void read (com.google.protobuf.CodedInputStream intput, Field field,int index,IntMap<Object> map) throws IOException{

        	try {
        		com.google.protobuf.Message.Builder builder = ProtobufRegistry.getBuilder(field.getType());
	        	intput.readMessage(builder, com.google.protobuf.ExtensionRegistryLite.getEmptyRegistry());
	        	Message message = builder.build();
	        	Object value = ProtobufRegistry.transform(message);
				map.put(index,value);
			} catch (Exception e) {
				throw new IOException("cannot instance builder for field="+field.toString(),e);
			}
		}
        
        @Override public void write (
            ArrayMask mask, MutableInteger midx, Object[] values,
            MutableInteger vidx, ObjectOutputStream out)
                throws IOException {
            if (mask.isSet(midx.value++)) {
                out.writeObject(values[vidx.value++]);
            }
        }
        @Override public void read (
            ArrayMask mask, MutableInteger midx, List<Object> values, ObjectInputStream in)
                throws IOException, ClassNotFoundException {
            if (mask.isSet(midx.value++)) {
                values.add(in.readObject());
            }
        }
        @Override public void apply (
            Field field, Object original, Object revised, ArrayMask mask,
            MutableInteger midx, Object[] values, MutableInteger vidx)
                throws IllegalAccessException {
            Object value;
            if (mask.isSet(midx.value++)) {
                value = values[vidx.value++];
                if (value instanceof Delta) {
                    value = ((Delta)value).apply(field.get(original));
                }
            } else {
                value = field.get(original);
            }
            field.set(revised, value);
        }
        @Override
        public void applyProtobuf (Field field, Object original, Object revised, IntMap<Object> map)
                throws IllegalAccessException{
        	Object value;
        	ProtobufField protoField = field.getAnnotation(ProtobufField.class);
        	
        	if (protoField != null && map.containsKey(protoField.fieldNumber())) {
                value = map.get(protoField.fieldNumber());
                if (value instanceof Delta) {
                    value = ((Delta)value).apply(field.get(original));
                }
            } else {
                value = field.get(original);
            }
        	
        	field.set(revised, value);
        }
        
        protected Object[] _oarray = new Object[1], _narray = new Object[1];
    };

    /** Handler for final object fields. */
    protected static final FieldHandler FINAL_OBJECT_FIELD_HANDLER = new FinalFieldHandler() {
        @Override public void apply (
            Field field, Object original, Object revised, ArrayMask mask,
            MutableInteger midx, Object[] values, MutableInteger vidx)
                throws IllegalAccessException {
            field.set(revised, field.get(original));
        }
        
        @Override
        public void applyProtobuf (Field field, Object original, Object revised, IntMap<Object> map)
                throws IllegalAccessException{
        	field.set(revised, field.get(original));
        }
    };
    
    
}
