package com.threerings.tudey.util;

import java.util.Arrays;

/**
 * @author BluseChen
 * 
 */
public class CoordIntArray {

	class Quaternion {
		int[] values = null;

		public Quaternion(int x, int y, int value) {
			this(x, y);
			Arrays.fill(values, _empty);
			values[this.getIndex(x, y)] = value;
		}

		public Quaternion(int x, int y) {
			int size = 4;
			if (x == 0) {
				size -= 2;
			}

			if (y == 0) {
				size -= 2;
			}

			if (size == 0) {
				size = 1;
			}
			values = new int[size];
		}

		public int getIndex(int x, int y) {
			if (x == 0 && y == 0) {
				return 0;
			}

			if (x == 0) {
				return y > 0 ? 1 : 0;
			}

			if (y == 0) {
				return x > 0 ? 1 : 0;
			}

			if (x < 0) {
				if (y < 0) {
					return 0;
				} else {
					return 1;
				}
			} else {
				if (y < 0) {
					return 2;
				} else {
					return 3;
				}
			}
		}

		public int get(int x, int y) {
			return values[this.getIndex(x, y)];
		}

		/**
         * Sets the value at the specified coordinates within the quaternion.
         *
         * @return the previously stored value.
         */
		public int put(int x, int y, int bits) {
			int index = this.getIndex(x, y);
			int value = values[index];
			values[index] = bits;
			return value;
		}
		
		 /**
         * Removes the value at the specified coordinates within the quaternion
         *
         * @return the previously stored value.
         */
		public int remove(int x, int y) {
			int idx = this.getIndex(x, y);
			int ovalue = values[idx];
			if (ovalue != _empty) {
				values[idx] = _empty;
			}
			
			return ovalue;
		}

		
		 /**
         * Sets the value at the specified coordinates to the bitwise OR of the previous
         * value and the new value.
         *
         * @return the previously stored value.
         */
		public int set(int x, int y, int bits) {
			int index = this.getIndex(x, y);
			int value = values[index];
			values[index] |= bits;
			return value;
		}

		public boolean isClear() {
			for (int value : values) {
				if (value != _empty) {
					return false;
				}
			}
			return true;
		}
		
		public void clear() {
			values = new int[values.length];
		}
	}

	private Quaternion[][] quaters = null;
	private int _empty;

	public CoordIntArray(int size,int empty) {
		quaters = new Quaternion[size][size];
		 _empty = empty;
	}

	public int get(int x, int y) {
		int absX = Math.abs(x);
		int absY = Math.abs(y);
		if (quaters.length <= absY || quaters[0].length <= absX) {
			return _empty;
		}else{
			Quaternion quater =  quaters[absY][absX];
			if(quater == null){
				return _empty;
			}else{
				return quater.get(x, y);
			}
		}
	}

	public int put(int x, int y, int bits) {
		if(bits == _empty) {
			return remove(x, y);
		}else {
			int absX = Math.abs(x);
			int absY = Math.abs(y);
			if (quaters.length <= absY || quaters[0].length <= absX) {
				quaters = expand(quaters,absX,absY);
			}
			
			if (quaters[absY][absX] == null) {
				quaters[absY][absX] = new Quaternion(x, y, bits);
				return _empty;
			} else {
				return quaters[absY][absX].put(x, y, bits);
			}
		}
	}

	public int remove(int x, int y) {
		int absX = Math.abs(x);
		int absY = Math.abs(y);
		if (quaters.length <= absY || quaters[0].length <= absX) {
			return _empty;
		}else{
			Quaternion quater =  quaters[absY][absX];
			if(quater == null){
				return _empty;
			}else{
				return quater.remove(x, y);
			}
		}
	}

	public int setBits(int x, int y, int bits) {
		if (bits == 0) {
            return get(x, y);
        }
		
		int absX = Math.abs(x);
		int absY = Math.abs(y);
		if (quaters.length <= absY || quaters[0].length <= absX) {
			quaters = expand(quaters,absX,absY);
		}
		
		if (quaters[absY][absX] == null) {
			quaters[absY][absX] = new Quaternion(x, y, bits);
			return _empty;
		} else {
			return quaters[absY][absX].set(x, y, bits);
		}
	}

	public int size() {
		return quaters.length;
	}
	
	public void clear() {
		for(Quaternion[] quater : quaters) {
			for(Quaternion q : quater) {
				if(q != null) {
					q.clear();
				}
			}
		}
	}

	/**
	 *  扩展二维数组
	 * @param array
	 * @param minX 最小应该满足
	 * @param minY 最小应该满足
	 * @return
	 */
	public static Quaternion[][] expand(Quaternion[][] array,int minX, int minY) {
		int sizeY = array.length>minY ? array.length : Math.max(array.length * 2, minY+1);
		int sizeX = array[0].length > minX ? array[0].length : Math.max(array[0].length * 2,minX+1);
		Quaternion[][] newArrays = new Quaternion[sizeY][sizeX];
		for (int i = 0; i < array.length; i++) {
			System.arraycopy(array[i], 0, newArrays[i], 0, array[i].length);
		}
		return newArrays;
	}
}
