//
// $Id$
//
// Clyde library - tools for developing networked games
// Copyright (C) 2005-2012 Three Rings Design, Inc.
// http://code.google.com/p/clyde/
//
// Redistribution and use in source and binary forms, with or without modification, are permitted
// provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice, this list of
//    conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright notice, this list of
//    conditions and the following disclaimer in the documentation and/or other materials provided
//    with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
// INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
// PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
// INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
// TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

package com.threerings.tudey.config;

import com.threerings.editor.Editable;
import com.threerings.editor.EditorTypes;
import com.threerings.expr.Scope;
import com.threerings.expr.util.ScopeUtil;
import com.threerings.math.Vector3f;
import com.threerings.opengl.effect.Easing;
import com.threerings.opengl.scene.Scene;
import com.threerings.opengl.scene.ViewerEffect;
import com.threerings.opengl.scene.config.ViewerEffectConfig;
import com.threerings.opengl.util.GlContext;
import com.threerings.tudey.client.TudeySceneView;

import static com.threerings.tudey.Log.log;

/**
 * Configurations for Tudey-specific viewer effects.
 */
@EditorTypes({ TudeyViewerEffectConfig.Camera.class })
public abstract class TudeyViewerEffectConfig extends ViewerEffectConfig
{
    /**
     * Adds a camera to the view.
     */
    public static class Camera extends TudeyViewerEffectConfig
    {
        /** The transition to use when switching to or from the camera. */
        @Editable(min=0.0, step=0.01)
        public float transition;

        /** The easing to use for the transition. */
        @Editable
        public Easing easing = new Easing.None();

        /** The camera configuration. */
        @Editable
        public CameraConfig camera = new CameraConfig();

        /** More camera configurations!  For backwards compatibility we keep the original. */
        @Editable
        public CameraConfig[] cameras = new CameraConfig[0];

        @Override
        public void preload (GlContext ctx)
        {
            // Do nothing
        }

        @Override
        public ViewerEffect getViewerEffect (GlContext ctx, Scope scope, ViewerEffect effect)
        {
        	
            final TudeySceneView view = ScopeUtil.resolve(
                scope, "view:this", null, TudeySceneView.class);
            if (view == null || !ScopeUtil.resolve(scope, "cameraEnabled", true)) {
                return getNoopEffect(effect);
            }
            
            if (effect instanceof CameraEffect) {
                ((CameraEffect)effect).setConfig(this);
            } else {
                effect = new CameraEffect(view,this);
            }
            
            return effect;
        }
    }
    
    public static class CameraEffect extends ViewerEffect {
    	
    	public CameraEffect(TudeySceneView view,Camera camera) {
    		this.view = view;
    		_transition = camera.transition;
    		_camcfg = camera.camera;
			_camcfgs = camera.cameras;		
    		_easing = camera.easing;
    	}
    	
        public void setConfig (Camera camera) {
        	if(!view.isCameraEffectEnable()){
        		  return;
        	}
        	
        	/**
        	 * 避免区域内摄像头瞬间切换
        	 */
        	if(_camcfg == camera.camera) {
        		return;
        	}
        	
            if (_activated) {
            	//log.warning("@removed ("+_camcfg.hashCode()+")="+_camcfg);
                view.removeCameraConfig(_camcfg, 0f, null,this._location);
                for (CameraConfig cc : _camcfgs) {
                    view.removeCameraConfig(cc, 0f, null,this._location);
                }
            }
            
            _transition = camera.transition;
            _easing = camera.easing;
            _camcfg = camera.camera;
            _camcfgs = camera.cameras;	
            if (_activated) {
            	//log.warning("@add 0,("+_camcfg.hashCode()+")="+_camcfg);
                view.addCameraConfig(_camcfg, 0, null,this._location);
                for (CameraConfig cc : _camcfgs) {
                    view.addCameraConfig(cc, 0, null,this._location);
                }
            }
        }
        public void activate (Scene scene) {
        	if(!view.isCameraEffectEnable()){
      		  return;
      	  	}
            _activated = true;
            //log.warning("@add (activated,("+_camcfg.hashCode()+")="+_camcfg);
            view.addCameraConfig(_camcfg, _transition, _easing , _location);
            if(_camcfgs != null) {
	            for (CameraConfig cc : _camcfgs) {
	                view.addCameraConfig(cc, _transition, _easing,this._location);
	            }
            }
        }
        public void deactivate () {
        	if(!view.isCameraEffectEnable()){
      		  return;
      	  	}
        	// log.warning("@remove (deactivate,("+_camcfg.hashCode()+")="+_camcfg);
            view.removeCameraConfig(_camcfg, _transition, _easing);
            for (CameraConfig cc : _camcfgs) {
                view.removeCameraConfig(cc, _transition, _easing);
            }
            _activated = false;
        }
        
        public void setLocation(Vector3f location) {
        	if(Vector3f.ZERO.equals(location)){
        		return;
        	}
        	
        	if(_location == null) {
        		_location = new Vector3f();
        	}
        	
        	_location.set(location);
        }
        
        protected  Vector3f _location = null;
        private TudeySceneView view;
        protected float _transition;
        protected Easing _easing;
        protected CameraConfig _camcfg;
        protected CameraConfig[] _camcfgs;
        protected boolean _activated;
    }
}
