//
// $Id$
//
// Clyde library - tools for developing networked games
// Copyright (C) 2005-2012 Three Rings Design, Inc.
// http://code.google.com/p/clyde/
//
// Redistribution and use in source and binary forms, with or without modification, are permitted
// provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice, this list of
//    conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright notice, this list of
//    conditions and the following disclaimer in the documentation and/or other materials provided
//    with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
// INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
// PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
// INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
// TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

package com.threerings.tudey.config;

import java.util.Calendar;
import java.util.Date;

import com.threerings.config.ConfigManager;
import com.threerings.editor.Editable;
import com.threerings.editor.EditorTypes;
import com.threerings.editor.Strippable;
import com.threerings.export.Exportable;
import com.threerings.io.Streamable;
import com.threerings.opengl.util.PreloadableSet;
import com.threerings.util.DeepObject;

/**
 * Configurations for handler conditions.
 */
@EditorTypes({
    ConditionConfig.Tagged.class, ConditionConfig.InstanceOf.class,
    ConditionConfig.Intersecting.class, ConditionConfig.IntersectsScene.class,
    ConditionConfig.DistanceWithin.class, ConditionConfig.Random.class,
    ConditionConfig.Limit.class, ConditionConfig.All.class, ConditionConfig.TimeRange.class, ConditionConfig.RefractoryTime.class,
    ConditionConfig.Any.class, ConditionConfig.FlagSet.class,
    ConditionConfig.Cooldown.class, ConditionConfig.Not.class,ConditionConfig.IntervalRange.class,
    ConditionConfig.Always.class, ConditionConfig.Evaluate.class,
    ConditionConfig.Action.class, ConditionConfig.Is.class, ConditionConfig.HasTracer.class,
    ConditionConfig.DateRange.class,ConditionConfig.DistanceGreater.class,ConditionConfig.Interval.class,
    ConditionConfig.DistanceGreaterSpawnOrigin.class,ConditionConfig.CountDownLatch.class,ConditionConfig.IsState.class, 
    ConditionConfig.AngularGreater.class,ConditionConfig.Alive.class,
    ConditionConfig.Fail.class,ConditionConfig.AngularWithin.class})
@Strippable
public abstract class ConditionConfig extends DeepObject
    implements Exportable, Streamable
{
	
	
	public static class  Fail  extends ConditionConfig{
		public String getLogicClassName ()
        {
            return "com.threerings.tudey.server.logic.ConditionLogic$Fail";
        }
	}
    /**
     * Determines whether an entity has a tag.
     */
    public static class Tagged extends ConditionConfig
    {
        /** The tag of interest. */
        @Editable(hgroup="t")
        public String tag = "";

        /** Whether or not all targets must match the condition (as opposed to any). */
        @Editable(hgroup="t")
        public boolean all;

        /** The target to check. */
        @Editable
        public TargetConfig target = new TargetConfig.Source();

        @Override // documentation inherited
        public String getLogicClassName ()
        {
            return "com.threerings.tudey.server.logic.ConditionLogic$Tagged";
        }

        @Override // documentation inherited
        public void invalidate ()
        {
            target.invalidate();
        }
    }

    /**
     * Determines whether an entity('s logic) is an instance of some class.
     */
    public static class InstanceOf extends ConditionConfig
    {
        /** The name of the class to check. */
    	@Editable(hgroup="c", width=40)
        public String logicClass = "com.threerings.tudey.server.logic.PawnLogic";

        /** Whether or not all targets must match the condition (as opposed to any). */
        @Editable(hgroup="d")
        public boolean all;

        /** The target to check. */
        @Editable
        public TargetConfig target = new TargetConfig.Source();

        @Override // documentation inherited
        public String getLogicClassName ()
        {
            return "com.threerings.tudey.server.logic.ConditionLogic$InstanceOf";
        }

        @Override // documentation inherited
        public void invalidate ()
        {
            target.invalidate();
        }
    }
    
    
    
    
    /**
     * Determines whether an entity('s logic) is an instance of some class.
     */
    public static class DistanceGreaterSpawnOrigin extends ConditionConfig
    {
    	/** The minimum distance. */
        @Editable(min=0.0, step=0.1, hgroup="m")
        public float distance;
        
        /** Whether or not all targets must match the condition (as opposed to any). */
        @Editable(hgroup="t")
        public boolean all;
        
        /** The target to check. */
        @Editable
        public TargetConfig target = new TargetConfig.Source();

        @Override // documentation inherited
        public String getLogicClassName ()
        {
            return "com.threerings.tudey.server.logic.ConditionLogic$DistanceGreaterSpawnOrigin";
        }

        @Override // documentation inherited
        public void invalidate ()
        {
            target.invalidate();
        }
    }

    /**
     * Determines whether two regions intersect.
     */
    public static class Intersecting extends ConditionConfig
    {
        /** Whether or not to require all targets in the first region. */
        @Editable(hgroup="a")
        public boolean allFirst;

        /** Whether or not to require all targets in the second region. */
        @Editable(hgroup="a")
        public boolean allSecond;

        /** The first region to check. */
        @Editable
        public RegionConfig first = new RegionConfig.Default();

        /** The second region to check. */
        @Editable
        public RegionConfig second = new RegionConfig.Default();

        @Override // documentation inherited
        public String getLogicClassName ()
        {
            return "com.threerings.tudey.server.logic.ConditionLogic$Intersecting";
        }

        @Override // documentation inherited
        public void invalidate ()
        {
            first.invalidate();
            second.invalidate();
        }
    }

    /**
     * Determines if a region intersects the scene based on a collision mask.
     */
    public static class IntersectsScene extends ConditionConfig
    {
        /** The region. */
        @Editable
        public RegionConfig region = new RegionConfig.Default();

        /** The collision mask. */
        @Editable(editor="mask", mode="collision")
        public int collisionMask;

        @Override // documentation inherited
        public String getLogicClassName ()
        {
            return "com.threerings.tudey.server.logic.ConditionLogic$IntersectsScene";
        }

        @Override // documentation inherited
        public void invalidate ()
        {
            region.invalidate();
        }
    }

    /**
     * Determines whether the distance between two entities is within a pair of bounds.
     */
    public static class DistanceGreater extends ConditionConfig
    {
        /** The minimum distance. */
        @Editable(min=0.0, step=0.1, hgroup="m")
        public float distance;

        /** Whether or not to require all targets in the first region. */
        @Editable(hgroup="a")
        public boolean allFirst;

        /** Whether or not to require all targets in the second region. */
        @Editable(hgroup="a")
        public boolean allSecond;

        /** The first target to check. */
        @Editable
        public TargetConfig first = new TargetConfig.Source();

        /** The second target to check. */
        @Editable
        public TargetConfig second = new TargetConfig.Activator();

        @Override // documentation inherited
        public String getLogicClassName ()
        {
            return "com.threerings.tudey.server.logic.ConditionLogic$DistanceGreater";
        }

        @Override // documentation inherited
        public void invalidate ()
        {
            first.invalidate();
            second.invalidate();
        }
    }
    
    /**
     * Determines whether the distance between two entities is within a pair of bounds.
     */
    public static class DistanceWithin extends ConditionConfig
    {
        /** The minimum distance. */
        @Editable(min=0.0, step=0.1, hgroup="m")
        public float minimum;

        /** The maximum distance. */
        @Editable(min=0.0, step=0.1, hgroup="m")
        public float maximum;

        /** Whether or not to require all targets in the first region. */
        @Editable(hgroup="a")
        public boolean allFirst;

        /** Whether or not to require all targets in the second region. */
        @Editable(hgroup="a")
        public boolean allSecond;

        /** The first target to check. */
        @Editable
        public TargetConfig first = new TargetConfig.Source();

        /** The second target to check. */
        @Editable
        public TargetConfig second = new TargetConfig.Activator();

        @Override // documentation inherited
        public String getLogicClassName ()
        {
            return "com.threerings.tudey.server.logic.ConditionLogic$DistanceWithin";
        }

        @Override // documentation inherited
        public void invalidate ()
        {
            first.invalidate();
            second.invalidate();
        }
    }
    
    
    public static class AngularWithin extends ConditionConfig {
    	
    	 /** The minimum distance. */
        @Editable(min=0, max=180, step=1, scale=Math.PI/180.0, hgroup="m")
        public float minimum;

        /** The maximum distance. */
        @Editable(min=0, max=180, step=1, scale=Math.PI/180.0, hgroup="m")
        public float maximum;

        /** Whether or not to require all targets in the first region. */
        @Editable(hgroup="a")
        public boolean allFirst;

        /** Whether or not to require all targets in the second region. */
        @Editable(hgroup="a")
        public boolean allSecond;

        /** The first target to check. */
        @Editable
        public TargetConfig first = new TargetConfig.Source();

        /** The second target to check. */
        @Editable
        public TargetConfig second = new TargetConfig.Activator();

        @Override // documentation inherited
        public String getLogicClassName ()
        {
            return "com.threerings.tudey.server.logic.ConditionLogic$AngularWithin";
        }
        
        @Override // documentation inherited
        public void invalidate ()
        {
            first.invalidate();
            second.invalidate();
        }
    }

    
    /**
     * Satisfied with a fixed probability.
     */
    public static class Random extends ConditionConfig
    {
        /** The probability that the condition is satisfied. */
        @Editable(min=0, max=1, step=0.01)
        public float probability = 0.5f;

        @Override // documentation inherited
        public String getLogicClassName ()
        {
            return "com.threerings.tudey.server.logic.ConditionLogic$Random";
        }
    }

    
    /**
     * Satisfied with a fixed probability.
     */
    public static class Alive extends ConditionConfig
    {
    	 /** Whether or not all targets must match the condition (as opposed to any). */
        @Editable(hgroup="c")
		public boolean all;
        
    	/** The target to check. */
        @Editable
        public TargetConfig target = new TargetConfig.Source();
        
        @Override // documentation inherited
        public String getLogicClassName ()
        {
            return "com.threerings.tudey.server.logic.ConditionLogic$Alive";
        }
    }
    
    
    /**
     * Satisfied with a fixed probability.
     */
    public static class IsState extends ConditionConfig
    {
        /** The probability that the condition is satisfied. */
        @Editable(min=0, step=1)
        public int state = 0;

        
        /** The target to check. */
        @Editable
        public TargetConfig target = new TargetConfig.Source();
        
        /** Whether or not all targets must match the condition (as opposed to any). */
        @Editable(hgroup="c")
        public boolean all;


        
        @Override // documentation inherited
        public String getLogicClassName ()
        {
            return "com.threerings.tudey.server.logic.ConditionLogic$IsState";
        }
       
        @Override // documentation inherited
        public void invalidate ()
        {
            target.invalidate();
        }
    }
    
    
    /**
     * Will be satisfied a fixed number of times.
     */
    public static class Limit extends ConditionConfig
    {
        /** The number of times this condition is satisfied. */
        @Editable
        public int limit = 1;

        @Override // documentation inherited
        public String getLogicClassName ()
        {
            return "com.threerings.tudey.server.logic.ConditionLogic$Limit";
        }
    }

    /**
     * Will be satisfied a fixed number of times.
     */
    public static class CountDownLatch extends Limit
    {
        @Override // documentation inherited
        public String getLogicClassName ()
        {
            return "com.threerings.tudey.server.logic.ConditionLogic$CountDownLatch";
        }
    }
    
    /**
     * Will be satisfied a fixed number of times.
     */
    public static class Interval extends ConditionConfig
    {
        /** The number of times this condition is satisfied. */
    	@Editable(min=0 , step=1)
        public int delay = 1;
    	
    	@Editable(min=0 , step=1)
        public int interval = 1;

        @Override // documentation inherited
        public String getLogicClassName ()
        {
            return "com.threerings.tudey.server.logic.ConditionLogic$Interval";
        }
    }

    
    
    /**
     * Satisfied if all of the component conditions are satisfied.
     */
    public static class All extends ConditionConfig
    {
        /** The component conditions. */
        @Editable
        public ConditionConfig[] conditions = new ConditionConfig[0];

        @Override // documentation inherited
        public String getLogicClassName ()
        {
            return "com.threerings.tudey.server.logic.ConditionLogic$All";
        }

        @Override // documentation inherited
        public void invalidate ()
        {
            for (ConditionConfig condition : conditions) {
                condition.invalidate();
            }
        }
    }

    /**
     * Satisfied if any of the component conditions are satisfied.
     */
    public static class Any extends ConditionConfig
    {
        /** The component conditions. */
        @Editable
        public ConditionConfig[] conditions = new ConditionConfig[0];

        @Override // documentation inherited
        public String getLogicClassName ()
        {
            return "com.threerings.tudey.server.logic.ConditionLogic$Any";
        }

        @Override // documentation inherited
        public void invalidate ()
        {
            for (ConditionConfig condition : conditions) {
                condition.invalidate();
            }
        }
    }

    /**
     * Determines whether an actor's flag is set.
     */
    public static class FlagSet extends ConditionConfig
    {
        /** The name of the flag definition. */
        @Editable(hgroup="f")
        public String flagName = "WARP";

        /** If we're checking for set or not set. */
        @Editable(hgroup="f")
        public boolean set = true;

        /** The target to check. */
        @Editable
        public TargetConfig target = new TargetConfig.Source();

        @Override // documentation inherited
        public String getLogicClassName ()
        {
            return "com.threerings.tudey.server.logic.ConditionLogic$FlagSet";
        }

        @Override // documentation inherited
        public void invalidate ()
        {
            target.invalidate();
        }
    }
    
    public static class AngularGreater extends ConditionConfig{
       
        /** The source region to check. */
        @Editable
        public TargetConfig source = new TargetConfig.Source();

        
        /** The second region to check. */
        @Editable
        public TargetConfig target = new TargetConfig.Source();
        
        @Editable(min = 0D, max= 180, scale = Math.PI/180,hgroup="d")
        public double angle;
        
        /** Whether or not to require all targets in the second region. */
        @Editable(hgroup="d")
        public boolean allTarget;

        @Override // documentation inherited
        public String getLogicClassName ()
        {
            return "com.threerings.tudey.server.logic.ConditionLogic$AngularGreater";
        }

        @Override // documentation inherited
        public void invalidate ()
        {
        	source.invalidate();
        	target.invalidate();
        }
    	
    }

    /**
     * Ensures a cooldown time is met between satisfied conditions.
     */
    public static class Cooldown extends ConditionConfig
    {
        /** The amount of cooldown time. */
        @Editable
        public int time = 0;
        
        /**
         * 如果为true：则开始第一次满足条件的时间设置为time
         * 否则刚开始即可满足条件
         */
        @Editable
        public boolean delay = false;

        @Override // documentation inherited
        public String getLogicClassName ()
        {
            return "com.threerings.tudey.server.logic.ConditionLogic$Cooldown";
        }
    }
    

    /**
     * range 为一段时间窗口，interval则是没间隔多少时间，设置下一个时间窗口
     * 下个时间窗口设置时间为：timestamp + interval;
     * 时间窗口范围:[ timestamp + interval,timestamp + interval+range )
     */
    public static class IntervalRange extends ConditionConfig
    {
        /** The amount of cooldown time. */
        @Editable
        public int interval = 0;
        
        @Editable
        public int range = 0;

        @Override // documentation inherited
        public String getLogicClassName ()
        {
            return "com.threerings.tudey.server.logic.ConditionLogic$IntervalRange";
        }
    }


    /**
     * Satisfied if the component condition is not satisfied.
     */
    public static class Not extends ConditionConfig
    {
        /** The component condition. */
        @Editable
        public ConditionConfig condition = new ConditionConfig.Tagged();

        @Override // documentation inherited
        public String getLogicClassName ()
        {
            return "com.threerings.tudey.server.logic.ConditionLogic$Not";
        }

        @Override // documentation inherited
        public void invalidate ()
        {
            condition.invalidate();
        }
    }

    /**
     * Satisfied always.
     */
    public static class Always extends ConditionConfig
    {
        @Override // documentation inherited
        public String getLogicClassName ()
        {
            return "com.threerings.tudey.server.logic.ConditionLogic$Always";
        }
    }

    /**
     * Satisfied if the expression evaluates to true.
     */
    public static class Evaluate extends ConditionConfig
    {
        /** The expression to evaluate. */
        @Editable
        public ExpressionConfig expression = new ExpressionConfig.Constant();

        @Override // documentation inherited
        public String getLogicClassName ()
        {
            return "com.threerings.tudey.server.logic.ConditionLogic$Evaluate";
        }

        @Override // documentation inherited
        public void invalidate ()
        {
            expression.invalidate();
        }
    }

    /**
     * Satisfied if executing the aciton returns true.
     */
    public static class Action extends ConditionConfig
    {
        /** The action to perform. */
        @Editable
        public ActionConfig action = new ActionConfig.SpawnActor();

        @Override // documentation inherited
        public String getLogicClassName ()
        {
            return "com.threerings.tudey.server.logic.ConditionLogic$Action";
        }

        @Override // documentation inherited
        public void getPreloads (ConfigManager cfgmgr, PreloadableSet preloads)
        {
            action.getPreloads(cfgmgr, preloads);
        }

        @Override // documentation inherited
        public void invalidate ()
        {
            action.invalidate();
        }
    }

    /**
     * Determines whether two entities are the same
     */
    public static class Is extends ConditionConfig
    {
        /** Whether or not all targets must match the condition (as opposed to any). */
        @Editable(hgroup="t")
        public boolean all;

        /** The target to check. */
        @Editable
        public TargetConfig target = new TargetConfig.Tagged();

        /** The target to check. */
        @Editable
        public TargetConfig source = new TargetConfig.Source();

        @Override // documentation inherited
        public String getLogicClassName ()
        {
            return "com.threerings.tudey.server.logic.ConditionLogic$Is";
        }

        @Override // documentation inherited
        public void invalidate ()
        {
            target.invalidate();
            source.invalidate();
        }
    }
    
    public static class HasTracer extends ConditionConfig {

		@Editable
		public TargetConfig target = new TargetConfig.Source();

		@Editable(hgroup="d")
        public boolean all;
		
		public String getLogicClassName() {
			return "com.threerings.tudey.server.logic.ConditionLogic$HasTracer";
		}

		public void invalidate() {
			this.target.invalidate();
		}
	}

    /**
     * Determines if the current date is within the range.
     */
    public static class DateRange extends ConditionConfig
    {
        /** The starting date. */
		@Editable(editor = "datetime", mode = "format=yyyy-MM-dd HH:mm:ss, style=full", nullable = true, hgroup = "a")
        public Long start;

        /** The ending date. */
		@Editable(editor = "datetime", mode = "format=yyyy-MM-dd HH:mm:ss, style=full", nullable = true, hgroup = "a")
        public Long end;

        @Override // documentation inherited
        public String getLogicClassName ()
        {
            return "com.threerings.tudey.server.logic.ConditionLogic$DateRange";
        }
    }
    
    public static class Time implements Exportable,Streamable,Comparable<Time>{
    	
    	public static final Time ZERO = new Time(0,0,0);
    	
    	public Time(int hour,int minute,int second) {
    		this.hour = hour;
    		this.minute = minute;
    		this.second = second;
    	}
    	
    	public Time() {
    		
    	}
    	
    	@Editable(min = 0, hgroup = "a")
		public int hour = 0;
    	@Editable(min = 0, hgroup = "a")
		public int minute = 0;
    	@Editable(min = 0, hgroup = "a")
		public int second = 0;
		@Override
		public int compareTo(Time o) {
			int diff = 0;
			if((diff = hour - o.hour )== 0) {
				if((diff =minute - o.minute) == 0) {
					return second - o.second;
				}else {
					return diff;
				}
			}else {
				return diff;
			}
		}
		
		public static Time getTime(Calendar calendar) {
			
			int hour = calendar.get(Calendar.HOUR_OF_DAY);
			int minute = calendar.get(Calendar.MINUTE);
			int second = calendar.get(Calendar.SECOND);
			
			return new Time(hour,minute,second);
		}
    }
    
    /**
     * 
     *
     */
    public static class RefractoryTime extends ConditionConfig {
    	
    	/**
    	 * 是否在固定时间触发，如果是，则将会在这时间段之后，的多少秒内则触发，否则不触发
    	 * 如果是否，则会判断在触发时间点之后，是否触发过，如果没有，则触发，然后记录该次触发时间
    	 * 
    	 */
    	@Editable(min = 0, hgroup = "a")
    	public boolean startup = false;
    	
    	@Editable(min = 0, hgroup = "b")
    	public Time[] times = new Time[0];
    	
    	@Override
        public String getLogicClassName () {
            return "com.threerings.tudey.server.logic.ConditionLogic$RefractoryTime";
        }
    	
    	public Calendar getNextTime(Calendar cal) {
			Time last = null;
			Time now = Time.getTime(cal);
			for(Time time : times) {
				//查找比now大的，times中最小那个
				if(now.compareTo(time) < 0) {
					if(last != null) {
						if(last.compareTo(time) >0) {
							last = time;
						}
					}else {
						last = time;
					}
				}
			}
			
			//如果找不到，说明now现在比 times中最大的还要大，则需要找到最小的Time，然后将时间调整到后面一天，再设置最小的Time的值
			if(last == null) {
				last = getLowest();
				Calendar next = Calendar.getInstance();
				next.add(Calendar.DAY_OF_YEAR, 1);
				next.set(Calendar.HOUR_OF_DAY, last.hour);
				next.set(Calendar.SECOND, last.second);
				next.set(Calendar.MINUTE, last.minute);
				return next;
			}else {
				Calendar next = Calendar.getInstance();
				next.set(Calendar.HOUR_OF_DAY, last.hour);
				next.set(Calendar.SECOND, last.second);
				next.set(Calendar.MINUTE, last.minute);
				return next;
			}
		}
    	
    	/**
    	 * 查找最近一次的时间
    	 * @param cal
    	 * @return
    	 */
    	public Calendar getLatest(Calendar cal) {
			Time last = null;
			Time now = Time.getTime(cal);
			for(Time time : times) {
				//查找比now小的，times中最大那个
				if(now.compareTo(time) >= 0) {
					if(last != null) {
						if(last.compareTo(time) <0) {
							last = time;
						}
					}else {
						last = time;
					}
				}
			}
			
			//如果找不到，说明now现在比 times中最大的还要大，则需要找到最小的Time，然后将时间调整到前面一天，再设置Time的值
			if(last == null) {
				last = getBiggest();
				Calendar next = Calendar.getInstance();
				next.add(Calendar.DAY_OF_YEAR, -1);
				next.set(Calendar.HOUR_OF_DAY, last.hour);
				next.set(Calendar.SECOND, last.second);
				next.set(Calendar.MINUTE, last.minute);
				return next;
			}else {
				Calendar next = Calendar.getInstance();
				next.set(Calendar.HOUR_OF_DAY, last.hour);
				next.set(Calendar.SECOND, last.second);
				next.set(Calendar.MINUTE, last.minute);
				return next;
			}
		}
    	
    	private Time getBiggest() {
    		Time last = null;
			for(Time time : times) {
				if(last != null) {
					if(last.compareTo(time) <0) {
						last = time;
					}
				}else {
					last = time;
				}
			}
			
            return last;
    	}
    	
    	private Time getLowest() {
			Time last = null;
			for(Time time : times) {
				if(last != null) {
					if(last.compareTo(time) >0) {
						last = time;
					}
				}else {
					last = time;
				}
			}
			
            return last;
		}
    	
    }
    
    
    public static class TimeRange extends ConditionConfig {
    	@Editable(min = 0, hgroup = "b")
		public int startHour = 0;
		
		@Editable(min = 0, hgroup = "b")
		public int startMinute = 0;
		
		@Editable(min = 0, hgroup = "b")
		public int startSecond = 0;
		
		@Editable(min = 0, hgroup = "d")
		public int endHour = 0;
		
		@Editable(min = 0, hgroup = "d")
		public int endMinute = 0;
		
		@Editable(min = 0, hgroup = "d")
		public int endSecond = 0;
		
		@Override
        public String getLogicClassName () {
            return "com.threerings.tudey.server.logic.ConditionLogic$TimeRange";
        }
		
		public long getTime(long now, boolean start) {
			Calendar calendar = Calendar.getInstance();
            calendar.setTime(new Date(now));
			
			if (start) {
				calendar.set(Calendar.HOUR_OF_DAY, this.startHour);
	            calendar.set(Calendar.MINUTE, this.startMinute);
	            calendar.set(Calendar.SECOND, startSecond);
			} else {
				calendar.set(Calendar.HOUR_OF_DAY, this.endHour);
	            calendar.set(Calendar.MINUTE, this.endMinute);
	            calendar.set(Calendar.SECOND, endSecond);
			}
			
            Date zero = calendar.getTime();
            return zero.getTime();
		}
    }

    /**
     * Returns the name of the server-side logic class for this condition.
     */
    public abstract String getLogicClassName ();

    /**
     * Adds the resources to preload for this action into the provided set.
     */
    public void getPreloads (ConfigManager cfgmgr, PreloadableSet preloads)
    {
        // nothing by default
    }

    /**
     * Invalidates any cached data.
     */
    public void invalidate ()
    {
        // nothing by default
    }
}
