package com.threerings.swing.filetree;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.StringSelection;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import javax.swing.Icon;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.border.EmptyBorder;
import javax.swing.event.TreeSelectionListener;
import javax.swing.filechooser.FileSystemView;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.TreePath;

import com.threerings.util.MessageManager;

public class FileTreePanel extends JPanel implements MouseListener,KeyListener {
	/**
	 * File system view.
	 */
	protected static FileSystemView fsv = FileSystemView.getFileSystemView();
	
	/**
	 * Clipboard
	 */
	private Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();

	/**
	 * Renderer for the file tree.
	 */
	private static class FileTreeCellRenderer extends DefaultTreeCellRenderer {
		/**
		 * Icon cache to speed the rendering.
		 */
		private Map<String, Icon> iconCache = new HashMap<String, Icon>();

		/**
		 * Root name cache to speed the rendering.
		 */
		private Map<File, String> rootNameCache = new HashMap<File, String>();

		/*
		 * (non-Javadoc)
		 * 
		 * @see
		 * javax.swing.tree.DefaultTreeCellRenderer#getTreeCellRendererComponent(javax.
		 * swing.JTree, java.lang.Object, boolean, boolean, boolean, int, boolean)
		 */
		@Override
		public Component getTreeCellRendererComponent(JTree tree, Object value, boolean sel, boolean expanded,
				boolean leaf, int row, boolean hasFocus) {
			FileTreeNode ftn = (FileTreeNode) value;
			File file = ftn.file;
			String filename = "";
			if (file != null) {
				if (ftn.isFileSystemRoot) {
					// long start = System.currentTimeMillis();
					filename = this.rootNameCache.get(file);
					if (filename == null) {
						filename = fsv.getSystemDisplayName(file);
						this.rootNameCache.put(file, filename);
					}
					// long end = System.currentTimeMillis();
					// System.out.println(filename + ":" + (end - start));
				} else {
					filename = file.getName();
				}
			}
			JLabel result = (JLabel) super.getTreeCellRendererComponent(tree, filename, sel, expanded, leaf, row,
					hasFocus);
			if (file != null) {
				Icon icon = this.iconCache.get(filename);
				if (icon == null) {
					// System.out.println("Getting icon of " + filename);
					icon = fsv.getSystemIcon(file);
					this.iconCache.put(filename, icon);
				}
				result.setIcon(icon);
			}
			return result;
		}
	}


	public void addTreeSelectionListener(TreeSelectionListener listener) {
		tree.addTreeSelectionListener(listener);
	}

	/**
	 * The file tree.
	 */
	private FileTree tree;
	private JPopupMenu menu = null;
	private File _root;
	private FileFilter _filter;
	private FileFilterPanel _filterPanel;
	/**
	 * Creates the file tree panel.
	 */
	public FileTreePanel(File root, FileFilter filter,MessageManager msg) {
		this.setLayout(new BorderLayout());
		this._root = root;
		this._filter = filter;
		File[] roots = root.listFiles(filter);
		this.tree = new FileTree("resource-tree",_filter,roots);
		this.tree.setCellRenderer(new FileTreeCellRenderer());
		this.tree.setRootVisible(false);
		final JScrollPane jsp = new JScrollPane(this.tree);
		jsp.setBorder(new EmptyBorder(0, 0, 0, 0));
		_filterPanel = new FileFilterPanel(msg);
		this.add(_filterPanel,BorderLayout.NORTH);
		this.add(jsp, BorderLayout.CENTER);
		menu = new JPopupMenu("");
		tree.add(menu);
		_filterPanel.setTree(tree);
		JMenuItem refresh = new JMenuItem("Refresh");
		menu.add(refresh);
		refresh.addActionListener(new ActionListener() {

			@Override
			public void actionPerformed(ActionEvent e) {
				TreePath treePath = tree.getSelectionPath();
				Object obj = treePath.getLastPathComponent();
				if (obj instanceof FileTreeNode) {
					if (!((FileTreeNode) obj).isFile()) {
						((FileTreeNode) obj).refresh();
					}
				}
				SwingUtilities.invokeLater(new Runnable() {
					public void run() {
						tree.updateUI();
					}
				});

			}
		});
		
		
		JMenuItem copy = new JMenuItem("Copy Name");
		menu.add(copy);
		copy.addActionListener(new ActionListener() {

			@Override
			public void actionPerformed(ActionEvent e) {
				TreePath path = tree.getSelectionPath();
				if (path == null) { // JTree上没有任何项被选中
					return;
				}
				Object obj = path.getLastPathComponent();
				if (obj instanceof FileTreeNode) {
					if (((FileTreeNode) obj).isFile()) {
						File file = ((FileTreeNode) obj).file;
						StringBuilder builder = new StringBuilder(file.getName());
						File parent;
						try {
							parent = file.getParentFile().getCanonicalFile();
							File root = _root.getCanonicalFile();
							while(!root.equals(parent)) {
								builder.insert(0, "/");
								builder.insert(0, parent.getName());
								parent = parent.getParentFile();
							}
							
							StringSelection selection = new StringSelection(builder.toString());
							clipboard.setContents(selection, null);
						} catch (IOException e1) {
							e1.printStackTrace();
						}
					}
				}
			}
		});
		
		tree.addMouseListener(this);
		tree.addKeyListener(this);
	}
	
	/**
     * Set the filter to use.
     */
    public void setSearchFilter (FileFilter filter)
    {
        this.tree.setSearchFilter(filter);
    }

	@Override
	public void mouseClicked(MouseEvent e) {
		// TODO Auto-generated method stub

	}

	@Override
	public void mousePressed(MouseEvent e) {

		TreePath path = tree.getPathForLocation(e.getX(), e.getY());
		if (path == null) { // JTree上没有任何项被选中
			return;
		}
		tree.setSelectionPath(path);
		if (e.getButton() == 3) {
			menu.show(tree, e.getX(), e.getY());
		}
	}

	@Override
	public void mouseReleased(MouseEvent e) {
		// TODO Auto-generated method stub

	}

	@Override
	public void mouseEntered(MouseEvent e) {
		// TODO Auto-generated method stub

	}

	@Override
	public void mouseExited(MouseEvent e) {
		// TODO Auto-generated method stub

	}

	@Override
	public void keyTyped(KeyEvent e) {
		// TODO Auto-generated method stub
		
	}

	@Override
	public void keyPressed(KeyEvent e) {

		if ((e.getKeyCode() == KeyEvent.VK_C) && e.isControlDown()) {  
			TreePath path = tree.getSelectionPath();
			if (path == null) { // JTree上没有任何项被选中
				return;
			}
			Object obj = path.getLastPathComponent();
			if (obj instanceof FileTreeNode) {
				if (((FileTreeNode) obj).isFile()) {
					File file = ((FileTreeNode) obj).file;
					StringBuilder builder = new StringBuilder(file.getName());
					File parent;
					try {
						parent = file.getParentFile().getCanonicalFile();
						File root = _root.getCanonicalFile();
						while(!root.equals(parent)) {
							builder.insert(0, "/");
							builder.insert(0, parent.getName());
							parent = parent.getParentFile();
						}
						
						StringSelection selection = new StringSelection(builder.toString());
						clipboard.setContents(selection, null);
					} catch (IOException e1) {
						e1.printStackTrace();
					}
				}
				
				e.consume();
			}
        }  
	}

	@Override
	public void keyReleased(KeyEvent e) {
		// TODO Auto-generated method stub
		
	}
	
	public static void main(String[] args) throws Exception {
		File file = new File("../");
		System.out.println(file.getCanonicalPath());
	}
}