//
// $Id$
//
// Clyde library - tools for developing networked games
// Copyright (C) 2005-2012 Three Rings Design, Inc.
// http://code.google.com/p/clyde/
//
// Redistribution and use in source and binary forms, with or without modification, are permitted
// provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice, this list of
//    conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright notice, this list of
//    conditions and the following disclaimer in the documentation and/or other materials provided
//    with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
// INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
// PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
// INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
// TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

package com.threerings.tudey.util;

import java.util.Date;

import com.samskivert.util.IntMap;
import com.samskivert.util.IntMaps;
import com.samskivert.util.ObjectUtil;
import com.threerings.tudey.data.actor.Active;
import com.threerings.tudey.data.actor.Actor;
import com.threerings.tudey.data.actor.Mobile;

/**
 * Advancer for active actors.
 */
public class ActiveAdvancer extends MobileAdvancer
{
    /**
     * Creates a new advancer for the supplied active.
     */
    public ActiveAdvancer (Environment environment, Active active, int timestamp)
    {
        super(environment, active, timestamp);
    }

    /**
     * Determines whether the actor can move.
     */
    public boolean canMove ()
    {
        Activity activity = getActivity();
        return activity == null || activity.allowsMovement();
    }

    /**
     * Determines whether the actor can rotate.
     */
    public boolean canRotate ()
    {
        Activity activity = getActivity();
        return activity == null || activity.allowsRotation();
    }

    @Override // documentation inherited
    public void init (Actor actor, int timestamp)
    {
        super.init(actor, timestamp);
        _active = (Active)actor;
    }

    @Override // documentation inherited
    protected void step (float elapsed)
    {
    	
        Activity activity = getActivity();
        int activityStart = _active.getActivityStarted();
        /*
         * 调试代码
         * if(_lastActivity != null && _active.getActivity() == 0 && _lastActivity._activityStart > activityStart && _lastActivity._activityStart + _lastActivity._clear > _timestamp) {
        	activity = _lastActivity;
        }*/
        
    	if(!ObjectUtil.equals(activity, _lastActivity)) {
    		
    		if(_lastActivity != null) {
    			_lastActivity.stop(elapsed,activity);
    		}
    		
    		_lastActivity = activity;
    		
    		if(activity != null) {
    			activity.start(activityStart);
    		}
    	}
    	
    	
    	// update the current activity
        if (activity != null) {
            activity.step(elapsed);
        }
        super.step(elapsed);
    }

    @Override // documentation inherited
    protected void mobileStep (float elapsed, int timestamp)
    {
        if (!canMove()) {
            _active.clear(Mobile.MOVING);
        }
        super.mobileStep(elapsed, timestamp);
    }

    /**
     * Returns a reference to the mapping for the current activity, or <code>null</code> for none.
     */
    protected Activity getActivity ()
    {
        return _activities.get(_active.getActivity());
    }

    
    protected boolean canStart(int activity, long currentTimeMillis) {
    	Activity act = _activities.get(activity);
    	if(act != null) {
    		return act.canStart(currentTimeMillis);
    	}else {
    		return false;
    	}
    }
    /**
     * The mapping for an activity.
     */
    protected class Activity
    {
        /**
         * Creates a new activity that does not allow movement or rotation.
         *
         * @param clear the interval after which to clear the activity.
         */
        public Activity (int id,int clear)
        {
            this(id,false, clear);
        }

        /**
         * Creates a new activity.
         *
         * @param movement whether or not to allow movement and/or rotation during the activity.
         * @param clear the interval after which to clear the activity.
         */
        public Activity (int id,boolean movement, int clear)
        {
            this(id,movement, movement, clear);
        }

        /**
         * Creates a new activity.
         *
         * @param movement whether or not to allow movement during the activity.
         * @param rotation whether or not to allow rotation during the activity.
         * @param clear the interval after which to clear the activity.
         */
        public Activity (int id,boolean movement, boolean rotation, int clear)
        {
        	this._id = id;
            _movement = movement;
            _rotation = rotation;
            _clear = clear;
        }

        /**
         * Checks whether the activity allows movement.
         */
        public boolean allowsMovement ()
        {
            return _movement;
        }

        /**
         * Checks whether the activity allows rotation.
         */
        public boolean allowsRotation ()
        {
            return _rotation;
        }
        
        /**
         * user input trigger activity to start, this method can check the activity whether or not to start;
         * @param current
         * @return
         */
        public boolean canStart(long current) {
        	return _stopped;
        }
        
        /**
         * activity开始执行的时候调用
         */
        public void start(int activityStart) {
        	_stopped = false;
        	_activityStart = activityStart;
        }
        
        protected void wasStopped(float elapsed,Activity next) {
        	
        }
        
        /**
         * 该方法在2个地方被执行，一个是自动停止（时间>=clear），第二个是强制停止，当其他activity执行的时候，上一个activity将被停止
         * @param elapsed
         */
        protected void stop(float elapsed,Activity next) {
        	if(!_stopped) {
        		wasStopped(elapsed,next);
        		_stopped = true;
        	}
        }

        /**
         * Updates the activity for the current timestamp.
         */
        public void step (float elapsed)
        {
            int started = _active.getActivityStarted();
            if (_timestamp - started >= _clear) {
            	stop(elapsed,null);
            	cleared(started);
            } else if (!canMove()) {
                _active.clear(Mobile.MOVING);
            }
        }

        public void cleared(int started){
        	_active.setActivity(Active.NONE, started + _clear);
        }
        
        /**
         * Allows the activity to respond to input.
         */
        public void updateInput ()
        {
            // nothing by default
        }
        
        public boolean equals(Object obj) {
        	
        	if(!(obj instanceof Activity)) {
        		return false;
        	}
        	
        	return _id == ((Activity)obj)._id;
        }
        
        public int hashCode() {
        	return _id;
        }

        /** 在一个activity的生命周期中表示是否已经停止 */
        protected boolean _stopped = true;
        
        /** Whether or not the activity allows movement. */
        protected boolean _movement;

        /** Whether or not the activity allows rotation. */
        protected boolean _rotation;

        /** The interval after which to clear the activity. */
        protected int _clear;
        
        protected int _activityStart;
        
        protected int _id;
    }

    protected Activity _lastActivity;
    
    /** A casted reference to the active. */
    protected Active _active;

    /** The mappings for the various activities. */
    protected IntMap<Activity> _activities = IntMaps.newHashIntMap();
}
