package com.threerings.swing.filetree;

import java.io.File;
import java.io.FileFilter;
import java.util.Enumeration;

import javax.swing.JTree;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.MutableTreeNode;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;

/**
 * A node in the file tree.
 */
public class FileTreeNode extends DefaultMutableTreeNode {
	/**
	 * Node file.
	 */
	protected File file;

	/** Whether or not this node is expanded in the tree. */
	protected boolean _expanded;

	/**
	 * Indication whether this node corresponds to a file system root.
	 */
	protected boolean isFileSystemRoot;

	private FileFilter _searchFilter;
	
	private FileFilter _baseFilter;
	
	private String nodeName;
	
	/**
	 * Creates a new file tree node.
	 * 
	 * @param children
	 *            Children files.
	 */
	public FileTreeNode(File[] children,FileFilter baseFilter, FileFilter searchFilter) {
		this.file = null;
		this.parent = null;
		this._baseFilter = baseFilter;
		this._searchFilter = searchFilter;
		int j = 0;
		for (int i = 0; i < children.length; i++) {
			if(children[i].isDirectory()){
				boolean isMatched = isAcceptable(children[i],searchFilter);
				FileTreeNode node = new FileTreeNode(children[i], false, this,isMatched,baseFilter, searchFilter);
				if(node.hasChildren() || isMatched){
					this.insert(node, j++);
				}
			}else if(searchFilter == null || searchFilter.accept(children[i])) {
				this.insert(new FileTreeNode(children[i], false, this,false,baseFilter, searchFilter), j++);
			}
		}
	}
	
	/**
	 * Creates a new file tree node.
	 * 
	 * @param file
	 *            Node file
	 * @param isFileSystemRoot
	 *            Indicates whether the file is a file system root.
	 * @param parent
	 *            Parent node.
	 */
	public FileTreeNode(File file, boolean isFileSystemRoot, MutableTreeNode parent, boolean parentMatched ,FileFilter baseFilter,FileFilter searchFilter) {
		super(file, file.isDirectory());
		this.file = file;
		this.setUserObject(file);
		this.isFileSystemRoot = isFileSystemRoot;
		this._baseFilter = baseFilter;
		this._searchFilter = searchFilter;
		int j =0;
		if(file.isDirectory()) {
			File[] children = this.file.listFiles(baseFilter);
			for (int i = 0; i < children.length; i++) {
				boolean isMatched = isAcceptable(children[i],searchFilter);
				FileTreeNode node = new FileTreeNode(children[i], false, this, parentMatched || isMatched,baseFilter,searchFilter);
				if(parentMatched || isMatched){
					this.insert(node, j++);
				}else if(children[i].isDirectory()){
					if(node.hasChildren()){
						this.insert(node, j++);
					}
				}else if(searchFilter == null || searchFilter.accept(children[i])) {
					this.insert(node, j++);
				}
				
			}
		}
	}

	public boolean hasChildren() {
		return file.isFile() || getChildCount() >0;
	}
	
	public void refresh() {
		this.removeAllChildren();
		if(file.isDirectory()) {
			File[] children = this.file.listFiles(_baseFilter);
			int j = 0;
			for (int i = 0; i < children.length; i++) {
				FileTreeNode node = new FileTreeNode(children[i], false, this,true,_baseFilter ,_searchFilter);
				if(node.hasChildren()) {
					this.insert(node, j++);
				}
			}
		}
	}

	public String getRelativePath() {
		String path = file.getName();
		TreeNode parent = this.parent;
		while (parent != null && parent instanceof FileTreeNode) {
			if (((FileTreeNode) parent).file != null) {
				path = ((FileTreeNode) parent).file.getName() + "/" + path;
			}
			parent = parent.getParent();
		}
		return path;
	}

	public boolean isFile() {
		return file.isFile();
	}

	private static boolean isAcceptable(File file,FileFilter filter){
		if(filter == null || filter.accept(file)){
			return true;
		}
		return false;
	}
	
	public boolean isLeaf() {
		return this.file != null && file.isFile();
	}
	
	 /**
     * Sets whether or not this node is expanded in the tree.
     */
    public void setExpanded (boolean expanded)
    {
        _expanded = expanded;
    }

    
    /**
     * Expands paths according to the {@link #_expanded} field.
     */
    public void expandPaths (JTree tree)
    {
        if (_expanded) {
            tree.expandPath(new TreePath(getPath()));
            if (children != null) {
                for (Object child : children) {
                    ((FileTreeNode)child).expandPaths(tree);
                }
            }
        }
    }

	/**
     * Expands all paths up to the specified depth.
     */
    public void expandPaths (JTree tree, int depth)
    {
        if (!getAllowsChildren()) {
            return;
        }
        tree.expandPath(new TreePath(getPath()));
        if (depth-- > 0 && children != null) {
            for (Object child : children) {
                ((FileTreeNode)child).expandPaths(tree, depth);
            }
        }
    }

	public FileTreeNode getNode(String name) {
		Enumeration<TreeNode> e = children();
		while (e.hasMoreElements()) {
			FileTreeNode node = (FileTreeNode)e.nextElement();
			if (node.getName().equals(name)) {
				return node;
			}else {
				int index = name.indexOf(node.getName());
				if(index == 0) {
					return node.getNode(name);
				}
			}
		}
		return null;
	}

	public String getName() {
		if(nodeName == null) {
			FileTreeNode parent = (FileTreeNode) this.getParent();
			StringBuilder buff = new StringBuilder(this.file.getName());
	
			while (parent != null && parent.file != null) {
				if(parent.file != null) {
					buff.insert(0, parent.file.getName() + "/");
				}
				
				parent = (FileTreeNode) parent.getParent();
			}
			
			if(this.file.isDirectory()) {
				buff.append("/");
			}
			
			return  nodeName = buff.toString();
		}else {
			return nodeName;
		}
	}
	
	 /**
     * Returns the index at which the specified node should be inserted to maintain the sort
     * order.
     */
    public int getInsertionIndex (FileTreeNode child)
    {
        if (children == null) {
            return 0;
        }
        File file = (File)child.getUserObject();
        boolean folder = child.getAllowsChildren();
        for (int ii = 0, nn = children.size(); ii < nn; ii++) {
        	FileTreeNode ochild = (FileTreeNode)children.get(ii);
            File oname = (File)ochild.getUserObject();
            boolean ofolder = ochild.getAllowsChildren();
            if ((folder == ofolder) ? (file.compareTo(oname) <= 0) : !folder) {
                return ii;
            }
        }
        return children.size();
    }
    
    public String toString() {
    	if(this.file == null) {
    		return super.toString();
    	}
    	return this.getName();
    }

}