//
// $Id$
//
// Clyde library - tools for developing networked games
// Copyright (C) 2005-2012 Three Rings Design, Inc.
// http://code.google.com/p/clyde/
//
// Redistribution and use in source and binary forms, with or without modification, are permitted
// provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice, this list of
//    conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright notice, this list of
//    conditions and the following disclaimer in the documentation and/or other materials provided
//    with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
// INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
// PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
// INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
// TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

package com.threerings.tudey.tools;

import java.awt.event.MouseEvent;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

import com.google.common.base.Predicate;
import com.google.common.collect.Maps;
import com.google.inject.internal.Lists;
import com.samskivert.util.ArrayUtil;
import com.threerings.config.ConfigReference;
import com.threerings.editor.Editable;
import com.threerings.math.FloatMath;
import com.threerings.math.Vector3f;
import com.threerings.opengl.model.Model;
import com.threerings.opengl.scene.SceneElement;
import com.threerings.tudey.client.sprite.GraphSprite;
import com.threerings.tudey.config.GraphConfig;
import com.threerings.tudey.data.TudeySceneModel;
import com.threerings.tudey.data.TudeySceneModel.Edge;
import com.threerings.tudey.data.TudeySceneModel.Entry;
import com.threerings.tudey.data.TudeySceneModel.GraphEntry;
import com.threerings.tudey.data.TudeySceneModel.Vertex;

/**
 * The Graph definer tool.
 */
public class GraphDefiner extends ConfigTool<GraphConfig>
{
	private static final long serialVersionUID = 1L;

	/**
     * Creates the path definer tool.
     */
    public GraphDefiner (SceneEditor editor)
    {
        super(editor, GraphConfig.class, new GraphReference());
    }

    @Override // documentation inherited
    public void deactivate ()
    {
        // release any vertex being moved
        super.deactivate();
        if (_entry != null) {
            release(_entry);
        }
    }

    @Override // documentation inherited
    public void sceneChanged (TudeySceneModel scene)
    {
        super.sceneChanged(scene);
        _entry = null;
        _idx = -1;
        _edge = null;
        for(Entry entry : scene.getEntries()) {
        	if(entry instanceof GraphEntry) {
        		_entry = (GraphEntry)entry;
        		break;
        	}
        }
        
        if(_entry == null) {
        	_entry = new GraphEntry();
        	scene.addEntry(_entry);
        }
    }
    
    @Override // documentation inherited
    public void entryUpdated (Entry oentry, Entry nentry)
    {
        if (_entry != null && _entry.getKey().equals(oentry.getKey()) && _entry != nentry) {
            _entry = (GraphEntry)nentry;
        }
    }
    
    /**
     * <pre>
     * 当鼠标左键点地图:
     * 1.如果鼠标左键点中顶点，则以这个顶点开始构建下一个点以及边
     * 2.如果鼠标左键点中边，则增加一个顶点，并且割裂这条边，并且开始构建下一个顶点以及边
     * 3.如果鼠标左键啥都没有点到，则增加一个顶点，并且开始构建下一个顶点以及边
     * 4.如果鼠标右键，则表示为删除
     * </pre>
     */

    @Override // documentation inherited
    public void mousePressed (MouseEvent event)
    {
        if (_editor.isSpecialDown()) {
            return;
        }
        int button = event.getButton();
        
       /* if (button == MouseEvent.BUTTON3 && _idx != -1) { // remove the vertex
            release();
        }*/
        
        if (_editor.getMouseRay(_pick)) {
        	
        	
        	//鼠标左键
        	if (button == MouseEvent.BUTTON1) {
                Model model = (Model)_editor.getView().getScene().getIntersection( _pick, _isect, Graph_FILTER);
                GraphEntry entry = (GraphEntry)_entry.clone();
                if (model != null) {
                	
                    GraphSprite sprite = (GraphSprite)model.getUserObject();
                    int idx = sprite.getVertexIndex(model);
                    	//是顶点：
                    if (idx != -1) {
                    	//有移动点
                    	if(_idx != -1) {
                    		
                    		//检测是否是相连的点
                    		List<Integer> oneLevelNeighbor = Lists.newArrayList();
                    		List<Integer> twoLevelNeighbor = Lists.newArrayList();
                    		
                    		for(Edge edge : entry.edges) {
                    			if(edge.end == _idx) {
                    				oneLevelNeighbor.add(edge.start);
                    			}
                    			
                    			if(edge.start == _idx) {
                    				oneLevelNeighbor.add(edge.end);
                    			}
                    		}
                    		
                    		for(Edge edge : entry.edges) {
                    			if(oneLevelNeighbor.contains(edge.start)) {
                    				twoLevelNeighbor.add(edge.end);
                    			}
                    			
                    			if(oneLevelNeighbor.contains(edge.end)) {
                    				twoLevelNeighbor.add(edge.start);
                    			}
                    		}

                    		if(oneLevelNeighbor.contains(idx) || twoLevelNeighbor.contains(idx)) {
                    			return;
                    		}
                    		
                    		System.out.println("idx="+idx);
                    		//------------------------
                    		
                    		for(Edge edge : entry.edges) {
                    			if(edge.end == _idx) {
                    				edge.end = idx;
                    			}
                    			
                    			if(edge.start == _idx) {
                    				edge.start = idx;
                    			}
                    		}
                    		_edge = insertEdge(entry, idx, _idx);
                    		_editor.updateEntries(entry);
                    	}else {
                    		//无移动点
                    		
                    		/*int newIdx = _entry.vertices.length;
                    		insertVertex(_entry, newIdx);
                    		_idx = newIdx;
                    		_edge = insertEdge(_entry, idx, newIdx);*/
                    		
                    		//仅仅是选择一个点作为可移动点
                    		_idx = idx;
                    		
                    	}
                        return;
                    }else {
                    	//是边：
                        idx = sprite.getEdgeIndex(model);
                        //获得边
                        Edge edge = entry.edges[idx];
                        //有移动的点
                        if(_idx != -1) {
                        	
                        	//将移动的点固定在这个边上
                        	setMouseLocation(_entry.vertices[_idx]);
                        	
                        	//拆分边
                    		int newIdx = entry.vertices.length;
                    		int newStart = edge.start;
                    		edge.start = _idx;
                    		insertEdge(entry, newStart, _idx);
                    		
                    		//插入一个新的点
                            insertVertex(entry, newIdx);
                            //链接老点与新点
                            _edge = insertEdge(entry, _idx, newIdx);
                            
                            //使新的点可移动
                            _idx = newIdx;
                        }else {
                        	//无移动点
                        	//拆分边
                    		int one = entry.vertices.length;
                    		int newEnd = edge.end;
                    		//插入一个点
                            insertVertex(entry, one);
                            //修改1条边
                            edge.end = one;
                            insertEdge(entry, one,newEnd);
                            //end 修改边
                            
                            //插入一个移动点
                            int two = entry.vertices.length;
                            insertVertex(entry, two);
                            
                            _edge = insertEdge(entry, one, two);
                            _idx = two;
                           
                        }
                        _editor.updateEntries(entry);
                        return;
                    }
                } else {
                	//点了空地
                	
                	//无移动点
                	 if (_idx == -1) { // insert in between
                     	 int newIdx = entry.vertices.length;
                         insertVertex(entry, newIdx);
                         insertVertex(entry, newIdx+1);
                         _edge = insertEdge(entry, newIdx, newIdx+1);
                         _idx = newIdx+1;
                         //System.out.println("无移动点,_idx="+_idx);
                         _editor.updateEntries(entry);
                     } else {
                    	 
                    //有移动点
                    	 int newIdx = entry.vertices.length;
                         insertVertex(entry, newIdx);
                         _edge = insertEdge(entry, _idx, newIdx);
                         _idx = newIdx;
                         //System.out.println("有移动点,_idx="+_idx);
                         _editor.updateEntries(entry);
                     }
                     return;
                }
        	}else if(button == MouseEvent.BUTTON3){
        		 GraphEntry entry = (GraphEntry)_entry.clone();
        		if(_idx != -1) {
        			release(entry);
        		}else {
        			Model model = (Model)_editor.getView().getScene().getIntersection( _pick, _isect, Graph_FILTER);
                    if (model != null) {
                    	
                        GraphSprite sprite = (GraphSprite)model.getUserObject();
                        int idx = sprite.getVertexIndex(model);
                        	//是顶点：
                        if (idx != -1) {
                        	removeVertices(entry, idx);
                        }else {
                        	idx = sprite.getEdgeIndex(model);
                        	Edge edge = entry.edges[idx];
                        	removeEdge(entry,edge);
                        }
                        _editor.updateEntries(entry);
                    }
                    _edge = null;
                	_idx = -1;
        		}
        		
        	}else if (button == MouseEvent.BUTTON2) {
        		Model model = (Model)_editor.getView().getScene().getIntersection(_pick, _isect, Graph_FILTER);
                if (model != null) {
                    GraphSprite sprite = (GraphSprite)model.getUserObject();
                    int idx = sprite.getVertexIndex(model);
                    	//是顶点：
                    if(_idx == -1) {
                    	if(idx != -1) {
                    		_idx = idx;
                    	}
                    }else {
                    	if(idx == -1) {
                    		setMouseLocation(_entry.vertices[_idx]);
                    		_idx = -1;
                    	}
                    }
                }else {
                	if(_idx != -1) {
                		setMouseLocation(_entry.vertices[_idx]);
                		_idx = -1;
                	}
                }
        	}
        }
    }

    @Override // documentation inherited
    public void tick (float elapsed)
    {
        if (_entry == null || !getMousePlaneIntersection(_isect) || _editor.isSpecialDown()) {
            return;
        }

        if(_entry.vertices.length == 0) {
        	return;
        }
        
        if(_idx != -1) {
        	setMouseLocation(_entry.vertices[_idx]);
        	_editor.updateEntries(_entry);
        }
    }

    /**
     * Sets the location of the specified vertex to the one indicated by the mouse cursor.
     */
    protected void setMouseLocation (Vertex vertex)
    {
        // snap to tile grid if shift not held down
        if (!_editor.isShiftDown()) {
            _isect.x = FloatMath.floor(_isect.x) + 0.5f;
            _isect.y = FloatMath.floor(_isect.y) + 0.5f;
        }
        float z = Math.max( _editor.getGrid().getZ(), 10);
        z = Math.min(z, -10);
        vertex.set(_isect.x, _isect.y, _editor.getGrid().getZ());
    }

    /**
     * Releases the vertex being moved.
     */
    protected void release (GraphEntry entry)
    {
    	if(_idx != -1) {
    		removeVertices(entry, _idx);
        	_idx = -1;
        	_edge = null;
        	_editor.updateEntries(entry);
    	}
    }

    /**
     * Inserts a new vertex at the specified location and starts moving it.
     */
    protected void insertVertex (GraphEntry entry, int idx)
    {
    	if(_editor.getGrid().getZ() > 50 || _editor.getGrid().getZ() < -50) {
    		return ;
    	}
        Vertex  vertex = new Vertex();
        setMouseLocation(vertex);
        entry.vertices = ArrayUtil.insert(entry.vertices, vertex, idx);
        
        System.out.println("insert Vertex="+vertex+",idx="+idx);
        
        for(Edge edge : entry.edges) {
        	if(edge.start >= idx) {
        		edge.start = edge.start + 1;
        	}
        	
        	if(edge.end >= idx) {
        		edge.end = edge.end + 1;
        	}
        }
    }
    
    /**
     * Inserts a new vertex at the specified location and starts moving it.
     */
    protected Edge insertEdge (GraphEntry entry, int start, int end)
    {
    	
        Edge edge = new Edge();
        edge.set(start, end);
        entry.edges = ArrayUtil.append(entry.edges,edge);
        return edge;
    }

    /**
     * Removes the indexed vertices from the supplied entry (removing the entry itself if it has no
     * more vertices).
     */
    protected void removeVertices (GraphEntry nentry, int idx)
    {
    	Map<Integer,AtomicInteger> checked = Maps.newHashMap();
    	
        List<Edge> list = Lists.newArrayList();
        for(Edge edge : nentry.edges) {
        	if(edge.end != idx && edge.start != idx) {
        		list.add(edge);
        		
        		AtomicInteger counter = checked.get(edge.start);
        		if(counter == null) {
        			checked.put(edge.start, counter = new AtomicInteger(1));
        		}else {
        			counter.incrementAndGet();
        		}
        		
        		counter = checked.get(edge.end);
        		if(counter == null) {
        			checked.put(edge.end, counter = new AtomicInteger(1));
        		}else {
        			counter.incrementAndGet();
        		}
        		
        	}else if((edge.end == idx) || edge.start == idx){
        		Integer key = edge.start == idx ? edge.end : edge.start;
        		AtomicInteger counter = checked.get(key);
        		if(counter == null) {
        			checked.put(key, counter = new AtomicInteger(0));
        		}else {
        			counter.incrementAndGet();
        		}
        	}
        	
        }
        
        
        nentry.edges = list.toArray(new Edge[list.size()]);
        
        List<Vertex> vertics = Lists.newArrayList();
        
        for(int i =nentry.vertices.length - 1 ;i >= 0;i--) {
        	AtomicInteger counter = checked.get(i);
        	if(counter != null && counter.get() > 0 && i != idx) {
        		vertics.add(nentry.vertices[i]);
        	}else {
        		for(Edge edge : nentry.edges) {
        			if(edge.start>i) {
                		edge.start -=1; 
                	}
                	
                	if(edge.end>i) {
                		edge.end -=1; 
                	}
        		}
        	}
        }
        
        nentry.vertices = vertics.toArray(new Vertex[vertics.size()]);
        ArrayUtil.reverse(nentry.vertices);
    }
    
    protected void removeEdge (GraphEntry nentry, Edge removed)
    {
            int start = 0; 
            int end =0;
            
            Edge[] edges = new Edge[nentry.edges.length -1];
            int idx = 0;
            for(Edge edge : nentry.edges) {
            	if(edge != removed) {
            		edges[idx++] = edge;
            		
            		if(edge.end == removed.end || edge.start == removed.end) {
            			end ++;
            		}
            		
            		if(edge.start == removed.start || edge.end == removed.start) {
            			start ++;
            		}
            	}
            }
            nentry.edges = edges;
            
            if(removed.start > removed.end) {
	            if(start == 0) {
	            	removeVertices(nentry,removed.start);
	            }
	            
	            if(end == 0) {
	            	removeVertices(nentry,removed.end);
	            }
            }else {
            	if(end == 0) {
	            	removeVertices(nentry,removed.end);
	            }
            	if(start == 0) {
	            	removeVertices(nentry,removed.start);
	            }
            }
    }

    /**
     * Allows us to edit the path reference.
     */
    protected static class GraphReference extends EditableReference<GraphConfig>
    {
        /** The path reference. */
        @Editable(nullable=true)
        public ConfigReference<GraphConfig> path;

        @Override // documentation inherited
        public ConfigReference<GraphConfig> getReference ()
        {
            return path;
        }

        @Override // documentation inherited
        public void setReference (ConfigReference<GraphConfig> ref)
        {
            path = ref;
        }
    }

    /** The entry containing the vertex we're moving, if any. */
    protected GraphEntry _entry;

    /** The index of the vertex we're moving. */
    protected int _idx;
    
    protected Edge _edge;

    /** Holds the result of an intersection test. */
    protected Vector3f _isect = new Vector3f();

    /** A filter that only passes path vertex or edge models. */
    protected final Predicate<SceneElement> Graph_FILTER = new Predicate<SceneElement>() {
        public boolean apply (SceneElement element) {
        	 Object obj = element.getUserObject();
        	if (obj instanceof GraphSprite) {
        		int idx = ((GraphSprite)obj).getVertexIndex((Model)element);
        		
        		if(idx != -1) {
	        		if(_edge != null) {
	        			for(Edge edge : _entry.edges) {
	        				if((edge.start == _edge.start || edge.end == _edge.start || edge.end == _edge.start || edge.end == _edge.end) && (idx == edge.end || idx == edge.start)) {
	        					return false;
	        				}
	        			}
	        		}
	        		
	        		if(_idx >=0 && idx == _idx) {
	        			return false;
	        		}
	        		
        		}else {
	        		idx = ((GraphSprite)obj).getEdgeIndex((Model)element);
	        		
	        		if(idx != -1) {
	        			Edge edge = _entry.edges[idx];
	        			
	        			if(edge.start == _idx || edge.end == _idx) {
	        				return false;
	        			}
	        			
	        		}
        		}
        		
        		return  _editor.getLayerPredicate().apply(((GraphSprite) obj).getEntry());
        	}
        		
        	return false;
        }
    };
}
