package com.threerings.tudey.server.graph;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;

import com.google.common.collect.Lists;
import com.threerings.math.Vector2f;
import com.threerings.tudey.data.TudeySceneModel;
import com.threerings.tudey.data.TudeySceneModel.GraphEntry;
import com.threerings.tudey.shape.Capsule;

public class SimpleGraphFinder implements GraphFinder {
	private List<Node> nodes; // stores all the nodes.
	private GraphEntry entry = null;
	private int sceneId ;
	private TudeySceneModel _model;
	public SimpleGraphFinder(TudeySceneModel model) {
		_model = model;
		sceneId = model.sceneId;
		
		Iterable<GraphEntry> ge = model.getEntries(GraphEntry.class);

		Iterator<GraphEntry> it = ge.iterator();
		if (it.hasNext()) {
			entry = it.next();
		}
		
		if (entry == null || entry.vertices.length == 0 || entry.edges.length == 0) {
			return;
		}
		// set up graph nodes
		setupGraphNodes();
	}

	private Node findStartNode(Vector2f location) {
		float dist = Integer.MAX_VALUE;
		int radius = 10;
		int mask = 1<<0 | 1<<1;
		Node start = new Node("START",location,1);
		Capsule segment = new Capsule();
		segment.radius = 0.35f;
		Node result = null;
		for (Node node : nodes) {
			float dd = node.getLocation().distance(location);
			if (dd < dist) {
				result = node;
				dist = dd;
			}
			
			if(dd < radius) {
				segment.getStart().set(location);
				segment.getEnd().set(node.getLocation());
				segment.updateBounds();
				if(!_model.collides(mask, segment)) {
					start.addNeighbor(node,(int) dd);
				}
			}
		}
		
		if(start.getNeighbors().size() != 0) {
			return start;
		}

		return result;

	}
	
	private Node findNearestNode(Vector2f location) {
		float dist = Integer.MAX_VALUE;
		Node result = null;
		for (Node node : nodes) {
			float dd = node.getLocation().distance(location);
			if (dd < dist) {
				result = node;
				dist = dd;
			}
		}

		return result;

	}


	/* (non-Javadoc)
	 * @see com.threerings.tudey.server.graph.GraphFinder#runAStar(com.threerings.math.Vector2f, com.threerings.math.Vector2f)
	 */
	@Override
	public Vector2f[] runAStar(Vector2f start, Vector2f end) {
		if (entry == null || entry.vertices.length == 0 || entry.edges.length == 0) {
			return null;
		}
		Node startNode = findStartNode(start); // get selected node

		resetNode(startNode); // reset source field, if exists (avoid errors)

		List<Vector2f> result = AStarSearch(startNode, findNearestNode(end)); // goal = Bucharest.

		result.add(0, end);

		return Lists.reverse(result).toArray(new Vector2f[result.size()]);
	}

	/* (non-Javadoc)
	 * @see com.threerings.tudey.server.graph.GraphFinder#runBreadthFirst(com.threerings.math.Vector2f, com.threerings.math.Vector2f)
	 */
	@Override
	public Vector2f[] runBreadthFirst(Vector2f start, Vector2f end) {
		
		if (entry == null || entry.vertices.length == 0 || entry.edges.length == 0) {
			return null;
		}
		
		Node startNode = findStartNode(start); // get selected node
		resetNode(startNode); // reset source field, if exists (avoid errors)
		
		List<Vector2f> result = BreadthFirstSearch(startNode, findNearestNode(end)); // goal = Bucharest.
		result.add(0, end);
		
		return Lists.reverse(result).toArray(new Vector2f[result.size()]);
	}

	private void setupGraphNodes() {
		Node[] nodeArrays = new Node[entry.vertices.length];
		for (int i = 0; i < entry.vertices.length; i++) {
			nodeArrays[i] = new Node("" + i, entry.vertices[i].createVector(), 1);
		}
		for (com.threerings.tudey.data.TudeySceneModel.Edge edge : entry.edges) {
			int dist = (int) nodeArrays[edge.end].getLocation().distance(nodeArrays[edge.start].getLocation());
			nodeArrays[edge.start].addNeighbor(nodeArrays[edge.end], dist);
			nodeArrays[edge.end].addNeighbor(nodeArrays[edge.start], dist);
		}
		
		checkNode(nodeArrays[0]);
		boolean hasError = false;
		for(int i = 0 ;i< nodeArrays.length;i++) {
			if(!nodeArrays[i].isChecked()) {
				System.out.println("sceneId="+sceneId+", Node="+nodeArrays[i].getName()+" is invalid");
				hasError = true;
			}
		}
		
		if(!hasError) {
			System.out.println("sceneId="+sceneId+" all nodes are valid");
		}
		
		nodes = Arrays.asList(nodeArrays);
	}
	
	private void checkNode(Node n) {
		for(Node node : n.getNeighbors().keySet()) {
			if(!node.isChecked()) {
				node.setChecked(true);
				checkNode(node);
			}
		}
	}

	// ALGORITHMS \\
	private List<Vector2f> AStarSearch(Node start, Node goal) {
		Map<Node, Integer> nodeQueue = new HashMap(); // works as a priority queue. First index: node with minimum total score.
		List<Node> expanded = new ArrayList(); // keeps track of the nodes which are expanded.

		int startScore = start.getHeuristic() + 0; // initialize start node total score.
		nodeQueue.put(start, startScore);
		Node current = start; // current keeps track of the node that is currently being processed.

		// while current node is not the goal node.
		while (!current.getName().equals(goal.getName())) {
			// iterate through the neighbors of the current node.
			for (Node neighbor : current.getNeighbors().keySet()) {
				// if this neighbor is already expanded, ignore it.
				if (expanded.contains(neighbor)) {
					continue;
				}
				// if the queue contains the neighbor.
				if (nodeQueue.containsKey(neighbor)) {
					// find the total score of the neighbor from the current node.
					int tempNeighborScore = nodeQueue.get(current) - current.getHeuristic()
							+ current.getNeighbors().get(neighbor) + neighbor.getHeuristic();

					int existedNodeScore = nodeQueue.get(neighbor); // get neighbor's total score from the queue.

					if (tempNeighborScore < existedNodeScore) { // if the new distance is less than the one in the
																// queue:
						neighbor.setSource(current); // neighbor's new source = current.
						nodeQueue.put(neighbor, tempNeighborScore); // and update neighbor's total score in the queue.
					}
				} else { // the map-queue does not contain the neighbor node.
					neighbor.setSource(current); // neighbor's source = current.
					// calculate neighbor's total score.
					int neighborScore = nodeQueue.get(neighbor.getSource()) - neighbor.getSource().getHeuristic()
							+ neighbor.getSource().getNeighbors().get(neighbor) + neighbor.getHeuristic();

					// add the neighbor with its score to the queue-map.
					nodeQueue.put(neighbor, neighborScore);
				}
			}

			expanded.add(current); // current has been expanded, so add him to the expanded list.
			nodeQueue.remove(current); // we remove from the queue the current node. (which had the minimum total
										// score)

			current = getMinValue(nodeQueue); // get the node with the minimum total score and make it the current one.
		}

		Node tracker = current; // used to trace the path back.

		List<Vector2f> result = Lists.newArrayList();
		result.add(goal.getLocation());
		while (tracker.getSource() != null) {
			result.add(tracker.getSource().getLocation());
			tracker = tracker.getSource();
		}


		return result;
	}

	private List<Vector2f> BreadthFirstSearch(Node start, Node goal) {
		// some comments are mutual with AStarSearch algorithm so I won't repeat myself.

		Queue<Node> nodeQueue = new LinkedList(); // queue that strores nodes to be expanded.
		List<Node> expanded = new ArrayList();

		Node current = start;

		while (!current.getName().equals(goal.getName())) {
			for (Node neighbor : current.getNeighbors().keySet()) {
				// if the neighbor is already expanded OR the neighbor is already in the queue
				// to be expanded in the future, ignore the neighbor.
				if (expanded.contains(neighbor) || nodeQueue.contains(neighbor)) {
					continue;
				}

				neighbor.setSource(current);
				nodeQueue.add(neighbor);
			}

			expanded.add(current);
			current = nodeQueue.remove();
		}

		Node tracker = current;

		List<Vector2f> result = Lists.newArrayList();
		result.add(goal.getLocation());

		while (tracker.getSource() != null) {

			result.add(tracker.getSource().getLocation());

			tracker = tracker.getSource();
		}


		return result;
	}

	private Node getMinValue(Map<Node, Integer> treeMap) {
		Node minNode = null;
		int min = Integer.MAX_VALUE;

		for (Map.Entry<Node, Integer> entry : treeMap.entrySet()) {
			Node node = entry.getKey();
			Integer value = entry.getValue();

			if (value < min) {
				min = value;
				minNode = node;
			}
		}
		return minNode;
	}

	// set nodes's source field null
	private void resetNode(Node startNode) {
		if (startNode.getSource() != null) {
			startNode.setSource(null);
		}
	}
}