//
// $Id$
//
// Clyde library - tools for developing networked games
// Copyright (C) 2005-2012 Three Rings Design, Inc.
// http://code.google.com/p/clyde/
//
// Redistribution and use in source and binary forms, with or without modification, are permitted
// provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice, this list of
//    conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright notice, this list of
//    conditions and the following disclaimer in the documentation and/or other materials provided
//    with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
// INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
// PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
// INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
// TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

package com.threerings.tudey.util;

import java.util.Date;

import com.threerings.math.Vector2f;
import com.threerings.tudey.data.InputFrame;
import com.threerings.tudey.data.actor.Actor;
import com.threerings.tudey.data.actor.Agent;
import com.threerings.tudey.data.actor.Mobile;
import com.threerings.tudey.data.actor.Pawn;
import com.threerings.tudey.shape.Shape;

/**
 * Used on the client and the server to advance the state of a pawn based on its inputs and
 * surroundings.
 */
public class PawnAdvancer extends ActiveAdvancer
{
    /**
     * Creates a new advancer for the supplied pawn.
     */
    public PawnAdvancer (Environment environment, Pawn pawn, int timestamp)
    {
        super(environment, pawn, timestamp);
        this._activities.put(Pawn.LOADING, new ActiveAdvancer.Activity(2147483647));
		this._activities.put(Pawn.USE_INTERACT, new ActiveAdvancer.Activity(670));
    }

    /**
     * Advances to the timestamp of the provided input frame and sets the pawn's current input.
     */
    public void advance (InputFrame frame)
    {
        // save the input frame and advance
        _frame = frame;
        setPath(_frame.getPath());
        advance(frame.getTimestamp());
    }

    public boolean inputFrameCompeleted(){
    	return _frame == null;
    }
    
    @Override // documentation inherited
    public void init (Actor actor, int timestamp)
    {
        super.init(actor, timestamp);
        _pawn = (Pawn)actor;
        _frame = null;
    }

    @Override // documentation inherited
    protected void step (float elapsed)
    {
        super.step(elapsed);

        // update based on most recent input
        if (_frame != null) {
            updateInput();
        }
    }

    @Override // documentation inherited
    protected void takeSubsteps (float elapsed)
    {
        // if the input frame provides a computed position, we shall attempt to validate
        if (_frame == null || _timestamp != _frame.getTimestamp() ||
                _frame.getTranslation() == null || ignoreInputPosition()) {
            super.takeSubsteps(elapsed);
            return;
        }

        // make sure this is cleared in case we don't take any mobile steps
        if (!canMove()) {
            _active.clear(Mobile.MOVING);
        }

        // make sure they haven't exceeded their speed
        Vector2f ptrans = _pawn.getTranslation();
        Vector2f ftrans = _frame.getTranslation();
        float distance = ptrans.distance(ftrans);
        if (distance == 0f) {
            return; // no movement, no problem
        }
        if (distance > _pawn.getSpeed() * elapsed + 0.5f) {
            super.takeSubsteps(elapsed);
            return;
        }

        // make sure they didn't run into anything
        updateShape();
        _swept = _shape.sweep(ftrans.subtract(ptrans, _penetration), _swept);
        if (_environment.collides(_pawn, _swept)) {
            super.takeSubsteps(elapsed);
            return;
        }

        // otherwise, assume validity
        ptrans.set(ftrans);
        _pawn.setDirty(true);
    }

    /**
     * Checks whether we should ignore the input position and instead perform the full movement
     * simulation.
     */
    protected boolean ignoreInputPosition ()
    {
        return false;
    }
    
    /**
     * Returns the radius within which we can be consider ourselves to have reached a node
     * (which depends on the actor's speed, since it's possible to overshoot).
     */
    protected float getReachRadius (int ticksPerSecond)
    {
        // radius is the distance we can travel in a single tick
        float speed = ((Mobile)getActor()).getSpeed();
        return speed / ticksPerSecond;
    }
    

    /**
     * Updates the pawn's state based on the current input frame.
     */
    protected void updateInput ()
    {
        Activity activity = getActivity();
        if (activity != null) {
            activity.updateInput();
        }
        if (canRotate()) {
            updateRotation();
        }
        
        Vector2f[] path = _frame.getPath();
        if ((path != null ||_frame.isSet(InputFrame.MOVE)) && canMove()) {
        	
        	if(path != null){
                setPath(_frame.getPath());
               _pawn.set(Mobile.MOVING);
        	}else{
                clearPath();
                _pawn.setDirection(_frame.getDirection());
                _pawn.set(Mobile.MOVING);
        	}
        } else {
            _pawn.clear(Mobile.MOVING);
        }
        
     // see if we've reached the current node (looping around in case the notification
        // sets us on a new path)
        
        
        /*if(_path == null){
        	return;
        }
        
        Vector2f trans = _actor.getTranslation();
        boolean completedPath = false;
        System.out.println("dist="+_path[_pidx].distance(trans)+",speed="+getReachRadius(50));
        while (_path[_pidx].distance(trans) <= getReachRadius(50)) {
            if (++_pidx == _path.length) {
            	_pawn.clear(Mobile.MOVING);
                _path = null;
                // If we've already completed a path then just exit to prevent an infinite loop
                if (completedPath) {
                    return;
                }
                _frame = null;
                completedPath = true;
            }
            
            if (_path == null) {
                return;
            }
        }
        // make sure we're facing the right direction
        Vector2f node = _path[_pidx];
        
        float rotation = _actor.getTranslation().direction(node);
        
        if (canRotate() && canMove()) {
        	_pawn.setRotation(rotation);
        	_pawn.setDirection(rotation);
            _pawn.set(Mobile.MOVING);
        }else{
        	_pawn.clear(Mobile.MOVING);
        }*/
        
        /*float rot = FloatMath.atan2(node.y - trans.y, node.x - trans.x);
        float dist = FloatMath.getAngularDistance(_actor.getRotation(), rot);
        System.out.println(node+","+trans+",hashcode="+this._frame.hashCode());
        if (dist > 0.0001f) {
            setTargetRotation(rot);
        }
        
        if (dist < 0.001f) {
        	_actor.clear(Mobile.MOVING);
        } else {
        	_actor.set(Mobile.MOVING);
        }*/
    }
    
    public void setTargetRotation (float rotation)
    {
        if (rotation == _actor.getRotation()) {
        	// clear turn direction
            ((Agent)_actor).setTurnDirection(0);
        }
    }
    

    /**
     * Called to update a pawns rotation based on input.
     */
    protected void updateRotation ()
    {
        _pawn.setRotation(_frame.getRotation());
    }

    /**
     * Sets the path to follow.
     */
    protected void setPath (Vector2f[] path)
    {
        if (_path != path) {
            _path = path;
            _pidx = 0;
        }
    }

    /**
     * Clears the path.
     */
    protected void clearPath ()
    {
        if (_path != null) {
            _pawn.stopMoving();
            _path = null;
        }
    }
    
    /** The waypoints of the path being followed. */
    protected Vector2f[] _path;

    /** The index of the next point on the path. */
    protected int _pidx;
    
    /** A casted reference to the pawn. */
    protected Pawn _pawn;

    /** The most current input frame. */
    protected InputFrame _frame;

    /** Holds the actor's swept shape. */
    protected Shape _swept;
}
