//
// $Id$
//
// Clyde library - tools for developing networked games
// Copyright (C) 2005-2012 Three Rings Design, Inc.
// http://code.google.com/p/clyde/
//
// Redistribution and use in source and binary forms, with or without modification, are permitted
// provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice, this list of
//    conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright notice, this list of
//    conditions and the following disclaimer in the documentation and/or other materials provided
//    with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
// INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
// PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
// INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
// TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

package com.threerings.config.tools;

import static com.threerings.ClydeLog.log;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.EventQueue;
import java.awt.Rectangle;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.ClipboardOwner;
import java.awt.datatransfer.Transferable;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;

import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.InputMap;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JComboBox;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTabbedPane;
import javax.swing.KeyStroke;
import javax.swing.SwingUtilities;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.MenuEvent;
import javax.swing.event.MenuListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.filechooser.FileFilter;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreePath;

import com.samskivert.swing.GroupLayout;
import com.samskivert.swing.VGroupLayout;
import com.samskivert.swing.util.SwingUtil;
import com.samskivert.util.QuickSort;
import com.threerings.config.ConfigGroup;
import com.threerings.config.ConfigManager;
import com.threerings.config.ManagedConfig;
import com.threerings.config.swing.ConfigTree;
import com.threerings.config.swing.ConfigTreeFilterPanel;
import com.threerings.config.swing.ConfigTreeNode;
import com.threerings.editor.swing.BaseEditorPanel;
import com.threerings.editor.swing.EditorPanel;
import com.threerings.editor.swing.TreeEditorPanel;
import com.threerings.editor.util.EditorContext;
import com.threerings.media.image.ColorPository;
import com.threerings.resource.ResourceManager;
import com.threerings.swing.PrintStreamDialog;
import com.threerings.util.MessageManager;
import com.threerings.util.ToolUtil;

/**
 * Allows editing the configuration database.  Can either be invoked standalone or from within
 * another application.
 */
public class ConfigEditor extends BaseConfigEditor
    implements ClipboardOwner
{
    /**
     * The program entry point.
     */
    public static void main (String[] args)
    {
        ResourceManager rsrcmgr = new ResourceManager("rsrc/");
        MessageManager msgmgr = new MessageManager("rsrc.i18n");
        ConfigManager cfgmgr = new ConfigManager(rsrcmgr, msgmgr, "config/");
        ColorPository colorpos = ColorPository.loadColorPository(rsrcmgr);
        new ConfigEditor(msgmgr, cfgmgr, colorpos).setVisible(true);
    }

    /**
     * Creates a new config editor.
     */
    public ConfigEditor (MessageManager msgmgr, ConfigManager cfgmgr, ColorPository colorpos)
    {
        this(msgmgr, cfgmgr, colorpos, null, null);
    }

    /**
     * Creates a new config editor.
     */
    public ConfigEditor (
        MessageManager msgmgr, ConfigManager cfgmgr, ColorPository colorpos,
        Class<?> clazz, String name)
    {
        super(msgmgr, cfgmgr, colorpos, "config");

        // populate the menu bar
        JMenuBar menubar = new JMenuBar();
        setJMenuBar(menubar);

        JMenu file = createMenu("file", KeyEvent.VK_F);
        menubar.add(file);

        JMenu nmenu = createMenu("new", KeyEvent.VK_N);
        file.add(nmenu);
        nmenu.add(createMenuItem("window", KeyEvent.VK_W, KeyEvent.VK_N));
        nmenu.addSeparator();
        Action nconfig = createAction("config", KeyEvent.VK_C, KeyEvent.VK_O);
        nmenu.add(new JMenuItem(nconfig));
        Action nfolder = createAction("folder", KeyEvent.VK_F, KeyEvent.VK_D);
        nmenu.add(new JMenuItem(nfolder));
        file.addSeparator();
        file.add(_save = createMenuItem("save_group", KeyEvent.VK_S, KeyEvent.VK_S));
        file.add(_revert = createMenuItem("revert_group", KeyEvent.VK_R, KeyEvent.VK_R));
        file.addSeparator();
        file.add(createMenuItem("import_group", KeyEvent.VK_I, KeyEvent.VK_I));
        file.add(createMenuItem("export_group", KeyEvent.VK_E, KeyEvent.VK_E));
        file.addSeparator();
        file.add(createMenuItem("import_configs", KeyEvent.VK_M, -1));
        file.add(_exportConfigs = createMenuItem("export_configs", KeyEvent.VK_X, -1));
        file.addSeparator();
        file.add(createMenuItem("close", KeyEvent.VK_C, KeyEvent.VK_W));
        file.add(createMenuItem("quit", KeyEvent.VK_Q, KeyEvent.VK_Q));

        final JMenu edit = createMenu("edit", KeyEvent.VK_E);
        edit.addMenuListener(new MenuListener() {
            public void menuSelected (MenuEvent event) {
                // hackery to allow cut/copy/paste/delete to act on editor tree
                TreeEditorPanel panel = (TreeEditorPanel)SwingUtilities.getAncestorOfClass(
                    TreeEditorPanel.class, getFocusOwner());
                if (panel != null) {
                    edit.getItem(0).setAction(panel.getCutAction());
                    edit.getItem(1).setAction(panel.getCopyAction());
                    edit.getItem(2).setAction(panel.getPasteAction());
                    edit.getItem(3).setAction(panel.getDeleteAction());
                } else {
                    restoreActions();
                }
            }
            public void menuDeselected (MenuEvent event) {
                // restore after a delay so as not to interfere with selected item
                EventQueue.invokeLater(new Runnable() {
                    public void run () {
                        restoreActions();
                    }
                });
            }
            public void menuCanceled (MenuEvent event) {
                // no-op
            }
            protected void restoreActions () {
                edit.getItem(0).setAction(_cut);
                edit.getItem(1).setAction(_copy);
                edit.getItem(2).setAction(_paste);
                edit.getItem(3).setAction(_delete);
            }
        });
        menubar.add(edit);
        edit.add(new JMenuItem(_cut = createAction("cut", KeyEvent.VK_T, KeyEvent.VK_X)));
        edit.add(new JMenuItem(_copy = createAction("copy", KeyEvent.VK_C, KeyEvent.VK_C)));
        edit.add(new JMenuItem(_paste = createAction("paste", KeyEvent.VK_P, KeyEvent.VK_V)));
        edit.add(new JMenuItem(
            _delete = createAction("delete", KeyEvent.VK_D, KeyEvent.VK_DELETE, 0)));
        addFindMenu(edit);
        edit.addSeparator();
        edit.add(createMenuItem("validate_refs", KeyEvent.VK_V, -1));
        addEditMenuItems(edit);
        edit.addSeparator();
        edit.add(createMenuItem("resources", KeyEvent.VK_R, KeyEvent.VK_U));
        edit.add(createMenuItem("preferences", KeyEvent.VK_F, -1));

        JMenu view = createMenu("view", KeyEvent.VK_V);
        menubar.add(view);
        view.add(_treeMode = ToolUtil.createCheckBoxMenuItem(
            this, _msgs, "tree_mode", KeyEvent.VK_T, -1));

        JMenu gmenu = createMenu("groups", KeyEvent.VK_G);
        menubar.add(gmenu);
        gmenu.add(_saveAll = createMenuItem("save_all", KeyEvent.VK_S, KeyEvent.VK_A));
        gmenu.add(_revertAll = createMenuItem("revert_all", KeyEvent.VK_R, KeyEvent.VK_T));

        // create the pop-up menu
        _popup = new JPopupMenu();
        nmenu = createMenu("new", KeyEvent.VK_N);
        _popup.add(nmenu);
        nmenu.add(new JMenuItem(nconfig));
        nmenu.add(new JMenuItem(nfolder));
        _popup.addSeparator();
        _popup.add(new JMenuItem(_cut));
        _popup.add(new JMenuItem(_copy));
        _popup.add(new JMenuItem(_paste));
        _popup.add(new JMenuItem(_delete));
        _validate = createAction("validate_item", KeyEvent.VK_I, KeyEvent.VK_L);
        _popup.add(new JMenuItem(_validate));
        // create the file chooser
        _chooser = new JFileChooser(_prefs.get("config_dir", null));
        _chooser.setFileFilter(new FileFilter() {
            public boolean accept (File file) {
                return file.isDirectory() || (file.toString() != null && file.toString().toLowerCase().endsWith(".xml"));
            }
            public String getDescription () {
                return _msgs.get("m.xml_files");
            }
        });

        // create the split pane
        add(_split = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, true), BorderLayout.CENTER);

        // create the tabbed pane
        _split.setLeftComponent(_tabs = new JTabbedPane());
        _tabs.setPreferredSize(new Dimension(250, 1));
        _tabs.setMaximumSize(new Dimension(250, Integer.MAX_VALUE));

        // create the tabs for each configuration manager
        for (; cfgmgr != null; cfgmgr = cfgmgr.getParent()) {
            _tabs.add(new ManagerPanel(cfgmgr), getLabel(cfgmgr.getType()), 0);
        }

        // activate the first tab
        final ManagerPanel panel = (ManagerPanel)_tabs.getComponentAt(0);
        _tabs.setSelectedComponent(panel);
        panel.activate();

        // add a listener for tab change
        _tabs.addChangeListener(new ChangeListener() {
            public void stateChanged (ChangeEvent event) {
                (_panel = (ManagerPanel)_tabs.getSelectedComponent()).activate();
            }
            protected ManagerPanel _panel = panel;
        });

        // set sensible default bounds
        setSize(850, 600);
        SwingUtil.centerWindow(this);

        // restore out prefs (may override bounds)
        restorePrefs();
        
        
        // open the initial config, if one was specified
        if (clazz != null) {
            select(clazz, name);
        }
    }

    // documentation inherited from interface ClipboardOwner
    public void lostOwnership (Clipboard clipboard, Transferable contents)
    {
        _paste.setEnabled(false);
        _clipclass = null;
    }

    @Override // documentation inherited
    public void actionPerformed (ActionEvent event)
    {
        String action = event.getActionCommand();
        ManagerPanel panel = (ManagerPanel)_tabs.getSelectedComponent();
        ManagerPanel.GroupItem item = (ManagerPanel.GroupItem)panel.gbox.getSelectedItem();
        if (action.equals("window")) {
            showFrame(new ConfigEditor(_msgmgr, _cfgmgr, _colorpos));
        } else if (action.equals("config")) {
            item.newConfig();
        } else if (action.equals("folder")) {
            item.newFolder();
        } else if (action.equals("save_group")) {
            item.group.save();
        } else if (action.equals("revert_group")) {
            if (showCantUndo()) {
                item.group.revert();
            }
        } else if (action.equals("import_group")) {
            item.importGroup();
        } else if (action.equals("export_group")) {
            item.exportGroup();
        } else if (action.equals("import_configs")) {
            item.importConfigs();
        } else if (action.equals("export_configs")) {
            item.exportConfigs();
        } else if (action.equals("cut")) {
            item.cutNode();
        } else if (action.equals("copy")) {
            item.copyNode();
        } else if (action.equals("paste")) {
            item.pasteNode();
        } else if (action.equals("delete")) {
            item.deleteNode();
        } else if (action.equals("validate_item")) {
            item.validate();
        } else if (action.equals("validate_refs")) {
            validateReferences();
        } else if (action.equals("resources")) {
            showFrame(new ResourceEditor(_msgmgr, _cfgmgr, _colorpos));
        } else if (action.equals("tree_mode")) {
            boolean enabled = _treeMode.isSelected();
            for (int ii = _tabs.getComponentCount() - 1; ii >= 0; ii--) {
                ((ManagerPanel)_tabs.getComponentAt(ii)).setTreeModeEnabled(enabled);
            }
        } else if (action.equals("save_all")) {
            panel.cfgmgr.saveAll();
        } else if (action.equals("revert_all")) {
            panel.cfgmgr.revertAll();
        } else {
            super.actionPerformed(event);
        }
    }

    @Override // documentation inherited
    public void removeNotify ()
    {
        super.removeNotify();
        for (int ii = 0, nn = _tabs.getComponentCount(); ii < nn; ii++) {
            ((ManagerPanel)_tabs.getComponentAt(ii)).dispose();
        }
    }

    /**
     * Selects a configuration.
     */
    protected void select (Class<?> clazz, String name)
    {
        for (int ii = _tabs.getComponentCount() - 1; ii >= 0; ii--) {
            ManagerPanel panel = (ManagerPanel)_tabs.getComponentAt(ii);
            if (panel.select(clazz, name)) {
                return;
            }
        }
    }

    /**
     * Shows a confirm dialog.
     */
    protected boolean showCantUndo ()
    {
        return JOptionPane.showConfirmDialog(this, _msgs.get("m.cant_undo"),
                _msgs.get("t.cant_undo"), JOptionPane.OK_CANCEL_OPTION,
                JOptionPane.WARNING_MESSAGE) == 0;
    }

    /**
     * Validates the references.
     */
    protected void validateReferences ()
    {
        PrintStreamDialog dialog = new PrintStreamDialog(
            this, _msgs.get("m.validate_refs"), _msgs.get("m.ok"));
        _cfgmgr.validateReferences("", dialog.getPrintStream());
        dialog.maybeShow();
    }

    /**
     * Used to add addition items to the edit menu.
     */
    protected void addEditMenuItems (JMenu edit)
    {
    }

    @Override // documentation inherited
    protected BaseEditorPanel getFindEditorPanel ()
    {
        return ((ManagerPanel)_tabs.getSelectedComponent()).getEditorPanel();
    }

    /**
     * The panel for a single manager.
     */
    protected class ManagerPanel extends JPanel
        implements EditorContext, ItemListener, ChangeListener
    {
        /**
         * Contains the state of a single group.
         */
        public class GroupItem
            implements TreeSelectionListener
        {
            /** The actual group reference. */
            public ConfigGroup<ManagedConfig> group;

            public GroupItem (ConfigGroup group)
            {
                @SuppressWarnings("unchecked") ConfigGroup<ManagedConfig> mgroup =
                    group;
                this.group = mgroup;
                _label = getLabel(group.getConfigClass(), group.getName());
            }

            public void validate() {
            	ConfigTreeNode node = _tree.getSelectedNode();
                PrintStreamDialog dialog = new PrintStreamDialog(ConfigEditor.this, _msgs.get("m.validate_refs"), _msgs.get("m.ok"));
                node.getConfig().validateReferences(node.getConfig().getName(), dialog.getPrintStream());
                dialog.maybeShow();
			}

			/**
             * Activates this group.
             */
            public void activate ()
            {
                if (_tree == null) {
                    _tree = new ConfigTree(group, true) {
                        public void selectedConfigUpdated () {
                            _epanel.update();
                        }
                    };
                    _tree.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
                    _tree.addTreeSelectionListener(this);
                    _tree.setComponentPopupMenu(_popup);

                    // remove the mappings for cut/copy/paste since we handle those ourself
                    InputMap imap = _tree.getInputMap();
                    imap.put(KeyStroke.getKeyStroke(KeyEvent.VK_X, KeyEvent.CTRL_MASK), "noop");
                    imap.put(KeyStroke.getKeyStroke(KeyEvent.VK_C, KeyEvent.CTRL_MASK), "noop");
                    imap.put(KeyStroke.getKeyStroke(KeyEvent.VK_V, KeyEvent.CTRL_MASK), "noop");
                }
                _pane.setViewportView(_tree);
                _filterPanel.setTree(_tree);
                _paste.setEnabled(_clipclass == group.getConfigClass());
                updateSelection();
            }

            /**
             * Creates a new configuration and prepares it for editing.
             */
            public void newConfig ()
            {
                Class<?> clazz = group.getConfigClass();
                try {
                    newNode((ManagedConfig)clazz.newInstance());
                } catch (Exception e) {
                    log.warning("Failed to instantiate config [class=" + clazz + "].", e);
                }
            }

            /**
             * Creates a new folder and prepares it for editing.
             */
            public void newFolder ()
            {
                newNode(null);
            }

            /**
             * Brings up the import group dialog.
             */
            public void importGroup ()
            {
                if (_chooser.showOpenDialog(ConfigEditor.this) == JFileChooser.APPROVE_OPTION) {
                    group.load(_chooser.getSelectedFile());
                }
                _prefs.put("config_dir", _chooser.getCurrentDirectory().toString());
            }

            /**
             * Brings up the export group dialog.
             */
            public void exportGroup ()
            {
                if (_chooser.showSaveDialog(ConfigEditor.this) == JFileChooser.APPROVE_OPTION) {
                    group.save(_chooser.getSelectedFile());
                }
                _prefs.put("config_dir", _chooser.getCurrentDirectory().toString());
            }

            /**
             * Brings up the import config dialog.
             */
            public void importConfigs ()
            {
                if (_chooser.showOpenDialog(ConfigEditor.this) == JFileChooser.APPROVE_OPTION) {
                    group.load(_chooser.getSelectedFile(), true);
                }
                _prefs.put("config_dir", _chooser.getCurrentDirectory().toString());
            }

            /**
             * Brings up the export config dialog.
             */
            public void exportConfigs ()
            {
                if (_chooser.showOpenDialog(ConfigEditor.this) == JFileChooser.APPROVE_OPTION) {
                    ArrayList<ManagedConfig> configs = new ArrayList<ManagedConfig>();
                    _tree.getSelectedNode().getConfigs(configs);
                    group.save(configs, _chooser.getSelectedFile());
                }
                _prefs.put("config_dir", _chooser.getCurrentDirectory().toString());
            }

            /**
             * Cuts the currently selected node.
             */
            public void cutNode ()
            {
                copyNode();
                deleteNode();
            }

            /**
             * Copies the currently selected node.
             */
            public void copyNode ()
            {
                Clipboard clipboard = _tree.getToolkit().getSystemClipboard();
                clipboard.setContents(_tree.createClipboardTransferable(), ConfigEditor.this);
                _clipclass = group.getConfigClass();
                _paste.setEnabled(true);
            }

            /**
             * Pastes the node in the clipboard.
             */
            public void pasteNode ()
            {
                Clipboard clipboard = _tree.getToolkit().getSystemClipboard();
                _tree.getTransferHandler().importData(_tree, clipboard.getContents(this));
            }

            /**
             * Deletes the currently selected node.
             */
            public void deleteNode ()
            {
                ConfigTreeNode node = _tree.getSelectedNode();
                ConfigTreeNode parent = (ConfigTreeNode)node.getParent();
                int index = parent.getIndex(node);
                ((DefaultTreeModel)_tree.getModel()).removeNodeFromParent(node);
                int ccount = parent.getChildCount();
                node = (ccount > 0) ?
                    (ConfigTreeNode)parent.getChildAt(Math.min(index, ccount - 1)) : parent;
                if (node != _tree.getModel().getRoot()) {
                    _tree.setSelectionPath(new TreePath(node.getPath()));
                }
            }

            /**
             * Notes that the state of the currently selected configuration has changed.
             */
            public void configChanged ()
            {
                _tree.selectedConfigChanged();
            }

            /**
             * Attempts to select the specified config within this group.
             */
            public boolean select (String name)
            {
                if (group.getConfig(name) == null) {
                    return false;
                }
                _tabs.setSelectedComponent(ManagerPanel.this);
                gbox.setSelectedItem(this);
                _tree.setSelectedNode(name);
                return true;
            }

            /**
             * Disposes of the resources held by this item.
             */
            public void dispose ()
            {
                if (_tree != null) {
                    _tree.dispose();
                    _tree = null;
                }
            }

            // documentation inherited from interface TreeSelectionListener
            public void valueChanged (TreeSelectionEvent event)
            {
                updateSelection();
            }

            @Override // documentation inherited
            public String toString ()
            {
                return _label;
            }

            /**
             * Updates the state of the UI based on the selection.
             */
            protected void updateSelection ()
            {
                // find the selected node
                ConfigTreeNode node = _tree.getSelectedNode();

                // update the editor panel
                _epanel.setObject(node == null ? null : node.getConfig());

                // enable or disable the menu items
                boolean enable = (node != null);
                _exportConfigs.setEnabled(enable);
                _cut.setEnabled(enable);
                _copy.setEnabled(enable);
                _delete.setEnabled(enable);
            }

            /**
             * Creates a new node for the supplied configuration (or a folder node, if the
             * configuration is <code>null</code>).
             */
            protected void newNode (ManagedConfig config)
            {
            	
            	// presently we must clear the filter
                _filterPanel.clearFilter();
                
                // find the parent under which we want to add the node
                ConfigTreeNode snode = _tree.getSelectedNode();
                ConfigTreeNode parent = (ConfigTreeNode)(snode == null ?
                    _tree.getModel().getRoot() : snode.getParent());

                // create a node with a unique name and start editing it
                String name = parent.findNameForChild(
                    _msgs.get(config == null ? "m.new_folder" : "m.new_config"));
                ConfigTreeNode child = new ConfigTreeNode(name, config);
                ((DefaultTreeModel)_tree.getModel()).insertNodeInto(
                    child, parent, parent.getInsertionIndex(child));
                _tree.startEditingAtPath(new TreePath(child.getPath()));
            }

            /** The (possibly translated) group label. */
            protected String _label;

            /** The configuration tree. */
            protected ConfigTree _tree;
        }

        /** The configuration manager. */
        public ConfigManager cfgmgr;

        /** Determines the selected group. */
        public JComboBox gbox;

        public ManagerPanel (ConfigManager cfgmgr)
        {
            super(new VGroupLayout(GroupLayout.STRETCH, GroupLayout.STRETCH, 5, GroupLayout.TOP));
            this.cfgmgr = cfgmgr;

            setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));

            // create the group panel
            JPanel gpanel = GroupLayout.makeHStretchBox(5);
            add(gpanel, GroupLayout.FIXED);
            gpanel.add(new JLabel(_msgs.get("m.group")), GroupLayout.FIXED);

            // initialize the list of groups
            Collection<ConfigGroup> groups = cfgmgr.getGroups();
            GroupItem[] items = new GroupItem[groups.size()];
            int idx = 0;
            for (ConfigGroup group : groups) {
            	if(group != null){
            		items[idx++] = new GroupItem(group);
            	}
            }
            QuickSort.sort(items, new Comparator<GroupItem>() {
                public int compare (GroupItem g1, GroupItem g2) {
                    return g1.toString().compareTo(g2.toString());
                }
            });
            gpanel.add(gbox = new JComboBox(items));
            gbox.addItemListener(this);


            // add the filtering panel
            add(_filterPanel = new ConfigTreeFilterPanel(_msgmgr), VGroupLayout.FIXED);
            
            // add the pane that will contain the group tree
            add(_pane = new JScrollPane());

            // create the editor panel
            _epanel = new EditorPanel(this, EditorPanel.CategoryMode.TABS, null);
            _epanel.addChangeListener(this);
        }

        /**
         * Called when the panel is shown.
         */
        public void activate ()
        {
            // add the editor panel
            _split.setRightComponent(_epanel);
            SwingUtil.refresh(_epanel);

            // activate the selected item
            if(gbox.getSelectedItem() != null){
            	((GroupItem)gbox.getSelectedItem()).activate();
            }

            // can only save/revert configurations with a config path
            boolean enable = (cfgmgr.getConfigPath() != null);
            _save.setEnabled(enable);
            _revert.setEnabled(enable);
            _saveAll.setEnabled(enable);
            _revertAll.setEnabled(enable);
        }

        /**
         * Attempts to select the specified config.
         */
        public boolean select (Class<?> clazz, String name)
        {
            for (int ii = 0, nn = gbox.getItemCount(); ii < nn; ii++) {
                GroupItem item = (GroupItem)gbox.getItemAt(ii);
                if (item.group.getConfigClass() == clazz) {
                    return item.select(name);
                }
            }
            return false;
        }

        /**
         * Enables or disables tree view mode.
         */
        public void setTreeModeEnabled (boolean enabled)
        {
            BaseEditorPanel opanel = _epanel;
            _epanel = enabled ? new TreeEditorPanel(this) :
                new EditorPanel(this, EditorPanel.CategoryMode.TABS);
            _epanel.addChangeListener(this);
            _epanel.setObject(opanel.getObject());
            if (_split.getRightComponent() == opanel) {
                _split.setRightComponent(_epanel);
                SwingUtil.refresh(_epanel);
            }
        }

        /**
         * Disposes of the resources held by this manager.
         */
        public void dispose ()
        {
            for (int ii = 0, nn = gbox.getItemCount(); ii < nn; ii++) {
                ((GroupItem)gbox.getItemAt(ii)).dispose();
            }
        }

        /**
         * Returns the editor panel.
         */
        public BaseEditorPanel getEditorPanel ()
        {
            return _epanel;
        }

        // documentation inherited from interface EditorContext
        public ResourceManager getResourceManager ()
        {
            return _rsrcmgr;
        }

        // documentation inherited from interface EditorContext
        public MessageManager getMessageManager ()
        {
            return _msgmgr;
        }

        // documentation inherited from interface EditorContext
        public ConfigManager getConfigManager ()
        {
            return cfgmgr;
        }

        // documentation inherited from interface EditorContext
        public ColorPository getColorPository ()
        {
            return _colorpos;
        }

        // documentation inherited from interface ItemListener
        public void itemStateChanged (ItemEvent event)
        {
            ((GroupItem)gbox.getSelectedItem()).activate();
        }

        // documentation inherited from interface ChangeListener
        public void stateChanged (ChangeEvent event)
        {
            ((GroupItem)gbox.getSelectedItem()).configChanged();
        }

        /** Holds a configuration filtering panel. */
        protected ConfigTreeFilterPanel _filterPanel;
        
        /** The scroll pane that holds the group trees. */
        protected JScrollPane _pane;

        /** The object editor panel. */
        protected BaseEditorPanel _epanel;
    }

    /**
     * Restore and bind prefs.
     */
    protected void restorePrefs ()
    {
        final String p = "ConfigEditor."; // TODO? getClass().getSimpleName() + "." ???

        // restore/bind window bounds
        Rectangle r = getBounds();
        setBounds(_prefs.getInt(p + ".x", r.x),
                _prefs.getInt(p + ".y", r.y),
                _prefs.getInt(p + ".w", r.width),
                _prefs.getInt(p + ".h", r.height));
        addComponentListener(new ComponentAdapter() {
            @Override public void componentMoved (ComponentEvent event) {
                saveBounds();
            }
            @Override public void componentResized (ComponentEvent event) {
                saveBounds();
            }

            protected void saveBounds () {
                Rectangle r = getBounds();
                _prefs.putInt(p + ".x", r.x);
                _prefs.putInt(p + ".y", r.y);
                _prefs.putInt(p + ".w", r.width);
                _prefs.putInt(p + ".h", r.height);
            }
        });

        // restore/bind the location of the divider
        _split.setDividerLocation(_prefs.getInt(p + ".div", _split.getDividerLocation()));
        _split.addPropertyChangeListener(new PropertyChangeListener() {
            public void propertyChange (PropertyChangeEvent event) {
                if (JSplitPane.DIVIDER_LOCATION_PROPERTY.equals(event.getPropertyName())) {
                    _prefs.putInt(p + ".div", _split.getDividerLocation());
                }
            }
        });

        // restore/bind the selected group
        String cat = _prefs.get(p + ".group", null);
        for (int tab = _tabs.getComponentCount() - 1; tab >= 0; tab--) {
            final JComboBox gbox = ((ManagerPanel)_tabs.getComponentAt(tab)).gbox;
            if (cat != null) {
                for (int ii = 0, nn = gbox.getItemCount(); ii < nn; ii++) {
                    if (cat.equals(String.valueOf(gbox.getItemAt(ii)))) {
                        gbox.setSelectedIndex(ii);
                        break;
                    }
                }
            }
            gbox.addActionListener(new ActionListener() {
                public void actionPerformed (ActionEvent event) {
                    _prefs.put(p + ".group", String.valueOf(gbox.getSelectedItem()));
                }
            });
        }
    }
    
    /** The config tree pop-up menu. */
    protected JPopupMenu _popup;

    /** The save and revert menu items. */
    protected JMenuItem _save, _revert, _saveAll, _revertAll;

    /** The configuration export menu item. */
    protected JMenuItem _exportConfigs;

    /** The edit menu actions. */
    protected Action _cut, _copy, _paste, _delete,_validate;

    /** The tree mode toggle. */
    protected JCheckBoxMenuItem _treeMode;

    /** The file chooser for opening and saving config files. */
    protected JFileChooser _chooser;

    /** The split pane containing the tabs and the editor panel. */
    protected JSplitPane _split;

    /** The tabs for each manager. */
    protected JTabbedPane _tabs;

    /** The class of the clipboard selection. */
    protected Class<?> _clipclass;
}
