package com.threerings.coin.server.persist;

import com.google.common.collect.Lists;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import com.hexnova.platform.payment.service.api.PlatformAccountService;
import com.hexnova.platform.payment.service.api.PlatformTransferService;
import com.hexnova.platform.payment.service.entity.AccountInfo;
import com.samskivert.depot.DateFuncs;
import com.samskivert.depot.DepotRepository;
import com.samskivert.depot.DuplicateKeyException;
import com.samskivert.depot.Funcs;
import com.samskivert.depot.MathFuncs;
import com.samskivert.depot.Ops;
import com.samskivert.depot.PersistenceContext;
import com.samskivert.depot.PersistentRecord;
import com.samskivert.depot.clause.FieldOverride;
import com.samskivert.depot.clause.FromOverride;
import com.samskivert.depot.clause.GroupBy;
import com.samskivert.depot.clause.Limit;
import com.samskivert.depot.clause.OrderBy;
import com.samskivert.depot.clause.QueryClause;
import com.samskivert.depot.clause.Where;
import com.samskivert.depot.expression.SQLExpression;
import com.samskivert.jdbc.ConnectionProvider;
import com.samskivert.util.ArrayIntSet;
import com.samskivert.util.AuditLogger;
import com.samskivert.util.Calendars;
import com.samskivert.util.Tuple;
import com.threerings.coin.Log;
import com.threerings.coin.server.CoinManager;
import com.threerings.user.depot.AccountActionRepository;

import java.sql.Date;
import java.sql.Timestamp;
import java.util.List;
import java.util.Set;

@Singleton
public class DepotCoinRepository extends DepotRepository implements CoinRepository {
	
	protected static final String RETURN_RESERVATION = "@@returnRez";
	protected static final String SPEND_RESERVATION = "@@spendRez";
	protected String _serverId;
	protected AuditLogger _auditLog = AuditLogger.getAuditLogger("coin");
	protected AccountActionRepository _actionRepo;

	@Inject
	private PlatformAccountService _platformAccountService;
	@Inject
	private PlatformTransferService _platformTransferService;

	public DepotCoinRepository(PersistenceContext ctx, String serverId, AccountActionRepository accountActionRepo) {
		super(ctx);
		this._serverId = (serverId == null ? "" : serverId);
		this._actionRepo = accountActionRepo;
	}
	
	@Inject
	public DepotCoinRepository(ConnectionProvider conprov, String serverId,  AccountActionRepository accountActionRepo) {
		this(new PersistenceContext("coindb", conprov, null), serverId, accountActionRepo);
	}

	/* (non-Javadoc)
	 * @see com.threerings.coin.server.persist.CoinRepository#unreserveAllCoinsForThisServer()
	 */
	@Override
	public void unreserveAllCoinsForThisServer() {
		ArrayIntSet ids = new ArrayIntSet();
		Where where = new Where(ReservedCoinsRecord.SERVER_ID.eq(this._serverId));
		for (ReservedCoinsRecord rrec : findAll(ReservedCoinsRecord.class, new QueryClause[] { where })) {
			ids.add(rrec.reservationId);
		}

		int count = ids.size();
		if (count > 0) {
			Log.log.info("!!! Clearing " + count + " stale coin reservation records.", new Object[0]);
			for (int ii = 0; ii < count; ii++)
				returnReservation(ids.get(ii));
		}
	}

	/* (non-Javadoc)
	 * @see com.threerings.coin.server.persist.CoinRepository#getCoinCount(java.lang.String)
	 */
	@Override
	public int getCoinCount(String accountName) {
		CoinsRecord rec = (CoinsRecord) load(CoinsRecord.getKey(accountName), new QueryClause[0]);
		return rec == null ? 0 : rec.coins;
	}
	
	/* (non-Javadoc)
	 * @see com.threerings.coin.server.persist.CoinRepository#getCoinCount(long)
	 */
	@Override
	public int getCoinCount(long coinAccountId) {
		AccountInfo accountInfo = this._platformAccountService.account(coinAccountId);
		return accountInfo == null ? 0 : accountInfo.getCoin().intValue();
	}

	/* (non-Javadoc)
	 * @see com.threerings.coin.server.persist.CoinRepository#addCoins(java.lang.String, int, int, java.lang.String)
	 */
	@Override
	public void addCoins(String accountName, int coins, int type, String descrip) {
		if (descrip == null) {
			throw new NullPointerException("Description must not be null.");
		}
		if (coins < 1) {
			throw new IllegalArgumentException("May not add less than 1 coin.");
		}

		addCoins(accountName, coins);

		noteTransaction(accountName, coins, type, descrip);

		notifyCoinChange(accountName);
	}

	public int reserveCoins(String accountName, int quantity) {
		if (quantity < 1) {
			throw new IllegalArgumentException("May not reserve less than 1 coin.");
		}

		Where guard = new Where(Ops.and(new SQLExpression[] { CoinsRecord.ACCOUNT_NAME.eq(accountName), CoinsRecord.COINS.greaterEq(Integer.valueOf(quantity)) }));

		if (updatePartial(CoinsRecord.class, guard, CoinsRecord.getKey(accountName), CoinsRecord.COINS, CoinsRecord.COINS.minus(Integer.valueOf(quantity)), new Object[0]) == 0) {
			return -1;
		}

		ReservedCoinsRecord rec = new ReservedCoinsRecord();
		rec.accountName = accountName;
		rec.coins = quantity;
		rec.serverId = this._serverId;
		insert(rec);

		notifyCoinChange(accountName);

		return rec.reservationId;
	}
	
	/* (non-Javadoc)
	 * @see com.threerings.coin.server.persist.CoinRepository#transferCoins(int, java.lang.String, int, java.lang.String, java.lang.String)
	 */
	@Override
	public boolean transferCoins(int reservationId, String targetAccountName, int type, String srcDescrip, String destDescrip) {
		if (srcDescrip == null) {
			throw new NullPointerException("Source description must not be null.");
		}
		if (destDescrip == null) {
			throw new NullPointerException("Destination description must not be null.");
		}
		if (targetAccountName == null) {
			throw new IllegalArgumentException("Target user ID must be greater than zero.");
		}

		return coinXfer(reservationId, targetAccountName, type, srcDescrip, destDescrip);
	}

	public boolean spendCoins(int reservationId, int type, String descrip) {
		if (descrip == null) {
			throw new NullPointerException("Description must not be null");
		}
		return coinXfer(reservationId, SPEND_RESERVATION, type, descrip, null);
	}

	public boolean returnReservation(int reservationId) {
		return coinXfer(reservationId, RETURN_RESERVATION, -1, null, null);
	}

	protected boolean coinXfer(int reservationId, String targetAccountName, int type, String srcDescrip, String destDescrip) {
		ReservedCoinsRecord rec = (ReservedCoinsRecord) load(ReservedCoinsRecord.class,
				new QueryClause[] { new Where(Ops.and(new SQLExpression[] { 
						ReservedCoinsRecord.RESERVATION_ID.eq(Integer .valueOf(reservationId)), ReservedCoinsRecord.SERVER_ID.eq(this._serverId) })) });

		if (rec == null) {
			return false;
		}

		delete(rec);
		String destUser;
		if (targetAccountName == SPEND_RESERVATION) {
			noteTransaction(rec.accountName, -rec.coins, type, srcDescrip);
			destUser = SPEND_RESERVATION;
		} else {
			if (targetAccountName == RETURN_RESERVATION) {
				addCoins(rec.accountName, rec.coins);
				destUser = rec.accountName;
			} else {
				addCoins(targetAccountName, rec.coins);
				noteTransaction(rec.accountName, -rec.coins, type, srcDescrip);
				noteTransaction(targetAccountName, rec.coins, type, destDescrip);
				destUser = targetAccountName;
			}
		}

		if (destUser != SPEND_RESERVATION) {
			notifyCoinChange(destUser);
		}

		return true;
	}

	public void notifyCoinChange(String accountName) {
		if (CoinManager.SERVER_ACCOUNT_NAME.equals(accountName)) {
			return;
		}

		if (this._actionRepo != null)
			this._actionRepo.addAction(accountName, 1, this._serverId);
	}

	public void summarizeHistory(Date start, Date end) {
		Date next = Calendars.at(end).addDays(1).toSQLDate();

		for (SummarizeRecord rec : findAll(SummarizeRecord.class, new QueryClause[] {
						new FieldOverride(SummarizeRecord.TIME, DateFuncs.date(CoinHistoryRecord.TIME)),
						new FieldOverride(SummarizeRecord.COINS, Funcs.sum(CoinHistoryRecord.COINS)),
						new FieldOverride(SummarizeRecord.ACCOUNTS, Funcs.countDistinct(CoinHistoryRecord.ACCOUNT_NAME)),
						new Where(Ops.and(new SQLExpression[] { CoinHistoryRecord.TIME.greaterEq(start), CoinHistoryRecord.TIME.lessThan(next) })),
						new GroupBy(new SQLExpression[] { DateFuncs.date(SummarizeRecord.TIME), SummarizeRecord.TYPE }) })) {
			
			CoinHistorySumRecord hrec = new CoinHistorySumRecord();
			hrec.sumdate = rec.time;
			hrec.type = rec.type;
			hrec.txCount = rec.count;
			hrec.coins = rec.coins;
			hrec.accounts = rec.accounts;
			store(hrec);
		}
	}

	public List<DailySummary> loadHistory(Date start, Date end) {
		List<CoinHistorySumRecord> records = findAll(CoinHistorySumRecord.class, new QueryClause[] {
						new Where(Ops.and(new SQLExpression[] { CoinHistorySumRecord.SUMDATE.greaterEq(start),
								CoinHistorySumRecord.SUMDATE.lessEq(end) })), OrderBy.ascending(CoinHistorySumRecord.SUMDATE) });

		List<DailySummary> data = Lists.newArrayList();
		DailySummary day = new DailySummary();
		day.date = start;
		for (CoinHistorySumRecord rec : records) {
			if (!rec.sumdate.equals(day.date)) {
				if (day.size() > 0) {
					data.add(day);
				}
				day = new DailySummary();
				day.date = rec.sumdate;
			}
			day.put(rec.type,
					new int[] { rec.txCount, rec.coins, rec.accounts });
		}
		if (day.size() > 0) {
			data.add(day);
		}
		return data;
	}

	public List<CoinTransaction> getTransactionHistory(String accountName) {
		List<CoinTransaction> list = Lists.newArrayList();
		for (CoinHistoryRecord rec : findAll(CoinHistoryRecord.class,
				new QueryClause[] { new Where(CoinHistoryRecord.ACCOUNT_NAME.eq(accountName)) })) {
			list.add(new CoinTransaction(rec.coins, rec.type, rec.description, rec.time));
		}
		return list;
	}

	/* (non-Javadoc)
	 * @see com.threerings.coin.server.persist.CoinRepository#pruneTransactions()
	 */
	@Override
	public void pruneTransactions() {
		Timestamp ucutoff = Calendars.now().addDays(-90).toTimestamp();
		int pruned = deleteAll(CoinHistoryRecord.class, new Where(CoinHistoryRecord.TIME.lessThan(ucutoff)), (Limit) null);

		if (pruned > 0) {
			Log.log.info("Pruned " + pruned + " coin transactions.", new Object[0]);
		}

		Timestamp scutoff = Calendars.now().addDays(-30).toTimestamp();
		pruned = deleteAll(CoinHistoryRecord.class,
				new Where(Ops.and(new SQLExpression[] { CoinHistoryRecord.ACCOUNT_NAME.eq(CoinManager.SERVER_ACCOUNT_NAME), CoinHistoryRecord.TIME.lessThan(scutoff) })), 
				(Limit) null);

		if (pruned > 0)
			Log.log.info("Pruned " + pruned + " additional server transactions.", new Object[0]);
	}

	protected void addCoins(String accountName, int coins) {
		int mods = updatePartial(CoinsRecord.getKey(accountName),
				CoinsRecord.COINS,
				CoinsRecord.COINS.plus(Integer.valueOf(coins)), new Object[0]);

		if (mods == 0)
			try {
				CoinsRecord rec = new CoinsRecord();
				rec.accountName = accountName;
				rec.coins = coins;
				insert(rec);
			} catch (DuplicateKeyException dke) {
				if (updatePartial(CoinsRecord.getKey(accountName),
						CoinsRecord.COINS,
						CoinsRecord.COINS.plus(Integer.valueOf(coins)),
						new Object[0]) == 0) {
					Log.log.warning("WTFingF? We failed to update, then failed to insert and now we've failed to update again. We clearly picked a bad week to stop sniffing glue!",
							new Object[] { "who", accountName, "coins", Integer.valueOf(coins) });
				}
			}
	}

	protected void noteTransaction(String accountName, int dcoins, int type, String descrip) {
		if (this._auditLog != null) {
			this._auditLog.log("coins " + dcoins + " " + accountName + " " + descrip, new Object[0]);
		}

		CoinHistoryRecord rec = new CoinHistoryRecord();
		rec.accountName = accountName;
		rec.coins = dcoins;
		rec.type = type;
		rec.description = descrip;
		rec.time = new Timestamp(System.currentTimeMillis());
		insert(rec);
	}

	public int getCoinsPurchasedSince(String accountName, long deltaMs) {
		Date date = new Date(System.currentTimeMillis() - deltaMs);
		Where where = new Where(Ops.and(new SQLExpression[] {
				CoinHistoryRecord.ACCOUNT_NAME.eq(accountName),
				CoinHistoryRecord.TYPE.eq(Integer.valueOf(1)),
				CoinHistoryRecord.TIME.greaterEq(date) }));

		int coins = 0;
		for (CoinHistoryRecord rec : findAll(CoinHistoryRecord.class, new QueryClause[] { where })) {
			coins += rec.coins;
		}
		return coins;
	}

	public List<Tuple<String, Integer>> getTopCoinUsers(Date start, Date end, int txType, int limit) {
		List exprs = Lists.newArrayList();
		exprs.add(CoinHistoryRecord.TIME.greaterEq(start));
		exprs.add(CoinHistoryRecord.TIME.lessEq(end));
		if (txType > 0) {
			exprs.add(CoinHistoryRecord.TYPE.eq(Integer.valueOf(txType)));
		}

		List<QueryClause> clauses = Lists.newArrayList();
		clauses.add(new FromOverride(CoinHistoryRecord.class));
		clauses.add(new FieldOverride(AccountDatumRecord.ACCOUNT_NAME, CoinHistoryRecord.ACCOUNT_NAME));

		clauses.add(new FieldOverride(AccountDatumRecord.DATUM, Funcs.sum(MathFuncs.abs(CoinHistoryRecord.COINS))));

		clauses.add(new Where(Ops.and(exprs)));
		clauses.add(new GroupBy(new SQLExpression[] { CoinHistoryRecord.ACCOUNT_NAME }));
		clauses.add(OrderBy.descending(AccountDatumRecord.DATUM));
		if (limit > 0) {
			clauses.add(new Limit(0, limit));
		}

		List list = Lists.newArrayList();
		for (AccountDatumRecord rec : findAll(AccountDatumRecord.class, clauses)) {
			list.add(Tuple.newTuple(rec.accountName, Integer.valueOf(rec.datum)));
		}
		return list;
	}

	public int getOutstandingCoins() {
		return ((OutstandingCoinsRecord) load(OutstandingCoinsRecord.class,
				new QueryClause[] { new FieldOverride(OutstandingCoinsRecord.COINS, Funcs.sum(CoinsRecord.COINS)) })).coins;
	}

	protected void getManagedRecords(Set<Class<? extends PersistentRecord>> classes) {
		classes.add(CoinsRecord.class);
		classes.add(CoinHistoryRecord.class);
		classes.add(CoinHistorySumRecord.class);
		classes.add(ReservedCoinsRecord.class);
	}
	
	public PlatformTransferService getPlatformTransferService() {
		return this._platformTransferService;
	}

}