//
// $Id$

package com.threerings.user;

import java.util.Map;
import java.util.Properties;
import javax.servlet.http.HttpServletRequest;

import com.google.common.collect.Maps;

import com.samskivert.io.PersistenceException;
import com.samskivert.util.RunQueue;
import com.samskivert.jdbc.ConnectionProvider;

import com.samskivert.servlet.RedirectException;

import com.samskivert.servlet.user.User;
import com.samskivert.servlet.user.UserManager;
import com.samskivert.servlet.user.UserRepository;

import static com.threerings.user.Log.log;

/**
 * Extends the standard user manager with OOO-specific support.
 */
public class OOOUserManager extends UserManager
{
    /**
     * Creates our OOO User manager and prepares it for operation.
     */
    public OOOUserManager (Properties config, ConnectionProvider conprov)
        throws PersistenceException
    {
        this(config, conprov, null);
    }

    /**
     * Creates our OOO user manager and prepares it for operation.
     */
    public OOOUserManager (Properties config, ConnectionProvider conprov, RunQueue pruneQueue)
        throws PersistenceException
    {
        // legacy business
        init(config, conprov, pruneQueue);
    }

    /**
     * Creates an OOO user manager which must subsequently be initialized with
     * a call to {@link #init}.
     */
    public OOOUserManager ()
    {
    }

    @Override
    public OOOUserRepository getRepository ()
    {
        return (OOOUserRepository)_repository;
    }

    @Override
    public void init (Properties config, ConnectionProvider conprov, RunQueue pruneQueue)
        throws PersistenceException
    {
        super.init(config, conprov, pruneQueue);

        // create the blast repository
        _blastRepo = new GameBlastAuxRepository(conprov);

        // look up the access denied URL
        _accessDeniedURL = config.getProperty("access_denied_url");
        if (_accessDeniedURL == null) {
            log.warning("No 'access_denied_url' supplied in user manager config. " +
                "Restricted pages will behave strangely.");
        }

    }

    /**
     * Get the gameblast aux data repository.
     */
    public GameBlastAuxRepository getBlastRepository ()
    {
        return _blastRepo;
    }

    
    /**
     * Extends the standard {@link #requireUser(HttpServletRequest)} with
     * the additional requirement that the user hold the specified token.
     * If they do not, they will be redirected to a page informing them
     * access is denied.
     *
     * @return the user associated with the request.
     */
    public User requireUser (HttpServletRequest req, byte token)
        throws PersistenceException, RedirectException
    {
        OOOUser user = (OOOUser)requireUser(req);
        if (!user.holdsToken(token)) {
            throw new RedirectException(_accessDeniedURL);
        }
        return user;
    }

    /**
     * Extends the standard {@link #requireUser(HttpServletRequest)} with
     * the additional requirement that the user hold one of the specified
     * tokens. If they do not, they will be redirected to a page informing
     * them access is denied.
     *
     * @return the user associated with the request.
     */
    public User requireUser (HttpServletRequest req, byte[] tokens)
        throws PersistenceException, RedirectException
    {
        OOOUser user = (OOOUser)requireUser(req);
        if (!user.holdsAnyToken(tokens)) {
            throw new RedirectException(_accessDeniedURL);
        }
        return user;
    }

    @Override
    protected UserRepository createRepository (ConnectionProvider conprov)
        throws PersistenceException
    {
        return new OOOUserRepository(conprov);
    }

    /** The repository for gameblast auxiliary data. */
    protected GameBlastAuxRepository _blastRepo;

    /** The URL to which we redirect users whose access is denied. */
    protected String _accessDeniedURL;
}
