package com.threerings.cron.server;

import java.util.Calendar;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Lists;
import com.google.inject.Inject;
import com.samskivert.util.Calendars;
import com.samskivert.util.Interval;
import com.samskivert.util.Lifecycle;
import com.samskivert.util.RandomUtil;
import com.threerings.cron.Log;
import com.threerings.cron.server.persist.CronRepository;

public class CronLogic
{
  protected JobTicker _ticker;
  protected ListMultimap<Integer, Job> _jobs = ArrayListMultimap.create();

  protected ConcurrentMap<String, Boolean> _running = new ConcurrentHashMap();

  @Inject
  protected CronRepository _cronRepo;
  protected static final int HOUR = 60;

  @Inject
  public CronLogic(Lifecycle cycle)
  {
    this._ticker = new JobTicker();
    cycle.addComponent(new Lifecycle.Component()
    {
      public void init()
      {
        Calendar cal = Calendar.getInstance();
        long curmils = cal.get(13) * 1000L + cal.get(14);
        CronLogic.this._ticker.schedule(60000L - curmils);
      }
      public void shutdown() {
        CronLogic.this._ticker.cancel();
      }
    });
  }

  public void scheduleEvery(int hourlyPeriod, String ident, Runnable job)
  {
    int minOfHour = Math.abs(ident.hashCode()) % 60;
    int minOfDay = 0;
    synchronized (this._jobs) {
      while (minOfDay < 1440) {
        this._jobs.put(Integer.valueOf(minOfDay + minOfHour), new Job(ident, job));
        minOfDay += hourlyPeriod * 60;
      }
    }
  }

  public void scheduleAt(int hour, String ident, Runnable job)
  {
    int minOfHour = Math.abs(ident.toString().hashCode()) % 60;
    synchronized (this._jobs) {
      this._jobs.put(Integer.valueOf(hour * 60 + minOfHour), new Job(ident, job));
    }
  }

  public void unschedule(String ident)
  {
    synchronized (this._jobs)
    {
      Iterator iter = this._jobs.entries().iterator();
      while (iter.hasNext()) {
        Map.Entry entry = (Map.Entry)iter.next();
        if (((Job)entry.getValue()).ident.equals(ident))
          iter.remove();
      }
    }
  }

  protected void executeJobs(int minuteOfDay)
  {
    int dayOfYear = Calendars.now().get(6);
    List<Job> jobs = Lists.newArrayList();
    synchronized (this._jobs) {
      List sched = this._jobs.get(Integer.valueOf(minuteOfDay));
      if (sched != null) {
        jobs.addAll(sched);
      }
    }
    for (Job job : jobs)
      executeJob(job, dayOfYear, minuteOfDay);
  }

  protected void executeJob(final Job job, int dayOfYear, int minuteOfDay)
  {
    if (!this._cronRepo.claimJob(job.ident, dayOfYear, minuteOfDay)) {
      return;
    }

    if (this._running.putIfAbsent(job.ident, Boolean.valueOf(true)) != null) {
      Log.log.info("Dropping job as it is still executing", new Object[] { "job", job });
      return;
    }

    new Thread() {
      public void run() {
        try {
          job.job.run();
        } catch (Throwable t) {
          Log.log.warning("Job failed", new Object[] { "job", job, t });
        } finally {
          CronLogic.this._running.remove(job.ident);
        }
      }
    }
    .start();
  }

  protected class JobTicker extends Interval
  {
    protected Calendar _cal = Calendar.getInstance();
    protected int _prevMinute = getMinuteOfDay();

    protected JobTicker()
    {
    }

    public void expired()
    {
      int curMinute = getMinuteOfDay();
      if (curMinute < this._prevMinute) {
        processMinutes(this._prevMinute + 1, 1439);
        processMinutes(0, curMinute);
      } else {
        processMinutes(this._prevMinute + 1, curMinute);
      }

      this._prevMinute = curMinute;

      schedule(61000L - RandomUtil.getInt(2000));
    }

    protected int getMinuteOfDay() {
      this._cal.setTimeInMillis(System.currentTimeMillis());
      return this._cal.get(11) * 60 + this._cal.get(12);
    }

    protected void processMinutes(int fromMinute, int toMinute) {
      for (int mm = fromMinute; mm <= toMinute; mm++)
        CronLogic.this.executeJobs(mm);
    }
  }

  protected static class Job
  {
    public final String ident;
    public final Runnable job;

    public Job(String ident, Runnable job)
    {
      this.ident = ident;
      this.job = job;
    }

    public String toString() {
      return this.ident + " " + this.job;
    }
  }
}