package com.threerings.coin.server;

import com.google.inject.Inject;
import com.hexnova.platform.payment.service.entity.TransferTask;
import com.meidusa.venus.notify.InvocationListener;
import com.samskivert.depot.PersistenceContext;
import com.samskivert.jdbc.ConnectionProvider;
import com.samskivert.jdbc.RepositoryListenerUnit;
import com.samskivert.jdbc.ResultUnit;
import com.samskivert.jdbc.WriteOnlyUnit;
import com.samskivert.util.Calendars;
import com.samskivert.util.IntResultListener;
import com.samskivert.util.Interval;
import com.samskivert.util.Invoker;
import com.samskivert.util.ResultListener;
import com.threerings.coin.Log;
import com.threerings.coin.server.persist.CoinRepository;
import com.threerings.coin.server.persist.DailySummary;
import com.threerings.presents.server.InvocationException;
import com.threerings.user.depot.AccountActionRepository;

import java.sql.Date;
import java.util.List;

public class CoinManager {
	
	public static final String SERVER_ACCOUNT_NAME = "@@SERVER@@";
	
	@Inject
	protected CoinRepository _coinRepo;
	protected Invoker _invoker;
	protected static final long COIN_HISTORY_FREQ = 21600000L;
	
	public CoinManager(ConnectionProvider conprov, String serverId, CoinRepository coinRep,AccountActionRepository accountActionRepo, Invoker invoker) {
		this(conprov, serverId, coinRep,accountActionRepo, invoker, true);
	}

	public CoinManager(ConnectionProvider conprov, String serverId, CoinRepository coinRep,AccountActionRepository accountActionRepo, Invoker invoker, boolean runCoinSummary) {
		this(new PersistenceContext("coindb", conprov, null), serverId, coinRep,accountActionRepo, invoker, runCoinSummary);
	}

	public CoinManager(PersistenceContext ctx, String serverId, CoinRepository coinRep,AccountActionRepository accountActionRepo, Invoker invoker, boolean runCoinSummary) {
		this._coinRepo = coinRep;//new CoinRepository(ctx, serverId, accountActionRepo);
		this._invoker = invoker;

		this._coinRepo.unreserveAllCoinsForThisServer();

		if (!runCoinSummary) {
			return;
		}

		new Interval() {
			public void expired() {
				CoinManager.this._invoker.postUnit(new WriteOnlyUnit("CoinManager:summarizeCoinHistory") {
					public void invokePersist() throws Exception {
						CoinManager.this.summarizeCoinHistory();
					}
				});
			}
		}.schedule(21600000L, true);
	}

	public CoinRepository getCoinRepository() {
		return this._coinRepo;
	}

//	public void getCoinSnapshot(final String accountName, final IntResultListener rl) {
//		this._invoker.postUnit(new ResultUnit<Integer>("CoinManager:getCoinSnapshot") {
//			public Integer computeResult() throws Exception {
//				return Integer.valueOf(CoinManager.this._coinRepo.getCoinCount(accountName));
//			}
//
//			public void handleResult(Integer result) {
//				rl.requestCompleted(result.intValue());
//			}
//
//			public void handleFailure(Exception e) {
//				rl.requestFailed(e);
//			}
//		});
//	}
	
	public void transferCoins(List<TransferTask> transferTaskList, final ResultListener<Void> rl) {
		int coins = 0;
		for (TransferTask transferTask : transferTaskList) {
			coins += transferTask.getCoin();
		}
		if (coins != 0) {
			rl.requestFailed(new IllegalArgumentException("" + coins));
			return;
		}
		
		/*InvocationListener<Void> listener = new InvocationListener<Void>() {
			public void callback(Void object) {
				rl.requestCompleted(null);
			}

			public void onException(Exception e) {
				rl.requestFailed(e);
			}
		};*/
		
		boolean isSuccess = false;
		try {
			isSuccess = this._coinRepo.getPlatformTransferService().doTransfer(transferTaskList);
		} catch (Exception e) {
			rl.requestFailed(new InvocationException("m.transaction_error"));
			return;
		}
		
		if (isSuccess) {
			rl.requestCompleted(null);
		} else {
			rl.requestFailed(new InvocationException("m.transaction_error"));
		}
	}

	public void reserveCoins(final String accountName, final int coins, final IntResultListener rl) {
		if (coins < 1) {
			rl.requestFailed(new IllegalArgumentException("" + coins));
			return;
		}

		this._invoker.postUnit(new ResultUnit<Integer>("CoinManager:reserveCoins") {
			public Integer computeResult() throws Exception {
				return Integer.valueOf(CoinManager.this._coinRepo.reserveCoins(accountName, coins));
			}

			public void handleResult(Integer result) {
				if (result.intValue() == -1)
					rl.requestFailed(new CoinManager.InsufficientFundsException());
				else
					rl.requestCompleted(result.intValue());
			}

			public void handleFailure(Exception e) {
				rl.requestFailed(e);
			}
		});
	}

	public void transferReservation(final int reservationId, final String destAccountName, final int type, final String srcDescrip, final String destDescrip,
			ResultListener<Void> rl) {
		this._invoker.postUnit(new RepositoryListenerUnit("CoinManager:transferReservation", rl) {
			public Void invokePersistResult() throws Exception {
				if (!CoinManager.this._coinRepo.transferCoins(reservationId, destAccountName, type, srcDescrip, destDescrip)) {
					throw new Exception("No such reservation.");
				}
				return null;
			}
		});
	}

	public void spendReservation(final int reservationId, final int type, final String descrip, ResultListener<Void> rl) {
		this._invoker.postUnit(new RepositoryListenerUnit("CoinManager:spendReservation", rl) {
			public Void invokePersistResult() throws Exception {
				if (!CoinManager.this._coinRepo.spendCoins(reservationId, type, descrip)) {
					throw new Exception("No such reservation.");
				}
				return null;
			}
		});
	}

	public void returnReservation(final int reservationId, ResultListener<Void> rl) {
		this._invoker.postUnit(new RepositoryListenerUnit("CoinManager:returnReservation", rl) {
			public Void invokePersistResult() throws Exception {
				if (!CoinManager.this._coinRepo.returnReservation(reservationId)) {
					throw new Exception("No such reservation.");
				}
				return null;
			}
		});
	}

/*	public void purchase(final String accountName, final int coins, final int type, final String descrip, final CoinOp op, final ResultListener<Object> rl) {
		reserveCoins(accountName, coins, new IntResultListener() {
			public void requestFailed(Exception cause) {
				rl.requestFailed(cause);
			}

			public void requestCompleted(final int reservationId) {
				try {
					op.processPurchase(new ResultListener() {
						public void requestCompleted(Object result) {
							CoinManager.this.finalizePurchase(reservationId, result, type, descrip, rl);
						}

						public void requestFailed(Exception cause) {
							CoinManager.this.rollbackPurchase(reservationId, cause, rl);
						}
					});
				} catch (Exception booch) {
					Log.log.warning("Error processing purchase", new Object[] { "accountName", accountName, "coins", Integer.valueOf(coins), booch });
					CoinManager.this.rollbackPurchase(reservationId, booch, rl);
				}
			}
		});
	}*/

	protected void rollbackPurchase(final int reservationId, final Exception cause, final ResultListener<Object> rl) {
		this._invoker.postUnit(new Invoker.Unit("CoinManager:rollbackPurchase") {
			public boolean invoke() {
				try {
					if (!CoinManager.this._coinRepo.returnReservation(reservationId)) {
						Log.log.warning("Holy crap! Unable to unreserve coins, what the hell happened to them? [reservationId=" + reservationId + "].", new Object[0]);
					}
				} catch (Exception pe) {
					Log.log.warning("Oh crap! Exception while attempting to rollback a coin purchase [reservationId=" + reservationId + ", error=" + pe + "].", new Object[0]);
				}

				return true;
			}

			public void handleResult() {
				rl.requestFailed(cause);
			}
		});
	}

	protected void finalizePurchase(final int reservationId, final Object result, final int type, final String descrip, final ResultListener<Object> rl) {
		this._invoker.postUnit(new Invoker.Unit("CoinManager:finalizePurchase") {
			public boolean invoke() {
				try {
					if (!CoinManager.this._coinRepo.spendCoins(reservationId, type, descrip)) {
						Log.log.warning("Holy crap! Unable to spend spent coins, what the hell happened to them? [reservationId=" + reservationId + "].", new Object[0]);
					}
				} catch (Exception pe) {
					Log.log.warning("Oh crap! Exception while attempting to finalize a coin purchase, looks like they got the goods for free [reservationId=" + reservationId + "].", new Object[] { pe });
				}

				return true;
			}

			public void handleResult() {
				rl.requestCompleted(result);
			}
		});
	}

	protected void summarizeCoinHistory() {
		Date yest = Calendars.now().addDays(-1).toSQLDate();
		try {
			List<DailySummary> histo = this._coinRepo.loadHistory(yest, yest);
			if (histo.isEmpty()) {
				this._coinRepo.summarizeHistory(yest, yest);
			}

			this._coinRepo.pruneTransactions();
		} catch (Exception e) {
			Log.log.warning("Choked while summarizing coin history.", new Object[] { e });
		}
	}

	public static class InsufficientFundsException extends Exception {
		public InsufficientFundsException() {
			super();
		}
	}

	public static abstract interface CoinOp {
		public abstract void processPurchase(ResultListener<Object> paramResultListener);
	}
}