package com.hexnova.common.http;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.lang.reflect.Type;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.http.HeaderElement;
import org.apache.http.HeaderElementIterator;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.config.SocketConfig;
import org.apache.http.conn.ConnectionKeepAliveStrategy;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.DefaultHttpRequestRetryHandler;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicHeaderElementIterator;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.protocol.HTTP;
import org.apache.http.protocol.HttpContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;

public class HttpUtil {
	

	private static Logger log = LoggerFactory.getLogger(HttpUtil.class);
	
	static RequestConfig requestConfig = RequestConfig.custom().setConnectTimeout(5000).setSocketTimeout(5000).build();
	static CloseableHttpClient httpClient = null;
	
	static {
		
		SocketConfig socketConfig = SocketConfig.custom().setSoTimeout(5000).setSoKeepAlive(true).setTcpNoDelay(true).build();  
  
        /** 
         * 每个默认的 ClientConnectionPoolManager 实现将给每个route创建不超过2个并发连接，最多20个连接总数。 
         */  
        PoolingHttpClientConnectionManager connManager = new PoolingHttpClientConnectionManager();  
        connManager.setMaxTotal(200);  
        connManager.setDefaultMaxPerRoute(50);  
        connManager.setDefaultSocketConfig(socketConfig);  
		
		ConnectionKeepAliveStrategy myStrategy = new ConnectionKeepAliveStrategy() {

		    public long getKeepAliveDuration(HttpResponse response, HttpContext context) {
		        // Honor 'keep-alive' header
		        HeaderElementIterator it = new BasicHeaderElementIterator(
		                response.headerIterator(HTTP.CONN_KEEP_ALIVE));
		        while (it.hasNext()) {
		            HeaderElement he = it.nextElement();
		            String param = he.getName();
		            String value = he.getValue();
		            if (value != null && param.equalsIgnoreCase("timeout")) {
		                try {
		                    return Long.parseLong(value) * 1000;
		                } catch(NumberFormatException ignore) {
		                }
		            }
		        }
	            return 5 * 1000 * 60;
		    }

		};
		DefaultHttpRequestRetryHandler retryHandler = new DefaultHttpRequestRetryHandler();
		httpClient = HttpClients.custom().setConnectionManager(connManager).setRetryHandler(retryHandler).build();
	}
	
	/**
     * http invoke，对返回的json进行Gson对象反序列化,日期格式为：yyyy-MM-dd HH:mm:ss
     * @param paramters
     * @param url
     * @param post
     * @param t
     * @return
     * @throws IOException
     */
    public static <T> T httpInvoke(Map<String,String> paramters,String url,boolean post,Type t) throws IOException{
    	return httpInvoke(paramters,null,url,post,t);
    }
	
	/**
     * http invoke，对返回的json进行Gson对象反序列化,日期格式为：yyyy-MM-dd HH:mm:ss
     * @param paramters
     * @param url
     * @param post
     * @param t
     * @return
     * @throws IOException
     */
    public static <T> T httpInvoke(Map<String,String> paramters,ContentType type,String url,boolean post,Type t) throws IOException{
    	HttpRequestBase request = null;
    	if(post) {
    		request = new HttpPost(url);
    	}else {
    		request = new HttpGet(url);
    	}
    	request.setConfig(requestConfig);
    	
    	if(type != null) {
    		request.addHeader("Content-Type", type.toString());
    	}
    	
    	if(post) {
        	List<NameValuePair> pairs = new ArrayList<NameValuePair>();  
        	if(paramters != null) {
    			for(Map.Entry<String, String> entry:paramters.entrySet()) {
    				NameValuePair pair = new BasicNameValuePair(entry.getKey(), entry.getValue());  
    				pairs.add(pair);
    			}
    		}
    		((HttpPost)request).setEntity(new UrlEncodedFormEntity(pairs, HTTP.UTF_8));
    	
    	}else {
    		
    		URIBuilder uriBuilder = new URIBuilder(request.getURI());
    		
    		if(paramters != null) {
    			for(Map.Entry<String, String> entry:paramters.entrySet()) {
    				uriBuilder.setParameter(entry.getKey(), entry.getValue());
    			}
    			try {
					request.setURI(uriBuilder.build());
				} catch (URISyntaxException e) {
					throw new IOException(e);
				}
    		}
    	}
    	
        ByteArrayOutputStream os = new ByteArrayOutputStream();
        CloseableHttpResponse response = httpClient.execute(request);
        
        
        try {
            HttpEntity entity = response.getEntity();
            if (entity != null) {
               entity.writeTo(os);
            }
            
            if(response.getStatusLine().getStatusCode() != 200){
        		log.error("url="+url+", http status="+response.getStatusLine().getStatusCode());
        		throw new IOException("http status="+response.getStatusLine().getStatusCode());
        	}
            
            Gson gson = new GsonBuilder().setDateFormat("yyyy-MM-dd HH:mm:ss").create();
           return gson.fromJson(new String(os.toByteArray(),"UTF8"),t);

        } finally {
            response.close();
        }
    	
    }
    
    
    /**
     * http invoke，对返回的json进行Gson对象反序列化，日期格式为：yyyy-MM-dd HH:mm:ss
     * @param body 
     * @param type
     * @param url
     * @param t
     * @return
     * @throws IOException
     */
	public static <T> T httpPostInvoke(String body, ContentType type, String url, Type t) throws IOException {
		return httpPostInvoke(body == null ?null :new StringEntity(body, type),type, url, t, null);
	}
	
    /**
     * http invoke，对返回的json进行Gson对象反序列化，日期格式为：yyyy-MM-dd HH:mm:ss
     * @param body
     * @param url
     * @param t
     * @param headers
     * @return
     * @throws IOException
     */
    public static <T> T httpPostInvoke(HttpEntity body,ContentType type, String url, Type t, Map<String, String> headers) throws IOException {
    	HttpRequestBase request = new HttpPost(url);
    	
    	request.setConfig(requestConfig);
    	
    	if (headers != null) {
    		for (String key : headers.keySet()) {
    			request.addHeader(key, headers.get(key));
    		}
    	}
    	if(type != null) {
    		request.addHeader("Content-Type", type.toString());
    	}
    	
    	if(body != null) {
    		((HttpPost) request).setEntity(body);
    	}
        ByteArrayOutputStream os = new ByteArrayOutputStream();
        CloseableHttpResponse response = httpClient.execute(request);
        
        try {
            HttpEntity entity = response.getEntity();
            if (entity != null) {
               entity.writeTo(os);
            }
            
            if(response.getStatusLine().getStatusCode() != 200){
        		log.error("url="+url+", http status="+response.getStatusLine().getStatusCode());
        		//throw new IOException("url="+url+", http status="+response.getStatusLine().getStatusCode());
        	}
            
           Gson gson = new GsonBuilder().setDateFormat("yyyy-MM-dd HH:mm:ss").create();
           return gson.fromJson(new String(os.toByteArray(),"UTF8"),t);

        } finally {
            response.close();
        }
    }
}
