define('util/presenter',
    ['underscore', 'zepto', 'backbone', 'util/app-data', 'util/scroll-tops'],
    function (_, $, Backbone, appData, Scrolling) {

        var isMobileSafari = /iPhone|iPad/.test(navigator.userAgent);

        // No Zombie attacks please....
        // http://lostechies.com/derickbailey/2011/09/15/zombies-run-managing-page-transitions-in-backbone-apps/
        Backbone.View.prototype.close = function(){
            console.log(">>> Closing: ", this);
            this.remove();
            this.unbind();
            if (this.onClose) {
                this.onClose();
            }
        };

        var presenter = {
            currentView: null,
            currentFragment: null,
            currentOverlay: null,

            /**
             * Should only be called by a Router
             */
            init: function (options) {
                this.cacheLength = options.cacheLength || 3;
                this.cache = [];
                this.historyTrail = [];
                this.container = options.container;
                var that = this;
                $(window).on('orientationchange', function() {
                    that.fixCardHeight(true);
                });
                $(document).on('touchmove', '.mob-overlay', this.preventOverlayScroll);
                return this;
            },

            /**
             * Display a Backbone view as the main page, and handle the cleanup of previous views.
             * This is the important method for transitioning between app pages.
             *
             * @param key String key to use as a cache map reference
             * @param view Backbone view to display
             * @param options
             */
            showView: function (key, view, options) {
                options || (options = {});
                if (options.clearCache) {
                    this.cache = [];
                }

                if (options.cacheView !== false) {
                    // If the view is already in the cache, remove it from the list so it can be added to the end
                    for (var i = 0, ii = this.cache.length; i < ii; i++) {
                        if (this.cache[i].view === view) {
                            this.cache.splice(i, 1);
                            break;
                        }
                    }
                    // Remove the oldest item from the cache if the cache is too large
                    if (this.cache.length == this.cacheLength) {
                        var item = this.cache.shift();
                        item.view.close();
                    }
                    // Add the view to the cache
                    this.cache.push({
                        view: view,
                        key: key || +new Date()
                    });
                }

                if (this.currentView && this.currentView.onHide) {
                    this.currentView.onHide();
                }
                this.currentView = view;
                this.currentFragment = Backbone.history.fragment;
                if (view.onShow) {
                    view.onShow();
                }
                view.render();
                this.container.html(view.el);
                this.fixCardHeight(false);
                this.fixScrollPosition();

                var title = _.result(view, 'pageTitle');
                document.title = (title ? title + ' - ' : '') + appData.get('app-title');
            },

            /**
             * Display a Backbone view over the top of page views. Only allows one overlay at a time.
             *
             * @param key String key to use as a cache map reference
             * @param view Backbone view to display
             */
            showOverlay: function (key, view) {
                if (this.currentOverlay && this.currentOverlay.key !== key) {
                    this.hideOverlay();
                }
                if (view.onShow) {
                    view.onShow();
                }
                view.render();

                var overlayId = 'overlay-' + key.replace(/[^a-zA-Z0-9\-]/g, '');
                var $elem = this.container.find('#' + overlayId);
                if (!$elem.length) {
                    $elem = $('<div id="' + overlayId + '"/>')
                        .addClass('mob-overlay full-page')
                        .prependTo(this.container);
                }
                $elem.html(view.el).addClass('show');
                if (view.autofocus) {
                    view.autofocus.focus();
                }

                this.currentOverlay = {
                    key: key,
                    elem: $elem,
                    view: view
                };
            },

            /**
             * Hide the currently displayed overlay
             *
             * @param key If provided, only hide the overlay if it matches the key
             */
            hideOverlay: function (key) {
                var $elem;
                if (!this.currentOverlay) {
                    return;
                }
                if (key && this.currentOverlay.key !== key) {
                    return;
                }
                $elem = this.currentOverlay.elem;
                $elem.removeClass('show');
                this.currentOverlay.view.close();
                $elem.empty();

                this.currentOverlay = null;
            },

            /**
             * Prevent scrolling if a touchmove event was fired directly on an overlay blanket, but not its children
             */
            preventOverlayScroll: function (e) {
                var $target = $(e.target);
                if ($target.hasClass('mob-overlay')) {
                    e.preventDefault();
                }
            },

            fixCardHeight: function (isEvent) {
                var $cards = $(".card");

                // If this isn't an event, then get the height.
                // If it is an event but is safari, then also get the height.
                // Otherwise get the width because android throws the event then changes orientation
                var viewportHeight = !isEvent || isMobileSafari ? window.innerHeight : window.innerWidth,
                    tooShort = $cards.height() <= viewportHeight;

                if (tooShort) {
                    // +60 to account for address bar hiding in iOS
                    // +1 to account for window.scroll(0, 1)
                    var newHeight = viewportHeight + (isMobileSafari ? 61 : 1);
                    $cards.css("min-height", newHeight);
                }
            },

            fixScrollPosition: function (){
                if (Scrolling.hasSavedPosition()) {
                    Scrolling.restorePosition();
                } else {
                    var selectedComment = $('.comment.selected-comment');
                    if (selectedComment.length !== 0) {
                        // Scroll to the selected comment
                        setTimeout(function () {
                            window.scroll(0, selectedComment.offset().top - (window.innerHeight / 4));
                        }, 0);
                    } else {
                        // if there was no previous scroll position, scroll to the top
                        setTimeout(function () {
                            window.scroll(0, 1);
                        }, 0);
                    }
                }
            },

            /**
             * Show a full-page view if it's already in the recently-viewed cache, to avoid network requests.
             *
             * @param key String key to check in cache map (as provided to showView())
             * @return bool True if cached view was used, false otherwise
             */
            useCachedView: function (key) {
                if (!this.cache.length) {
                    return false;
                }
                var that = this;
                return _.any(this.cache, function (item) {
                    if (item.key === key) {
                        that.showView(item.key, item.view);
                        return true;
                    }
                    return false;
                });
            },

            /**
             * Reloads the server data for the current full-page view, using Backbone's fetch() method
             *
             * @return bool True if the server was contacted, false for no view or a static view (not data-backed)
             */
            refreshCurrentView: function () {
                if (!this.currentView) {
                    return false;
                }
                var view = this.currentView;
                if (view.model) {
                    view.model.fetch();
                } else if (view.collection) {
                    view.collection.fetch();
                } else {
                    return false;
                }
                return true;
            }
        };

        // Proxy methods that are called on the currently active full-page view
        _.each(['showLoading', 'showHeaderLoading', 'hideLoading', 'cancelLoading'], function (method) {
            presenter[method] = function () {
                if (this.currentView) {
                    return this.currentView[method].apply(this.currentView, arguments);
                }
            }
        });

        return presenter;
    }
);
