/*
 * Decompiled with CFR 0.152.
 */
package com.android.builder.internal.utils;

import com.android.builder.internal.utils.ReadWriteProcessLock;
import com.android.utils.FileUtils;
import com.google.common.base.Joiner;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.collect.Maps;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hashing;
import com.google.common.io.Files;
import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

public final class FileCache {
    private static final ConcurrentMap<File, FileCache> sFileCacheMap = new ConcurrentHashMap<File, FileCache>();
    private final File mCacheDirectory;
    private final LockingScope mLockingScope;
    private final ConcurrentMap<File, ReentrantReadWriteLock> mLockMap = new ConcurrentHashMap<File, ReentrantReadWriteLock>();
    private final AtomicInteger mMisses = new AtomicInteger(0);
    private final AtomicInteger mHits = new AtomicInteger(0);

    private FileCache(File canonicalCacheDirectory, LockingScope lockingScope) {
        this.mCacheDirectory = canonicalCacheDirectory;
        this.mLockingScope = lockingScope;
    }

    private static FileCache getInstance(File cacheDirectory, LockingScope lockingScope) throws IOException {
        FileCache fileCache = sFileCacheMap.computeIfAbsent(cacheDirectory.getCanonicalFile(), canonicalCacheDirectory -> new FileCache((File)canonicalCacheDirectory, lockingScope));
        if (lockingScope != fileCache.mLockingScope) {
            if (lockingScope == LockingScope.INTER_PROCESS) {
                throw new IllegalStateException("Unable to create FileCache with inter-process locking enabled since inter-process locking was previously disabled on the same cache.");
            }
            throw new IllegalStateException("Unable to create FileCache with inter-process locking disabled since inter-process locking was previously enabled on the same cache.");
        }
        return fileCache;
    }

    public static FileCache getInstanceWithInterProcessLocking(File cacheDirectory) throws IOException {
        return FileCache.getInstance(cacheDirectory, LockingScope.INTER_PROCESS);
    }

    public static FileCache getInstanceWithSingleProcessLocking(File cacheDirectory) throws IOException {
        return FileCache.getInstance(cacheDirectory, LockingScope.SINGLE_PROCESS);
    }

    public File getCacheDirectory() {
        return this.mCacheDirectory;
    }

    public void createFile(File outputFile, Inputs inputs, Callable<Void> fileProducer) throws ExecutionException, IOException {
        Preconditions.checkArgument((!FileUtils.isFileInDirectory((File)outputFile, (File)this.mCacheDirectory) ? 1 : 0) != 0, (Object)String.format("Output file/directory '%1$s' must not be located in the cache directory '%2$s'", outputFile.getAbsolutePath(), this.mCacheDirectory.getAbsolutePath()));
        Preconditions.checkArgument((!FileUtils.isFileInDirectory((File)this.mCacheDirectory, (File)outputFile) ? 1 : 0) != 0, (Object)String.format("Output directory '%1$s' must not contain the cache directory '%2$s'", outputFile.getAbsolutePath(), this.mCacheDirectory.getAbsolutePath()));
        Preconditions.checkArgument((!outputFile.getCanonicalFile().equals(this.mCacheDirectory.getCanonicalFile()) ? 1 : 0) != 0, (Object)String.format("Output directory must not be the same as the cache directory '%1$s'", this.mCacheDirectory.getAbsolutePath()));
        File cacheEntryDir = new File(this.mCacheDirectory, inputs.getKey());
        File cachedFile = new File(cacheEntryDir, "output");
        File inputsFile = new File(cacheEntryDir, "inputs");
        Callable<Void> createOutputFile = () -> {
            FileUtils.deletePath((File)outputFile);
            Files.createParentDirs((File)outputFile);
            try {
                fileProducer.call();
            }
            catch (Exception exception) {
                throw new FileProducerException(exception);
            }
            return null;
        };
        Callable<Void> copyOutputFileToCachedFile = () -> {
            if (outputFile.exists()) {
                FileCache.copyFileOrDirectory(outputFile, cachedFile);
            }
            return null;
        };
        Callable<Void> copyCachedFileToOutputFile = () -> {
            FileUtils.deletePath((File)outputFile);
            Files.createParentDirs((File)outputFile);
            if (cachedFile.exists()) {
                FileCache.copyFileOrDirectory(cachedFile, outputFile);
            }
            return null;
        };
        try {
            this.doLocked(this.mCacheDirectory, LockingType.SHARED, () -> {
                FileUtils.mkdirs((File)this.mCacheDirectory);
                boolean isHit = this.doLocked(cacheEntryDir, LockingType.SHARED, () -> {
                    if (cacheEntryDir.exists()) {
                        this.mHits.incrementAndGet();
                        this.doLocked(outputFile, LockingType.EXCLUSIVE, copyCachedFileToOutputFile);
                        return true;
                    }
                    return false;
                });
                if (isHit) {
                    return null;
                }
                this.doLocked(cacheEntryDir, LockingType.EXCLUSIVE, () -> {
                    if (cacheEntryDir.exists()) {
                        this.mHits.incrementAndGet();
                        this.doLocked(outputFile, LockingType.EXCLUSIVE, copyCachedFileToOutputFile);
                    } else {
                        this.mMisses.incrementAndGet();
                        try {
                            FileUtils.mkdirs((File)cacheEntryDir);
                            this.doLocked(outputFile, LockingType.EXCLUSIVE, () -> {
                                createOutputFile.call();
                                copyOutputFileToCachedFile.call();
                                return null;
                            });
                            Files.write((CharSequence)inputs.toString(), (File)inputsFile, (Charset)StandardCharsets.UTF_8);
                        }
                        catch (Exception exception) {
                            FileUtils.deletePath((File)cacheEntryDir);
                            throw exception;
                        }
                    }
                    return null;
                });
                return null;
            });
        }
        catch (ExecutionException exception) {
            for (Throwable cause : Throwables.getCausalChain((Throwable)exception)) {
                if (cause instanceof FileProducerException) {
                    throw exception;
                }
                if (cause instanceof IOException) {
                    throw new IOException(exception);
                }
                if (cause instanceof ExecutionException) continue;
                throw new RuntimeException(exception);
            }
            throw new RuntimeException("Unable to find root cause of ExecutionException, " + exception);
        }
    }

    public void delete() throws IOException {
        try {
            this.doLocked(this.mCacheDirectory, LockingType.EXCLUSIVE, () -> {
                FileUtils.deletePath((File)this.mCacheDirectory);
                return null;
            });
        }
        catch (ExecutionException exception) {
            if (exception.getCause() instanceof IOException) {
                throw new IOException(exception);
            }
            throw new RuntimeException(exception);
        }
    }

    <V> V doLocked(File accessedFile, LockingType lockingType, Callable<V> action) throws ExecutionException, IOException {
        if (this.mLockingScope == LockingScope.INTER_PROCESS) {
            return this.doInterProcessLocked(accessedFile, lockingType, action);
        }
        return this.doSingleProcessLocked(accessedFile, lockingType, action);
    }

    private <V> V doInterProcessLocked(File accessedFile, LockingType lockingType, Callable<V> action) throws ExecutionException, IOException {
        String lockFileName = Hashing.sha1().hashString((CharSequence)FileUtils.getCaseSensitivityAwareCanonicalPath((File)accessedFile), StandardCharsets.UTF_8).toString();
        File lockFile = accessedFile.getCanonicalFile().equals(this.mCacheDirectory) ? new File(System.getProperty("java.io.tmpdir"), lockFileName) : new File(this.mCacheDirectory, lockFileName);
        ReadWriteProcessLock readWriteProcessLock = ReadWriteProcessLock.getInstance(lockFile);
        ReadWriteProcessLock.Lock lock = lockingType == LockingType.SHARED ? readWriteProcessLock.readLock() : readWriteProcessLock.writeLock();
        lock.lock();
        try {
            V v = action.call();
            return v;
        }
        catch (Exception exception) {
            throw new ExecutionException(exception);
        }
        finally {
            lock.unlock();
        }
    }

    private <V> V doSingleProcessLocked(File accessedFile, LockingType lockingType, Callable<V> action) throws ExecutionException, IOException {
        ReadWriteLock readWriteLock = this.mLockMap.computeIfAbsent(accessedFile.getCanonicalFile(), canonicalAccessedFile -> new ReentrantReadWriteLock());
        Lock lock = lockingType == LockingType.SHARED ? readWriteLock.readLock() : readWriteLock.writeLock();
        lock.lock();
        try {
            V v = action.call();
            return v;
        }
        catch (Exception exception) {
            throw new ExecutionException(exception);
        }
        finally {
            lock.unlock();
        }
    }

    int getMisses() {
        return this.mMisses.get();
    }

    int getHits() {
        return this.mHits.get();
    }

    public String toString() {
        return MoreObjects.toStringHelper((Object)this).add("cacheDirectory", (Object)this.mCacheDirectory).add("lockingScope", (Object)this.mLockingScope).toString();
    }

    private static void copyFileOrDirectory(File from, File to) throws IOException {
        Preconditions.checkArgument((boolean)from.exists(), (Object)("Source path " + from.getAbsolutePath() + " does not exists."));
        Preconditions.checkArgument((!FileUtils.isFileInDirectory((File)from, (File)to) ? 1 : 0) != 0);
        Preconditions.checkArgument((!FileUtils.isFileInDirectory((File)to, (File)from) ? 1 : 0) != 0);
        Preconditions.checkArgument((!from.getCanonicalFile().equals(to.getCanonicalFile()) ? 1 : 0) != 0);
        if (from.isFile()) {
            Files.createParentDirs((File)to);
            FileUtils.copyFile((File)from, (File)to);
        } else if (from.isDirectory()) {
            FileUtils.deletePath((File)to);
            FileUtils.copyDirectory((File)from, (File)to);
        }
    }

    public static final class Inputs {
        private final LinkedHashMap<String, String> mParameters;

        private Inputs(Builder builder) {
            this.mParameters = Maps.newLinkedHashMap((Map)builder.mParameters);
        }

        public String toString() {
            return Joiner.on((String)System.lineSeparator()).withKeyValueSeparator("=").join(this.mParameters);
        }

        public String getKey() {
            return Hashing.sha1().hashString((CharSequence)this.toString(), StandardCharsets.UTF_8).toString();
        }

        public static final class Builder {
            private final LinkedHashMap<String, String> mParameters = Maps.newLinkedHashMap();

            public Builder putFilePath(String name, File file) {
                this.mParameters.put(name, file.getPath());
                return this;
            }

            public Builder putFileHash(String name, File file) throws IOException {
                Preconditions.checkArgument((boolean)file.isFile(), (Object)(file + " is not a file."));
                this.mParameters.put(name, Files.hash((File)file, (HashFunction)Hashing.sha1()).toString());
                return this;
            }

            public Builder putString(String name, String value) {
                this.mParameters.put(name, value);
                return this;
            }

            public Builder putBoolean(String name, boolean value) {
                this.mParameters.put(name, String.valueOf(value));
                return this;
            }

            public Inputs build() {
                Preconditions.checkState((!this.mParameters.isEmpty() ? 1 : 0) != 0, (Object)"Inputs must not be empty.");
                return new Inputs(this);
            }
        }
    }

    private static final class FileProducerException
    extends ExecutionException {
        public FileProducerException(Exception exception) {
            super(exception);
        }
    }

    static enum LockingType {
        SHARED,
        EXCLUSIVE;

    }

    private static enum LockingScope {
        INTER_PROCESS,
        SINGLE_PROCESS;

    }
}

