/*
 * Copyright LWJGL. All rights reserved.
 * License terms: https://www.lwjgl.org/license
 * MACHINE GENERATED FILE, DO NOT EDIT
 */
package org.lwjgl.opengl;

import java.nio.*;

import org.lwjgl.system.*;

import static org.lwjgl.system.Checks.*;
import static org.lwjgl.system.JNI.*;
import static org.lwjgl.system.MemoryStack.*;
import static org.lwjgl.system.MemoryUtil.*;

/**
 * Native bindings to the <a target="_blank" href="http://www.opengl.org/registry/specs/ARB/viewport_array.txt">ARB_viewport_array</a> extension.
 * 
 * <p>OpenGL is modeled on a pipeline of operations. The final stage in this pipeline before rasterization is the viewport transformation. This stage
 * transforms vertices from view space into window coordinates and allows the application to specify a rectangular region of screen space into which OpenGL
 * should draw primitives. Unextended OpenGL implementations provide a single viewport per context. In order to draw primitives into multiple viewports,
 * the OpenGL viewport may be changed between several draw calls. With the advent of Geometry Shaders, it has become possible for an application to amplify
 * geometry and produce multiple output primitives for each primitive input to the Geometry Shader. It is possible to direct these primitives to render
 * into a selected render target. However, all render targets share the same, global OpenGL viewport.</p>
 * 
 * <p>This extension enhances OpenGL by providing a mechanism to expose multiple viewports. Each viewport is specified as a rectangle. The destination
 * viewport may be selected per-primitive by the geometry shader. This allows the Geometry Shader to produce different versions of primitives destined for
 * separate viewport rectangles on the same surface. Additionally, when combined with multiple framebuffer attachments, it allows a different viewport
 * rectangle to be selected for each. This extension also exposes a separate scissor rectangle for each viewport. Finally, the viewport bounds are now
 * floating point quantities allowing fractional pixel offsets to be applied during the viewport transform.</p>
 * 
 * <p>Requires {@link GL32 OpenGL 3.2} or <a target="_blank" href="http://www.opengl.org/registry/specs/EXT/geometry_shader4.txt">EXT_geometry_shader4</a> or {@link ARBGeometryShader4 ARB_geometry_shader4}. Promoted to core in {@link GL41 OpenGL 4.1}.</p>
 */
public class ARBViewportArray {

    /** Accepted by the {@code pname} parameter of GetBooleanv, GetIntegerv, GetFloatv, GetDoublev and GetInteger64v. */
    public static final int
        GL_MAX_VIEWPORTS                   = 0x825B,
        GL_VIEWPORT_SUBPIXEL_BITS          = 0x825C,
        GL_VIEWPORT_BOUNDS_RANGE           = 0x825D,
        GL_LAYER_PROVOKING_VERTEX          = 0x825E,
        GL_VIEWPORT_INDEX_PROVOKING_VERTEX = 0x825F;

    /** Returned in the {@code data} parameter from a Get query with a {@code pname} of LAYER_PROVOKING_VERTEX or VIEWPORT_INDEX_PROVOKING_VERTEX. */
    public static final int GL_UNDEFINED_VERTEX = 0x8260;

    static { GL.initialize(); }

    protected ARBViewportArray() {
        throw new UnsupportedOperationException();
    }

    static boolean isAvailable(GLCapabilities caps) {
        return checkFunctions(
            caps.glViewportArrayv, caps.glViewportIndexedf, caps.glViewportIndexedfv, caps.glScissorArrayv, caps.glScissorIndexed, caps.glScissorIndexedv, 
            caps.glDepthRangeArrayv, caps.glDepthRangeIndexed, caps.glGetFloati_v, caps.glGetDoublei_v
        );
    }

    // --- [ glViewportArrayv ] ---

    /**
     * Unsafe version of: {@link #glViewportArrayv ViewportArrayv}
     *
     * @param count the number of viewports to set
     */
    public static native void nglViewportArrayv(int first, int count, long v);

    /**
     * Sets multiple viewports.
     *
     * @param first the first viewport to set
     * @param v     an array containing the viewport parameters
     */
    public static void glViewportArrayv(@NativeType("GLuint") int first, @NativeType("const GLfloat *") FloatBuffer v) {
        nglViewportArrayv(first, v.remaining() >> 2, memAddress(v));
    }

    // --- [ glViewportIndexedf ] ---

    /**
     * Sets a specified viewport.
     *
     * @param index the viewport to set
     * @param x     the left viewport coordinate
     * @param y     the bottom viewport coordinate
     * @param w     the viewport width
     * @param h     the viewport height
     */
    public static native void glViewportIndexedf(@NativeType("GLuint") int index, @NativeType("GLfloat") float x, @NativeType("GLfloat") float y, @NativeType("GLfloat") float w, @NativeType("GLfloat") float h);

    // --- [ glViewportIndexedfv ] ---

    /** Unsafe version of: {@link #glViewportIndexedfv ViewportIndexedfv} */
    public static native void nglViewportIndexedfv(int index, long v);

    /**
     * Pointer version of {@link #glViewportIndexedf ViewportIndexedf}.
     *
     * @param index the viewport to set
     * @param v     the viewport parameters
     */
    public static void glViewportIndexedfv(@NativeType("GLuint") int index, @NativeType("const GLfloat *") FloatBuffer v) {
        if (CHECKS) {
            check(v, 4);
        }
        nglViewportIndexedfv(index, memAddress(v));
    }

    // --- [ glScissorArrayv ] ---

    /**
     * Unsafe version of: {@link #glScissorArrayv ScissorArrayv}
     *
     * @param count the number of scissor boxes to modify
     */
    public static native void nglScissorArrayv(int first, int count, long v);

    /**
     * Defines the scissor box for multiple viewports.
     *
     * @param first the index of the first viewport whose scissor box to modify
     * @param v     an array containing the left, bottom, width and height of each scissor box, in that order
     */
    public static void glScissorArrayv(@NativeType("GLuint") int first, @NativeType("const GLint *") IntBuffer v) {
        nglScissorArrayv(first, v.remaining() >> 2, memAddress(v));
    }

    // --- [ glScissorIndexed ] ---

    /**
     * Defines the scissor box for a specific viewport.
     *
     * @param index  the index of the viewport whose scissor box to modify
     * @param left   the left scissor box coordinate
     * @param bottom the bottom scissor box coordinate
     * @param width  the scissor box width
     * @param height the scissor box height
     */
    public static native void glScissorIndexed(@NativeType("GLuint") int index, @NativeType("GLint") int left, @NativeType("GLint") int bottom, @NativeType("GLsizei") int width, @NativeType("GLsizei") int height);

    // --- [ glScissorIndexedv ] ---

    /** Unsafe version of: {@link #glScissorIndexedv ScissorIndexedv} */
    public static native void nglScissorIndexedv(int index, long v);

    /**
     * Pointer version of {@link #glScissorIndexed ScissorIndexed}.
     *
     * @param index the index of the viewport whose scissor box to modify
     * @param v     an array containing the left, bottom, width and height of each scissor box, in that order
     */
    public static void glScissorIndexedv(@NativeType("GLuint") int index, @NativeType("const GLint *") IntBuffer v) {
        if (CHECKS) {
            check(v, 4);
        }
        nglScissorIndexedv(index, memAddress(v));
    }

    // --- [ glDepthRangeArrayv ] ---

    /**
     * Unsafe version of: {@link #glDepthRangeArrayv DepthRangeArrayv}
     *
     * @param count the number of viewports whose depth range to update
     */
    public static native void nglDepthRangeArrayv(int first, int count, long v);

    /**
     * Specifies mapping of depth values from normalized device coordinates to window coordinates for a specified set of viewports.
     *
     * @param first the index of the first viewport whose depth range to update
     * @param v     n array containing the near and far values for the depth range of each modified viewport
     */
    public static void glDepthRangeArrayv(@NativeType("GLuint") int first, @NativeType("const GLdouble *") DoubleBuffer v) {
        nglDepthRangeArrayv(first, v.remaining() >> 1, memAddress(v));
    }

    // --- [ glDepthRangeIndexed ] ---

    /**
     * Specifies mapping of depth values from normalized device coordinates to window coordinates for a specified viewport.
     *
     * @param index the index of the viewport whose depth range to update
     * @param zNear the mapping of the near clipping plane to window coordinates. The initial value is 0.
     * @param zFar  the mapping of the far clipping plane to window coordinates. The initial value is 1.
     */
    public static native void glDepthRangeIndexed(@NativeType("GLuint") int index, @NativeType("GLdouble") double zNear, @NativeType("GLdouble") double zFar);

    // --- [ glGetFloati_v ] ---

    /** Unsafe version of: {@link #glGetFloati_v GetFloati_v} */
    public static native void nglGetFloati_v(int target, int index, long data);

    /**
     * Queries the float value of an indexed state variable.
     *
     * @param target the indexed state to query
     * @param index  the index of the element being queried
     * @param data   a scalar or buffer in which to place the returned data
     */
    public static void glGetFloati_v(@NativeType("GLenum") int target, @NativeType("GLuint") int index, @NativeType("GLfloat *") FloatBuffer data) {
        if (CHECKS) {
            check(data, 1);
        }
        nglGetFloati_v(target, index, memAddress(data));
    }

    /**
     * Queries the float value of an indexed state variable.
     *
     * @param target the indexed state to query
     * @param index  the index of the element being queried
     */
    @NativeType("void")
    public static float glGetFloati(@NativeType("GLenum") int target, @NativeType("GLuint") int index) {
        MemoryStack stack = stackGet(); int stackPointer = stack.getPointer();
        try {
            FloatBuffer data = stack.callocFloat(1);
            nglGetFloati_v(target, index, memAddress(data));
            return data.get(0);
        } finally {
            stack.setPointer(stackPointer);
        }
    }

    // --- [ glGetDoublei_v ] ---

    /** Unsafe version of: {@link #glGetDoublei_v GetDoublei_v} */
    public static native void nglGetDoublei_v(int target, int index, long data);

    /**
     * Queries the double value of an indexed state variable.
     *
     * @param target the indexed state to query
     * @param index  the index of the element being queried
     * @param data   a scalar or buffer in which to place the returned data
     */
    public static void glGetDoublei_v(@NativeType("GLenum") int target, @NativeType("GLuint") int index, @NativeType("GLdouble *") DoubleBuffer data) {
        if (CHECKS) {
            check(data, 1);
        }
        nglGetDoublei_v(target, index, memAddress(data));
    }

    /**
     * Queries the double value of an indexed state variable.
     *
     * @param target the indexed state to query
     * @param index  the index of the element being queried
     */
    @NativeType("void")
    public static double glGetDoublei(@NativeType("GLenum") int target, @NativeType("GLuint") int index) {
        MemoryStack stack = stackGet(); int stackPointer = stack.getPointer();
        try {
            DoubleBuffer data = stack.callocDouble(1);
            nglGetDoublei_v(target, index, memAddress(data));
            return data.get(0);
        } finally {
            stack.setPointer(stackPointer);
        }
    }

    /** Array version of: {@link #glViewportArrayv ViewportArrayv} */
    public static void glViewportArrayv(@NativeType("GLuint") int first, @NativeType("const GLfloat *") float[] v) {
        long __functionAddress = GL.getICD().glViewportArrayv;
        if (CHECKS) {
            check(__functionAddress);
        }
        callPV(__functionAddress, first, v.length >> 2, v);
    }

    /** Array version of: {@link #glViewportIndexedfv ViewportIndexedfv} */
    public static void glViewportIndexedfv(@NativeType("GLuint") int index, @NativeType("const GLfloat *") float[] v) {
        long __functionAddress = GL.getICD().glViewportIndexedfv;
        if (CHECKS) {
            check(__functionAddress);
            check(v, 4);
        }
        callPV(__functionAddress, index, v);
    }

    /** Array version of: {@link #glScissorArrayv ScissorArrayv} */
    public static void glScissorArrayv(@NativeType("GLuint") int first, @NativeType("const GLint *") int[] v) {
        long __functionAddress = GL.getICD().glScissorArrayv;
        if (CHECKS) {
            check(__functionAddress);
        }
        callPV(__functionAddress, first, v.length >> 2, v);
    }

    /** Array version of: {@link #glScissorIndexedv ScissorIndexedv} */
    public static void glScissorIndexedv(@NativeType("GLuint") int index, @NativeType("const GLint *") int[] v) {
        long __functionAddress = GL.getICD().glScissorIndexedv;
        if (CHECKS) {
            check(__functionAddress);
            check(v, 4);
        }
        callPV(__functionAddress, index, v);
    }

    /** Array version of: {@link #glDepthRangeArrayv DepthRangeArrayv} */
    public static void glDepthRangeArrayv(@NativeType("GLuint") int first, @NativeType("const GLdouble *") double[] v) {
        long __functionAddress = GL.getICD().glDepthRangeArrayv;
        if (CHECKS) {
            check(__functionAddress);
        }
        callPV(__functionAddress, first, v.length >> 1, v);
    }

    /** Array version of: {@link #glGetFloati_v GetFloati_v} */
    public static void glGetFloati_v(@NativeType("GLenum") int target, @NativeType("GLuint") int index, @NativeType("GLfloat *") float[] data) {
        long __functionAddress = GL.getICD().glGetFloati_v;
        if (CHECKS) {
            check(__functionAddress);
            check(data, 1);
        }
        callPV(__functionAddress, target, index, data);
    }

    /** Array version of: {@link #glGetDoublei_v GetDoublei_v} */
    public static void glGetDoublei_v(@NativeType("GLenum") int target, @NativeType("GLuint") int index, @NativeType("GLdouble *") double[] data) {
        long __functionAddress = GL.getICD().glGetDoublei_v;
        if (CHECKS) {
            check(__functionAddress);
            check(data, 1);
        }
        callPV(__functionAddress, target, index, data);
    }

}