/*
 * Copyright (c) 2018, 2023, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * The Universal Permissive License (UPL), Version 1.0
 *
 * Subject to the condition set forth below, permission is hereby granted to any
 * person obtaining a copy of this software, associated documentation and/or
 * data (collectively the "Software"), free of charge and under any and all
 * copyright rights in the Software, and any and all patent rights owned or
 * freely licensable by each licensor hereunder covering either (i) the
 * unmodified Software as contributed to or provided by such licensor, or (ii)
 * the Larger Works (as defined below), to deal in both
 *
 * (a) the Software, and
 *
 * (b) any piece of software and/or hardware listed in the lrgrwrks.txt file if
 * one is included with the Software each a "Larger Work" to which the Software
 * is contributed by such licensors),
 *
 * without restriction, including without limitation the rights to copy, create
 * derivative works of, display, perform, and distribute the Software and make,
 * use, sell, offer for sale, import, export, have made, and have sold the
 * Software and the Larger Work(s), and to sublicense the foregoing rights on
 * either these or other terms.
 *
 * This license is subject to the following condition:
 *
 * The above copyright notice and either this complete permission notice or at a
 * minimum a reference to the UPL must be included in all copies or substantial
 * portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.oracle.truffle.js.nodes.function;

import com.oracle.truffle.api.CallTarget;
import com.oracle.truffle.api.CompilerDirectives.TruffleBoundary;
import com.oracle.truffle.api.dsl.Cached;
import com.oracle.truffle.api.dsl.Cached.Shared;
import com.oracle.truffle.api.dsl.Idempotent;
import com.oracle.truffle.api.dsl.NeverDefault;
import com.oracle.truffle.api.dsl.Specialization;
import com.oracle.truffle.api.nodes.DirectCallNode;
import com.oracle.truffle.api.nodes.IndirectCallNode;
import com.oracle.truffle.api.source.Source;
import com.oracle.truffle.js.nodes.JavaScriptBaseNode;
import com.oracle.truffle.js.nodes.interop.ImportValueNode;
import com.oracle.truffle.js.runtime.JSArguments;
import com.oracle.truffle.js.runtime.JSContext;
import com.oracle.truffle.js.runtime.JSRealm;

public abstract class JSLoadNode extends JavaScriptBaseNode {

    protected final JSContext context;

    @NeverDefault
    public static JSLoadNode create(JSContext context) {
        return JSLoadNodeGen.create(context);
    }

    protected JSLoadNode(JSContext context) {
        this.context = context;
    }

    public abstract Object executeLoad(Source source, JSRealm realm);

    protected static CallTarget loadScript(Source source, JSRealm realm) {
        return realm.getEnv().parsePublic(source);
    }

    @TruffleBoundary(allowInlining = true)
    static boolean equals(Source source, Source cachedSource) {
        return source.equals(cachedSource);
    }

    @SuppressWarnings("unused")
    @Specialization(guards = {"isCached(cachedSource)", "equals(source, cachedSource)"}, limit = "1")
    static Object cachedLoad(Source source, JSRealm realm,
                    @Cached @Shared("importValue") ImportValueNode importValue,
                    @Cached(value = "source") Source cachedSource,
                    @Cached("create(loadScript(source, realm))") DirectCallNode callNode) {
        return importValue.executeWithTarget(callNode.call(JSArguments.EMPTY_ARGUMENTS_ARRAY));
    }

    @Specialization(replaces = "cachedLoad")
    static Object uncachedLoad(Source source, JSRealm realm,
                    @Cached @Shared("importValue") ImportValueNode importValue,
                    @Cached IndirectCallNode callNode) {
        return importValue.executeWithTarget(callNode.call(loadScript(source, realm), JSArguments.EMPTY_ARGUMENTS_ARRAY));
    }

    @Idempotent
    static boolean isCached(Source source) {
        return source.isCached();
    }
}
