/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.mxnet

import org.apache.mxnet.annotation.Experimental
import scala.reflect.ClassTag

// scalastyle:off
abstract class SymbolRandomAPIBase {
  /**
  *
  * {{{
  *
  * Draw random samples from an an approximately log-uniform
  * or Zipfian distribution without replacement.
  * 
  * This operation takes a 2-D shape `(batch_size, num_sampled)`,
  * and randomly generates *num_sampled* samples from the range of integers [0, range_max)
  * for each instance in the batch.
  * 
  * The elements in each instance are drawn without replacement from the base distribution.
  * The base distribution for this operator is an approximately log-uniform or Zipfian distribution:
  * 
  *   P(class) = (log(class + 2) - log(class + 1)) / log(range_max + 1)
  * 
  * Additionaly, it also returns the number of trials used to obtain `num_sampled` samples for
  * each instance in the batch.
  * 
  * Example::
  * 
  *    samples, trials = _sample_unique_zipfian(750000, shape=(4, 8192))
  *    unique(samples[0]) = 8192
  *    unique(samples[3]) = 8192
  *    trials[0] = 16435
  * 
  * 
  * 
  * Defined in src/operator/random/unique_sample_op.cc:L66
  * }}}
  * 
  * @param range_max		The number of possible classes.
  * @param shape		2-D shape of the output, where shape[0] is the batch size, and shape[1] is the number of candidates to sample for each batch.
  * @return org.apache.mxnet.Symbol
  */
@Experimental
def unique_zipfian[T: SymbolOrScalar : ClassTag] (range_max : T, shape : Option[org.apache.mxnet.Shape] = None, name : String = null, attr : Map[String, String] = null): org.apache.mxnet.Symbol
  /**
  *
  * {{{
  *
  * Computes the value of the PDF of *sample* of
  * uniform distributions on the intervals given by *[low,high)*.
  * 
  * *low* and *high* must have the same shape, which must match the leftmost subshape
  * of *sample*.  That is, *sample* can have the same shape as *low* and *high*, in which
  * case the output contains one density per distribution, or *sample* can be a tensor
  * of tensors with that shape, in which case the output is a tensor of densities such that
  * the densities at index *i* in the output are given by the samples at index *i* in *sample*
  * parameterized by the values of *low* and *high* at index *i*.
  * 
  * Examples::
  * 
  *     random_pdf_uniform(sample=`[ [1,2,3,4] ], low=[0], high=[10]) = [0.1, 0.1, 0.1, 0.1]
  * 
  *     sample = `[ `[ [1, 2, 3],
  *                [1, 2, 3] ],
  *               `[ [1, 2, 3],
  *                [1, 2, 3] ] ]
  *     low  = `[ [0, 0],
  *             [0, 0] ]
  *     high = `[ [ 5, 10],
  *             [15, 20] ]
  *     random_pdf_uniform(sample=sample, low=low, high=high) =
  *         `[ `[ [0.2,        0.2,        0.2    ],
  *           [0.1,        0.1,        0.1    ] ],
  *          `[ [0.06667,    0.06667,    0.06667],
  *           [0.05,       0.05,       0.05   ] ] ]
  * 
  * 
  * 
  * Defined in src/operator/random/pdf_op.cc:L297
  * }}}
  * 
  * @param sample		Samples from the distributions.
  * @param low		Lower bounds of the distributions.
  * @param is_log		If set, compute the density of the log-probability instead of the probability.
  * @param high		Upper bounds of the distributions.
  * @return org.apache.mxnet.Symbol
  */
@Experimental
def pdf_uniform[T: SymbolOrScalar : ClassTag] (sample : Option[T] = None, low : Option[T] = None, is_log : Option[Boolean] = None, high : Option[T] = None, name : String = null, attr : Map[String, String] = null): org.apache.mxnet.Symbol
  /**
  *
  * {{{
  *
  * Computes the value of the PDF of *sample* of
  * exponential distributions with parameters *lam* (rate).
  * 
  * The shape of *lam* must match the leftmost subshape of *sample*.  That is, *sample*
  * can have the same shape as *lam*, in which case the output contains one density per
  * distribution, or *sample* can be a tensor of tensors with that shape, in which case
  * the output is a tensor of densities such that the densities at index *i* in the output
  * are given by the samples at index *i* in *sample* parameterized by the value of *lam*
  * at index *i*.
  * 
  * Examples::
  * 
  *   random_pdf_exponential(sample=`[ [1, 2, 3] ], lam=[1]) =
  *       `[ [0.36787945, 0.13533528, 0.04978707] ]
  * 
  *   sample = `[ [1,2,3],
  *             [1,2,3],
  *             [1,2,3] ]
  * 
  *   random_pdf_exponential(sample=sample, lam=[1,0.5,0.25]) =
  *       `[ [0.36787945, 0.13533528, 0.04978707],
  *        [0.30326533, 0.18393973, 0.11156508],
  *        [0.1947002,  0.15163267, 0.11809164] ]
  * 
  * 
  * Defined in src/operator/random/pdf_op.cc:L304
  * }}}
  * 
  * @param sample		Samples from the distributions.
  * @param lam		Lambda (rate) parameters of the distributions.
  * @param is_log		If set, compute the density of the log-probability instead of the probability.
  * @return org.apache.mxnet.Symbol
  */
@Experimental
def pdf_exponential[T: SymbolOrScalar : ClassTag] (sample : Option[T] = None, lam : Option[T] = None, is_log : Option[Boolean] = None, name : String = null, attr : Map[String, String] = null): org.apache.mxnet.Symbol
  /**
  *
  * {{{
  *
  * Draw random samples from a generalized negative binomial distribution according to the
  * input array shape.
  * 
  * Samples are distributed according to a generalized negative binomial distribution parametrized by
  * *mu* (mean) and *alpha* (dispersion). *alpha* is defined as *1/k* where *k* is the failure limit of the
  * number of unsuccessful experiments (generalized to real numbers).
  * Samples will always be returned as a floating point data type.
  * 
  * Example::
  * 
  *    generalized_negative_binomial(mu=2.0, alpha=0.3, data=ones(2,2)) = `[ [ 2.,  1.],
  *                                                                        [ 6.,  4.] ]
  * 
  * 
  * Defined in src/operator/random/sample_op.cc:L284
  * }}}
  * 
  * @param mu		Mean of the negative binomial distribution.
  * @param alpha		Alpha (dispersion) parameter of the negative binomial distribution.
  * @param data		The input
  * @return org.apache.mxnet.Symbol
  */
@Experimental
def generalized_negative_binomial_like[T: SymbolOrScalar : ClassTag] (mu : Option[T] = None, alpha : Option[T] = None, data : Option[T] = None, name : String = null, attr : Map[String, String] = null): org.apache.mxnet.Symbol
  /**
  *
  * {{{
  *
  * Computes the value of the PDF of *sample* of
  * gamma distributions with parameters *alpha* (shape) and *beta* (rate).
  * 
  * *alpha* and *beta* must have the same shape, which must match the leftmost subshape
  * of *sample*.  That is, *sample* can have the same shape as *alpha* and *beta*, in which
  * case the output contains one density per distribution, or *sample* can be a tensor
  * of tensors with that shape, in which case the output is a tensor of densities such that
  * the densities at index *i* in the output are given by the samples at index *i* in *sample*
  * parameterized by the values of *alpha* and *beta* at index *i*.
  * 
  * Examples::
  * 
  *   random_pdf_gamma(sample=`[ [1,2,3,4,5] ], alpha=[5], beta=[1]) =
  *       `[ [0.01532831, 0.09022352, 0.16803136, 0.19536681, 0.17546739] ]
  * 
  *   sample = `[ [1, 2, 3, 4, 5],
  *             [2, 3, 4, 5, 6],
  *             [3, 4, 5, 6, 7] ]
  * 
  *   random_pdf_gamma(sample=sample, alpha=[5,6,7], beta=[1,1,1]) =
  *       `[ [0.01532831, 0.09022352, 0.16803136, 0.19536681, 0.17546739],
  *        [0.03608941, 0.10081882, 0.15629345, 0.17546739, 0.16062315],
  *        [0.05040941, 0.10419563, 0.14622283, 0.16062315, 0.14900276] ]
  * 
  * 
  * Defined in src/operator/random/pdf_op.cc:L301
  * }}}
  * 
  * @param sample		Samples from the distributions.
  * @param alpha		Alpha (shape) parameters of the distributions.
  * @param is_log		If set, compute the density of the log-probability instead of the probability.
  * @param beta		Beta (scale) parameters of the distributions.
  * @return org.apache.mxnet.Symbol
  */
@Experimental
def pdf_gamma[T: SymbolOrScalar : ClassTag] (sample : Option[T] = None, alpha : Option[T] = None, is_log : Option[Boolean] = None, beta : Option[T] = None, name : String = null, attr : Map[String, String] = null): org.apache.mxnet.Symbol
  /**
  *
  * {{{
  *
  * Draw random samples from a normal (Gaussian) distribution.
  * 
  * .. note:: The existing alias ``normal`` is deprecated.
  * 
  * Samples are distributed according to a normal distribution parametrized by *loc* (mean) and *scale*
  * (standard deviation).
  * 
  * Example::
  * 
  *    normal(loc=0, scale=1, shape=(2,2)) = `[ [ 1.89171135, -1.16881478],
  *                                           [-1.23474145,  1.55807114] ]
  * 
  * 
  * Defined in src/operator/random/sample_op.cc:L113
  * }}}
  * 
  * @param mu		Mean of the distribution.
  * @param sigma		Standard deviation of the distribution.
  * @param shape		Shape of the output.
  * @param ctx		Context of output, in format [cpu|gpu|cpu_pinned](n). Only used for imperative calls.
  * @param dtype		DType of the output in case this can't be inferred. Defaults to float32 if not defined (dtype=None).
  * @return org.apache.mxnet.Symbol
  */
@Experimental
def normal[T: SymbolOrScalar : ClassTag] (mu : Option[T] = None, sigma : Option[T] = None, shape : Option[org.apache.mxnet.Shape] = None, ctx : Option[String] = None, dtype : Option[String] = None, name : String = null, attr : Map[String, String] = null): org.apache.mxnet.Symbol
  /**
  *
  * {{{
  *
  * Draw random samples from a generalized negative binomial distribution.
  * 
  * Samples are distributed according to a generalized negative binomial distribution parametrized by
  * *mu* (mean) and *alpha* (dispersion). *alpha* is defined as *1/k* where *k* is the failure limit of the
  * number of unsuccessful experiments (generalized to real numbers).
  * Samples will always be returned as a floating point data type.
  * 
  * Example::
  * 
  *    generalized_negative_binomial(mu=2.0, alpha=0.3, shape=(2,2)) = `[ [ 2.,  1.],
  *                                                                     [ 6.,  4.] ]
  * 
  * 
  * Defined in src/operator/random/sample_op.cc:L179
  * }}}
  * 
  * @param mu		Mean of the negative binomial distribution.
  * @param alpha		Alpha (dispersion) parameter of the negative binomial distribution.
  * @param shape		Shape of the output.
  * @param ctx		Context of output, in format [cpu|gpu|cpu_pinned](n). Only used for imperative calls.
  * @param dtype		DType of the output in case this can't be inferred. Defaults to float32 if not defined (dtype=None).
  * @return org.apache.mxnet.Symbol
  */
@Experimental
def generalized_negative_binomial[T: SymbolOrScalar : ClassTag] (mu : Option[T] = None, alpha : Option[T] = None, shape : Option[org.apache.mxnet.Shape] = None, ctx : Option[String] = None, dtype : Option[String] = None, name : String = null, attr : Map[String, String] = null): org.apache.mxnet.Symbol
  /**
  *
  * {{{
  *
  * Draw random samples from an exponential distribution according to the input array shape.
  * 
  * Samples are distributed according to an exponential distribution parametrized by *lambda* (rate).
  * 
  * Example::
  * 
  *    exponential(lam=4, data=ones(2,2)) = `[ [ 0.0097189 ,  0.08999364],
  *                                          [ 0.04146638,  0.31715935] ]
  * 
  * 
  * Defined in src/operator/random/sample_op.cc:L243
  * }}}
  * 
  * @param lam		Lambda parameter (rate) of the exponential distribution.
  * @param data		The input
  * @return org.apache.mxnet.Symbol
  */
@Experimental
def exponential_like[T: SymbolOrScalar : ClassTag] (lam : Option[T] = None, data : Option[T] = None, name : String = null, attr : Map[String, String] = null): org.apache.mxnet.Symbol
  /**
  *
  * {{{
  *
  * Draw random samples from a Poisson distribution.
  * 
  * Samples are distributed according to a Poisson distribution parametrized by *lambda* (rate).
  * Samples will always be returned as a floating point data type.
  * 
  * Example::
  * 
  *    poisson(lam=4, shape=(2,2)) = `[ [ 5.,  2.],
  *                                   [ 4.,  6.] ]
  * 
  * 
  * Defined in src/operator/random/sample_op.cc:L150
  * }}}
  * 
  * @param lam		Lambda parameter (rate) of the Poisson distribution.
  * @param shape		Shape of the output.
  * @param ctx		Context of output, in format [cpu|gpu|cpu_pinned](n). Only used for imperative calls.
  * @param dtype		DType of the output in case this can't be inferred. Defaults to float32 if not defined (dtype=None).
  * @return org.apache.mxnet.Symbol
  */
@Experimental
def poisson[T: SymbolOrScalar : ClassTag] (lam : Option[T] = None, shape : Option[org.apache.mxnet.Shape] = None, ctx : Option[String] = None, dtype : Option[String] = None, name : String = null, attr : Map[String, String] = null): org.apache.mxnet.Symbol
  /**
  *
  * {{{
  *
  * Draw random samples from a uniform distribution.
  * 
  * .. note:: The existing alias ``uniform`` is deprecated.
  * 
  * Samples are uniformly distributed over the half-open interval *[low, high)*
  * (includes *low*, but excludes *high*).
  * 
  * Example::
  * 
  *    uniform(low=0, high=1, shape=(2,2)) = `[ [ 0.60276335,  0.85794562],
  *                                           [ 0.54488319,  0.84725171] ]
  * 
  * 
  * 
  * Defined in src/operator/random/sample_op.cc:L96
  * }}}
  * 
  * @param low		Lower bound of the distribution.
  * @param high		Upper bound of the distribution.
  * @param shape		Shape of the output.
  * @param ctx		Context of output, in format [cpu|gpu|cpu_pinned](n). Only used for imperative calls.
  * @param dtype		DType of the output in case this can't be inferred. Defaults to float32 if not defined (dtype=None).
  * @return org.apache.mxnet.Symbol
  */
@Experimental
def uniform[T: SymbolOrScalar : ClassTag] (low : Option[T] = None, high : Option[T] = None, shape : Option[org.apache.mxnet.Shape] = None, ctx : Option[String] = None, dtype : Option[String] = None, name : String = null, attr : Map[String, String] = null): org.apache.mxnet.Symbol
  /**
  *
  * {{{
  *
  * Draw random samples from a gamma distribution according to the input array shape.
  * 
  * Samples are distributed according to a gamma distribution parametrized by *alpha* (shape) and *beta* (scale).
  * 
  * Example::
  * 
  *    gamma(alpha=9, beta=0.5, data=ones(2,2)) = `[ [ 7.10486984,  3.37695289],
  *                                                [ 3.91697288,  3.65933681] ]
  * 
  * 
  * Defined in src/operator/random/sample_op.cc:L232
  * }}}
  * 
  * @param alpha		Alpha parameter (shape) of the gamma distribution.
  * @param beta		Beta parameter (scale) of the gamma distribution.
  * @param data		The input
  * @return org.apache.mxnet.Symbol
  */
@Experimental
def gamma_like[T: SymbolOrScalar : ClassTag] (alpha : Option[T] = None, beta : Option[T] = None, data : Option[T] = None, name : String = null, attr : Map[String, String] = null): org.apache.mxnet.Symbol
  /**
  *
  * {{{
  *
  * Computes the value of the PDF of *sample* of
  * Dirichlet distributions with parameter *alpha*.
  * 
  * The shape of *alpha* must match the leftmost subshape of *sample*.  That is, *sample*
  * can have the same shape as *alpha*, in which case the output contains one density per
  * distribution, or *sample* can be a tensor of tensors with that shape, in which case
  * the output is a tensor of densities such that the densities at index *i* in the output
  * are given by the samples at index *i* in *sample* parameterized by the value of *alpha*
  * at index *i*.
  * 
  * Examples::
  * 
  *     random_pdf_dirichlet(sample=`[ [1,2],[2,3],[3,4] ], alpha=[2.5, 2.5]) =
  *         [38.413498, 199.60245, 564.56085]
  * 
  *     sample = `[ `[ [1, 2, 3], [10, 20, 30], [100, 200, 300] ],
  *               `[ [0.1, 0.2, 0.3], [0.01, 0.02, 0.03], [0.001, 0.002, 0.003] ] ]
  * 
  *     random_pdf_dirichlet(sample=sample, alpha=[0.1, 0.4, 0.9]) =
  *         `[ [2.3257459e-02, 5.8420084e-04, 1.4674458e-05],
  *          [9.2589635e-01, 3.6860607e+01, 1.4674468e+03] ]
  * 
  * 
  * Defined in src/operator/random/pdf_op.cc:L315
  * }}}
  * 
  * @param sample		Samples from the distributions.
  * @param alpha		Concentration parameters of the distributions.
  * @param is_log		If set, compute the density of the log-probability instead of the probability.
  * @return org.apache.mxnet.Symbol
  */
@Experimental
def pdf_dirichlet[T: SymbolOrScalar : ClassTag] (sample : Option[T] = None, alpha : Option[T] = None, is_log : Option[Boolean] = None, name : String = null, attr : Map[String, String] = null): org.apache.mxnet.Symbol
  /**
  *
  * {{{
  *
  * Draw random samples from a Poisson distribution according to the input array shape.
  * 
  * Samples are distributed according to a Poisson distribution parametrized by *lambda* (rate).
  * Samples will always be returned as a floating point data type.
  * 
  * Example::
  * 
  *    poisson(lam=4, data=ones(2,2)) = `[ [ 5.,  2.],
  *                                      [ 4.,  6.] ]
  * 
  * 
  * Defined in src/operator/random/sample_op.cc:L255
  * }}}
  * 
  * @param lam		Lambda parameter (rate) of the Poisson distribution.
  * @param data		The input
  * @return org.apache.mxnet.Symbol
  */
@Experimental
def poisson_like[T: SymbolOrScalar : ClassTag] (lam : Option[T] = None, data : Option[T] = None, name : String = null, attr : Map[String, String] = null): org.apache.mxnet.Symbol
  /**
  *
  * {{{
  *
  * Computes the value of the PDF of *sample* of
  * generalized negative binomial distributions with parameters *mu* (mean)
  * and *alpha* (dispersion).  This can be understood as a reparameterization of
  * the negative binomial, where *k* = *1 / alpha* and *p* = *1 / (mu \* alpha + 1)*.
  * 
  * *mu* and *alpha* must have the same shape, which must match the leftmost subshape
  * of *sample*.  That is, *sample* can have the same shape as *mu* and *alpha*, in which
  * case the output contains one density per distribution, or *sample* can be a tensor
  * of tensors with that shape, in which case the output is a tensor of densities such that
  * the densities at index *i* in the output are given by the samples at index *i* in *sample*
  * parameterized by the values of *mu* and *alpha* at index *i*.
  * 
  * Examples::
  * 
  *     random_pdf_generalized_negative_binomial(sample=`[ [1, 2, 3, 4] ], alpha=[1], mu=[1]) =
  *         `[ [0.25, 0.125, 0.0625, 0.03125] ]
  * 
  *     sample = `[ [1,2,3,4],
  *               [1,2,3,4] ]
  *     random_pdf_generalized_negative_binomial(sample=sample, alpha=[1, 0.6666], mu=[1, 1.5]) =
  *         `[ [0.25,       0.125,      0.0625,     0.03125   ],
  *          [0.26517063, 0.16573331, 0.09667706, 0.05437994] ]
  * 
  * 
  * Defined in src/operator/random/pdf_op.cc:L311
  * }}}
  * 
  * @param sample		Samples from the distributions.
  * @param mu		Means of the distributions.
  * @param is_log		If set, compute the density of the log-probability instead of the probability.
  * @param alpha		Alpha (dispersion) parameters of the distributions.
  * @return org.apache.mxnet.Symbol
  */
@Experimental
def pdf_generalized_negative_binomial[T: SymbolOrScalar : ClassTag] (sample : Option[T] = None, mu : Option[T] = None, is_log : Option[Boolean] = None, alpha : Option[T] = None, name : String = null, attr : Map[String, String] = null): org.apache.mxnet.Symbol
  /**
  *
  * {{{
  *
  * Draw random samples from a negative binomial distribution.
  * 
  * Samples are distributed according to a negative binomial distribution parametrized by
  * *k* (limit of unsuccessful experiments) and *p* (failure probability in each experiment).
  * Samples will always be returned as a floating point data type.
  * 
  * Example::
  * 
  *    negative_binomial(k=3, p=0.4, shape=(2,2)) = `[ [ 4.,  7.],
  *                                                  [ 2.,  5.] ]
  * 
  * 
  * Defined in src/operator/random/sample_op.cc:L164
  * }}}
  * 
  * @param k		Limit of unsuccessful experiments.
  * @param p		Failure probability in each experiment.
  * @param shape		Shape of the output.
  * @param ctx		Context of output, in format [cpu|gpu|cpu_pinned](n). Only used for imperative calls.
  * @param dtype		DType of the output in case this can't be inferred. Defaults to float32 if not defined (dtype=None).
  * @return org.apache.mxnet.Symbol
  */
@Experimental
def negative_binomial[T: SymbolOrScalar : ClassTag] (k : Option[T] = None, p : Option[T] = None, shape : Option[org.apache.mxnet.Shape] = None, ctx : Option[String] = None, dtype : Option[String] = None, name : String = null, attr : Map[String, String] = null): org.apache.mxnet.Symbol
  /**
  *
  * {{{
  *
  * Draw random samples from a negative binomial distribution according to the input array shape.
  * 
  * Samples are distributed according to a negative binomial distribution parametrized by
  * *k* (limit of unsuccessful experiments) and *p* (failure probability in each experiment).
  * Samples will always be returned as a floating point data type.
  * 
  * Example::
  * 
  *    negative_binomial(k=3, p=0.4, data=ones(2,2)) = `[ [ 4.,  7.],
  *                                                     [ 2.,  5.] ]
  * 
  * 
  * Defined in src/operator/random/sample_op.cc:L268
  * }}}
  * 
  * @param k		Limit of unsuccessful experiments.
  * @param p		Failure probability in each experiment.
  * @param data		The input
  * @return org.apache.mxnet.Symbol
  */
@Experimental
def negative_binomial_like[T: SymbolOrScalar : ClassTag] (k : Option[T] = None, p : Option[T] = None, data : Option[T] = None, name : String = null, attr : Map[String, String] = null): org.apache.mxnet.Symbol
  /**
  *
  * {{{
  *
  * Concurrent sampling from multiple multinomial distributions.
  * 
  * *data* is an *n* dimensional array whose last dimension has length *k*, where
  * *k* is the number of possible outcomes of each multinomial distribution. This
  * operator will draw *shape* samples from each distribution. If shape is empty
  * one sample will be drawn from each distribution.
  * 
  * If *get_prob* is true, a second array containing log likelihood of the drawn
  * samples will also be returned. This is usually used for reinforcement learning
  * where you can provide reward as head gradient for this array to estimate
  * gradient.
  * 
  * Note that the input distribution must be normalized, i.e. *data* must sum to
  * 1 along its last axis.
  * 
  * Examples::
  * 
  *    probs = `[ [0, 0.1, 0.2, 0.3, 0.4], [0.4, 0.3, 0.2, 0.1, 0] ]
  * 
  *    // Draw a single sample for each distribution
  *    sample_multinomial(probs) = [3, 0]
  * 
  *    // Draw a vector containing two samples for each distribution
  *    sample_multinomial(probs, shape=(2)) = `[ [4, 2],
  *                                            [0, 0] ]
  * 
  *    // requests log likelihood
  *    sample_multinomial(probs, get_prob=True) = [2, 1], [0.2, 0.3]
  * }}}
  * 
  * @param data		Distribution probabilities. Must sum to one on the last axis.
  * @param shape		Shape to be sampled from each random distribution.
  * @param get_prob		Whether to also return the log probability of sampled result. This is usually used for differentiating through stochastic variables, e.g. in reinforcement learning.
  * @param dtype		DType of the output in case this can't be inferred.
  * @return org.apache.mxnet.Symbol
  */
@Experimental
def multinomial[T: SymbolOrScalar : ClassTag] (data : Option[T] = None, shape : Option[org.apache.mxnet.Shape] = None, get_prob : Option[Boolean] = None, dtype : Option[String] = None, name : String = null, attr : Map[String, String] = null): org.apache.mxnet.Symbol
  /**
  *
  * {{{
  *
  * Draw random samples from an exponential distribution.
  * 
  * Samples are distributed according to an exponential distribution parametrized by *lambda* (rate).
  * 
  * Example::
  * 
  *    exponential(lam=4, shape=(2,2)) = `[ [ 0.0097189 ,  0.08999364],
  *                                       [ 0.04146638,  0.31715935] ]
  * 
  * 
  * Defined in src/operator/random/sample_op.cc:L137
  * }}}
  * 
  * @param lam		Lambda parameter (rate) of the exponential distribution.
  * @param shape		Shape of the output.
  * @param ctx		Context of output, in format [cpu|gpu|cpu_pinned](n). Only used for imperative calls.
  * @param dtype		DType of the output in case this can't be inferred. Defaults to float32 if not defined (dtype=None).
  * @return org.apache.mxnet.Symbol
  */
@Experimental
def exponential[T: SymbolOrScalar : ClassTag] (lam : Option[T] = None, shape : Option[org.apache.mxnet.Shape] = None, ctx : Option[String] = None, dtype : Option[String] = None, name : String = null, attr : Map[String, String] = null): org.apache.mxnet.Symbol
  /**
  *
  * {{{
  *
  * Computes the value of the PDF of samples of
  * negative binomial distributions with parameters *k* (failure limit) and *p* (failure probability).
  * 
  * *k* and *p* must have the same shape, which must match the leftmost subshape
  * of *sample*.  That is, *sample* can have the same shape as *k* and *p*, in which
  * case the output contains one density per distribution, or *sample* can be a tensor
  * of tensors with that shape, in which case the output is a tensor of densities such that
  * the densities at index *i* in the output are given by the samples at index *i* in *sample*
  * parameterized by the values of *k* and *p* at index *i*.
  * 
  * Examples::
  * 
  *     random_pdf_negative_binomial(sample=`[ [1,2,3,4] ], k=[1], p=a[0.5]) =
  *         `[ [0.25, 0.125, 0.0625, 0.03125] ]
  * 
  *     # Note that k may be real-valued
  *     sample = `[ [1,2,3,4],
  *               [1,2,3,4] ]
  *     random_pdf_negative_binomial(sample=sample, k=[1, 1.5], p=[0.5, 0.5]) =
  *         `[ [0.25,       0.125,      0.0625,     0.03125   ],
  *          [0.26516506, 0.16572815, 0.09667476, 0.05437956] ]
  * 
  * 
  * Defined in src/operator/random/pdf_op.cc:L308
  * }}}
  * 
  * @param sample		Samples from the distributions.
  * @param k		Limits of unsuccessful experiments.
  * @param is_log		If set, compute the density of the log-probability instead of the probability.
  * @param p		Failure probabilities in each experiment.
  * @return org.apache.mxnet.Symbol
  */
@Experimental
def pdf_negative_binomial[T: SymbolOrScalar : ClassTag] (sample : Option[T] = None, k : Option[T] = None, is_log : Option[Boolean] = None, p : Option[T] = None, name : String = null, attr : Map[String, String] = null): org.apache.mxnet.Symbol
  /**
  *
  * {{{
  *
  * Draw random samples from a gamma distribution.
  * 
  * Samples are distributed according to a gamma distribution parametrized by *alpha* (shape) and *beta* (scale).
  * 
  * Example::
  * 
  *    gamma(alpha=9, beta=0.5, shape=(2,2)) = `[ [ 7.10486984,  3.37695289],
  *                                             [ 3.91697288,  3.65933681] ]
  * 
  * 
  * Defined in src/operator/random/sample_op.cc:L125
  * }}}
  * 
  * @param alpha		Alpha parameter (shape) of the gamma distribution.
  * @param beta		Beta parameter (scale) of the gamma distribution.
  * @param shape		Shape of the output.
  * @param ctx		Context of output, in format [cpu|gpu|cpu_pinned](n). Only used for imperative calls.
  * @param dtype		DType of the output in case this can't be inferred. Defaults to float32 if not defined (dtype=None).
  * @return org.apache.mxnet.Symbol
  */
@Experimental
def gamma[T: SymbolOrScalar : ClassTag] (alpha : Option[T] = None, beta : Option[T] = None, shape : Option[org.apache.mxnet.Shape] = None, ctx : Option[String] = None, dtype : Option[String] = None, name : String = null, attr : Map[String, String] = null): org.apache.mxnet.Symbol
  /**
  *
  * {{{
  *
  * Draw random samples from a discrete uniform distribution.
  * 
  * Samples are uniformly distributed over the half-open interval *[low, high)*
  * (includes *low*, but excludes *high*).
  * 
  * Example::
  * 
  *    randint(low=0, high=5, shape=(2,2)) = `[ [ 0,  2],
  *                                           [ 3,  1] ]
  * 
  * 
  * 
  * Defined in src/operator/random/sample_op.cc:L194
  * }}}
  * 
  * @param low		Lower bound of the distribution.
  * @param high		Upper bound of the distribution.
  * @param shape		Shape of the output.
  * @param ctx		Context of output, in format [cpu|gpu|cpu_pinned](n). Only used for imperative calls.
  * @param dtype		DType of the output in case this can't be inferred. Defaults to int32 if not defined (dtype=None).
  * @return org.apache.mxnet.Symbol
  */
@Experimental
def randint[T: SymbolOrScalar : ClassTag] (low : Long, high : Long, shape : Option[org.apache.mxnet.Shape] = None, ctx : Option[String] = None, dtype : Option[String] = None, name : String = null, attr : Map[String, String] = null): org.apache.mxnet.Symbol
  /**
  *
  * {{{
  *
  * Draw random samples from a normal (Gaussian) distribution according to the input array shape.
  * 
  * Samples are distributed according to a normal distribution parametrized by *loc* (mean) and *scale*
  * (standard deviation).
  * 
  * Example::
  * 
  *    normal(loc=0, scale=1, data=ones(2,2)) = `[ [ 1.89171135, -1.16881478],
  *                                              [-1.23474145,  1.55807114] ]
  * 
  * 
  * Defined in src/operator/random/sample_op.cc:L221
  * }}}
  * 
  * @param mu		Mean of the distribution.
  * @param sigma		Standard deviation of the distribution.
  * @param data		The input
  * @return org.apache.mxnet.Symbol
  */
@Experimental
def normal_like[T: SymbolOrScalar : ClassTag] (mu : Option[T] = None, sigma : Option[T] = None, data : Option[T] = None, name : String = null, attr : Map[String, String] = null): org.apache.mxnet.Symbol
  /**
  *
  * {{{
  *
  * Computes the value of the PDF of *sample* of
  * normal distributions with parameters *mu* (mean) and *sigma* (standard deviation).
  * 
  * *mu* and *sigma* must have the same shape, which must match the leftmost subshape
  * of *sample*.  That is, *sample* can have the same shape as *mu* and *sigma*, in which
  * case the output contains one density per distribution, or *sample* can be a tensor
  * of tensors with that shape, in which case the output is a tensor of densities such that
  * the densities at index *i* in the output are given by the samples at index *i* in *sample*
  * parameterized by the values of *mu* and *sigma* at index *i*.
  * 
  * Examples::
  * 
  *     sample = `[ [-2, -1, 0, 1, 2] ]
  *     random_pdf_normal(sample=sample, mu=[0], sigma=[1]) =
  *         `[ [0.05399097, 0.24197073, 0.3989423, 0.24197073, 0.05399097] ]
  * 
  *     random_pdf_normal(sample=sample*2, mu=[0,0], sigma=[1,2]) =
  *         `[ [0.05399097, 0.24197073, 0.3989423,  0.24197073, 0.05399097],
  *          [0.12098537, 0.17603266, 0.19947115, 0.17603266, 0.12098537] ]
  * 
  * 
  * Defined in src/operator/random/pdf_op.cc:L299
  * }}}
  * 
  * @param sample		Samples from the distributions.
  * @param mu		Means of the distributions.
  * @param is_log		If set, compute the density of the log-probability instead of the probability.
  * @param sigma		Standard deviations of the distributions.
  * @return org.apache.mxnet.Symbol
  */
@Experimental
def pdf_normal[T: SymbolOrScalar : ClassTag] (sample : Option[T] = None, mu : Option[T] = None, is_log : Option[Boolean] = None, sigma : Option[T] = None, name : String = null, attr : Map[String, String] = null): org.apache.mxnet.Symbol
  /**
  *
  * {{{
  *
  * Computes the value of the PDF of *sample* of
  * Poisson distributions with parameters *lam* (rate).
  * 
  * The shape of *lam* must match the leftmost subshape of *sample*.  That is, *sample*
  * can have the same shape as *lam*, in which case the output contains one density per
  * distribution, or *sample* can be a tensor of tensors with that shape, in which case
  * the output is a tensor of densities such that the densities at index *i* in the output
  * are given by the samples at index *i* in *sample* parameterized by the value of *lam*
  * at index *i*.
  * 
  * Examples::
  * 
  *     random_pdf_poisson(sample=`[ [0,1,2,3] ], lam=[1]) =
  *         `[ [0.36787945, 0.36787945, 0.18393973, 0.06131324] ]
  * 
  *     sample = `[ [0,1,2,3],
  *               [0,1,2,3],
  *               [0,1,2,3] ]
  * 
  *     random_pdf_poisson(sample=sample, lam=[1,2,3]) =
  *         `[ [0.36787945, 0.36787945, 0.18393973, 0.06131324],
  *          [0.13533528, 0.27067056, 0.27067056, 0.18044704],
  *          [0.04978707, 0.14936121, 0.22404182, 0.22404182] ]
  * 
  * 
  * Defined in src/operator/random/pdf_op.cc:L306
  * }}}
  * 
  * @param sample		Samples from the distributions.
  * @param lam		Lambda (rate) parameters of the distributions.
  * @param is_log		If set, compute the density of the log-probability instead of the probability.
  * @return org.apache.mxnet.Symbol
  */
@Experimental
def pdf_poisson[T: SymbolOrScalar : ClassTag] (sample : Option[T] = None, lam : Option[T] = None, is_log : Option[Boolean] = None, name : String = null, attr : Map[String, String] = null): org.apache.mxnet.Symbol
  /**
  *
  * {{{
  *
  * Draw random samples from a uniform distribution according to the input array shape.
  * 
  * Samples are uniformly distributed over the half-open interval *[low, high)*
  * (includes *low*, but excludes *high*).
  * 
  * Example::
  * 
  *    uniform(low=0, high=1, data=ones(2,2)) = `[ [ 0.60276335,  0.85794562],
  *                                              [ 0.54488319,  0.84725171] ]
  * 
  * 
  * 
  * Defined in src/operator/random/sample_op.cc:L209
  * }}}
  * 
  * @param low		Lower bound of the distribution.
  * @param high		Upper bound of the distribution.
  * @param data		The input
  * @return org.apache.mxnet.Symbol
  */
@Experimental
def uniform_like[T: SymbolOrScalar : ClassTag] (low : Option[T] = None, high : Option[T] = None, data : Option[T] = None, name : String = null, attr : Map[String, String] = null): org.apache.mxnet.Symbol
}

