/*
 * Decompiled with CFR 0.152.
 */
package com.ctrip.framework.apollo.kubernetes;

import com.ctrip.framework.apollo.core.utils.StringUtils;
import com.ctrip.framework.apollo.exceptions.ApolloConfigException;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;
import io.kubernetes.client.openapi.ApiClient;
import io.kubernetes.client.openapi.ApiException;
import io.kubernetes.client.openapi.apis.CoreV1Api;
import io.kubernetes.client.openapi.models.V1ConfigMap;
import io.kubernetes.client.openapi.models.V1ObjectMeta;
import io.kubernetes.client.openapi.models.V1Pod;
import io.kubernetes.client.openapi.models.V1PodList;
import io.kubernetes.client.util.Config;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

@Service
public class KubernetesManager {
    private static final Logger logger = LoggerFactory.getLogger(KubernetesManager.class);
    private static final String RUNNING_POD_FIELD_SELECTOR = "status.phase=Running";
    private static final int MAX_SEARCH_NUM = 100;
    private ApiClient client;
    private CoreV1Api coreV1Api;
    private int propertyKubernetesMaxWritePods = 3;
    private String localPodName = System.getenv("HOSTNAME");

    public KubernetesManager() {
        try {
            this.client = Config.defaultClient();
            this.coreV1Api = new CoreV1Api(this.client);
        }
        catch (Exception e) {
            String errorMessage = "Failed to initialize Kubernetes client: " + e.getMessage();
            logger.error(errorMessage, (Throwable)e);
            throw new RuntimeException(errorMessage, e);
        }
    }

    @VisibleForTesting
    public KubernetesManager(CoreV1Api coreV1Api, String localPodName, int propertyKubernetesMaxWritePods) {
        this.coreV1Api = coreV1Api;
        this.localPodName = localPodName;
        this.propertyKubernetesMaxWritePods = propertyKubernetesMaxWritePods;
    }

    private V1ConfigMap buildConfigMap(String name, String namespace, Map<String, String> data) {
        V1ObjectMeta metadata = new V1ObjectMeta().name(name).namespace(namespace);
        return new V1ConfigMap().apiVersion("v1").kind("ConfigMap").metadata(metadata).data(data);
    }

    public String createConfigMap(String k8sNamespace, String name, Map<String, String> data) {
        if (StringUtils.isEmpty((String)k8sNamespace) || StringUtils.isEmpty((String)name)) {
            logger.error("create configmap failed due to null or empty parameter: k8sNamespace={}, name={}", (Object)k8sNamespace, (Object)name);
            return null;
        }
        V1ConfigMap configMap = this.buildConfigMap(name, k8sNamespace, data);
        try {
            this.coreV1Api.createNamespacedConfigMap(k8sNamespace, configMap, null, null, null, null);
            logger.info("ConfigMap created successfully: name: {}, namespace: {}", (Object)name, (Object)k8sNamespace);
            return name;
        }
        catch (Exception e) {
            logger.error("Failed to create ConfigMap: {}", (Object)e.getMessage(), (Object)e);
            throw new RuntimeException("Failed to create ConfigMap: " + e.getMessage(), e);
        }
    }

    public String getValueFromConfigMap(String k8sNamespace, String name, String key) {
        if (StringUtils.isEmpty((String)k8sNamespace) || StringUtils.isEmpty((String)name) || StringUtils.isEmpty((String)key)) {
            logger.error("Parameters can not be null or empty: k8sNamespace={}, name={}", (Object)k8sNamespace, (Object)name);
            return null;
        }
        try {
            V1ConfigMap configMap = this.coreV1Api.readNamespacedConfigMap(name, k8sNamespace, null);
            if (!Objects.requireNonNull(configMap.getData()).containsKey(key)) {
                logger.error("Specified key not found in ConfigMap: {}, k8sNamespace: {}, name: {}", new Object[]{name, k8sNamespace, name});
            }
            return (String)configMap.getData().get(key);
        }
        catch (Exception e) {
            logger.error("Error occurred while getting value from ConfigMap: {}", (Object)e.getMessage(), (Object)e);
            return null;
        }
    }

    public boolean updateConfigMap(String k8sNamespace, String name, Map<String, String> data) throws ApiException {
        if (StringUtils.isEmpty((String)k8sNamespace) || StringUtils.isEmpty((String)name)) {
            logger.error("Parameters can not be null or empty: k8sNamespace={}, name={}", (Object)k8sNamespace, (Object)name);
            return false;
        }
        if (!this.isWritePod(k8sNamespace)) {
            return true;
        }
        int maxRetries = 5;
        int retryCount = 0;
        long waitTime = 100L;
        while (retryCount < maxRetries) {
            try {
                V1ConfigMap configmap = this.coreV1Api.readNamespacedConfigMap(name, k8sNamespace, null);
                HashMap<String, String> existingData = configmap.getData();
                if (existingData == null) {
                    existingData = new HashMap<String, String>();
                }
                HashMap<String, String> finalExistingData = existingData;
                boolean containsEntry = data.entrySet().stream().allMatch(entry -> ((String)entry.getValue()).equals(finalExistingData.get(entry.getKey())));
                if (containsEntry) {
                    logger.info("Data is identical or already contains the entry, no update needed.");
                    return true;
                }
                existingData.putAll(data);
                configmap.setData(existingData);
                this.coreV1Api.replaceNamespacedConfigMap(name, k8sNamespace, configmap, null, null, null, null);
                return true;
            }
            catch (ApiException e) {
                if (e.getCode() == 409) {
                    logger.warn("Conflict occurred, retrying... ({})", (Object)(++retryCount));
                    try {
                        TimeUnit.MILLISECONDS.sleep((long)((double)waitTime * (0.9 + Math.random() * 0.2)));
                    }
                    catch (InterruptedException ie) {
                        Thread.currentThread().interrupt();
                    }
                    waitTime = Math.min(waitTime * 2L, 1000L);
                    continue;
                }
                logger.error("Error updating ConfigMap: {}", (Object)e.getMessage(), (Object)e);
                throw e;
            }
        }
        String errorMessage = String.format("Failed to update ConfigMap after %d retries: k8sNamespace=%s, name=%s", maxRetries, k8sNamespace, name);
        logger.error(errorMessage);
        throw new ApolloConfigException(errorMessage);
    }

    public boolean checkConfigMapExist(String k8sNamespace, String configMapName) {
        if (StringUtils.isEmpty((String)k8sNamespace) || StringUtils.isEmpty((String)configMapName)) {
            logger.error("Parameters can not be null or empty: k8sNamespace={}, configMapName={}", (Object)k8sNamespace, (Object)configMapName);
            return false;
        }
        try {
            logger.info("Check whether ConfigMap exists, configMapName: {}", (Object)configMapName);
            this.coreV1Api.readNamespacedConfigMap(configMapName, k8sNamespace, null);
            return true;
        }
        catch (Exception e) {
            logger.info("ConfigMap not existence");
            return false;
        }
    }

    private boolean isWritePod(String k8sNamespace) {
        try {
            if (Strings.isNullOrEmpty((String)this.localPodName)) {
                return true;
            }
            V1Pod localPod = this.coreV1Api.readNamespacedPod(this.localPodName, k8sNamespace, null);
            V1ObjectMeta localMetadata = localPod.getMetadata();
            if (localMetadata == null || localMetadata.getLabels() == null) {
                return true;
            }
            String appName = (String)localMetadata.getLabels().get("app");
            String labelSelector = "app=" + appName;
            V1PodList v1PodList = this.coreV1Api.listNamespacedPod(k8sNamespace, null, null, null, RUNNING_POD_FIELD_SELECTOR, labelSelector, Integer.valueOf(100), null, null, null, null);
            return v1PodList.getItems().stream().map(V1Pod::getMetadata).filter(Objects::nonNull).filter(metadata -> metadata.getCreationTimestamp() != null).sorted(Comparator.comparing(V1ObjectMeta::getCreationTimestamp)).map(V1ObjectMeta::getName).limit(this.propertyKubernetesMaxWritePods).anyMatch(this.localPodName::equals);
        }
        catch (Exception e) {
            logger.info("Error determining write pod eligibility:{}", (Object)e.getMessage(), (Object)e);
            return true;
        }
    }
}

