package com.aliyun.core.utils;

import java.io.ByteArrayInputStream;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.Base64;

public final class BinaryUtils {
    private static final byte[] EMPTY_BYTE_ARRAY = new byte[0];
    private static final char[] HEX_DIGITS = {'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D',
            'E', 'F'};

    private BinaryUtils() {
    }

    public static String toBase64(byte[] data) {
        return data == null ? null : new String(toBase64Bytes(data), StandardCharsets.UTF_8);
    }

    public static byte[] toBase64Bytes(byte[] data) {
        return data == null ? null : Base64.getEncoder().encode(data);
    }

    public static byte[] fromBase64(String b64Data) {
        return b64Data == null ? null : Base64.getDecoder().decode(b64Data);
    }

    public static byte[] fromBase64Bytes(byte[] b64Data) {
        return b64Data == null ? null : Base64.getDecoder().decode(b64Data);
    }

    public static ByteArrayInputStream toStream(ByteBuffer byteBuffer) {
        if (byteBuffer == null) {
            return new ByteArrayInputStream(new byte[0]);
        }
        return new ByteArrayInputStream(copyBytesFrom(byteBuffer));
    }

    public static byte[] copyAllBytesFrom(ByteBuffer bb) {
        if (bb == null) {
            return null;
        }

        if (bb.hasArray()) {
            return Arrays.copyOfRange(
                    bb.array(),
                    bb.arrayOffset(),
                    bb.arrayOffset() + bb.limit());
        }

        ByteBuffer copy = bb.asReadOnlyBuffer();
        copy.rewind();

        byte[] dst = new byte[copy.remaining()];
        copy.get(dst);
        return dst;
    }

    public static byte[] copyRemainingBytesFrom(ByteBuffer bb) {
        if (bb == null) {
            return null;
        }

        if (!bb.hasRemaining()) {
            return EMPTY_BYTE_ARRAY;
        }

        if (bb.hasArray()) {
            int endIdx = bb.arrayOffset() + bb.limit();
            int startIdx = endIdx - bb.remaining();
            return Arrays.copyOfRange(bb.array(), startIdx, endIdx);
        }

        ByteBuffer copy = bb.asReadOnlyBuffer();

        byte[] dst = new byte[copy.remaining()];
        copy.get(dst);

        return dst;
    }

    public static byte[] copyBytesFrom(ByteBuffer bb) {
        if (bb == null) {
            return null;
        }

        if (bb.hasArray()) {
            return Arrays.copyOfRange(
                    bb.array(),
                    bb.arrayOffset() + bb.position(),
                    bb.arrayOffset() + bb.limit());
        }

        byte[] dst = new byte[bb.remaining()];
        bb.asReadOnlyBuffer().get(dst);
        return dst;
    }

    public static byte[] calculateMd5(byte[] binaryData) {
        MessageDigest messageDigest = null;
        try {
            messageDigest = MessageDigest.getInstance("MD5");
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("MD5 algorithm not found.");
        }
        messageDigest.update(binaryData);
        return messageDigest.digest();
    }

    public static String encodeMD5(byte[] binaryData) {
        byte[] md5Bytes = calculateMd5(binaryData);
        int len = md5Bytes.length;
        char buf[] = new char[len * 2];
        for (int i = 0; i < len; i++) {
            buf[i * 2] = HEX_DIGITS[(md5Bytes[i] >>> 4) & 0x0f];
            buf[i * 2 + 1] = HEX_DIGITS[md5Bytes[i] & 0x0f];
        }
        return new String(buf);
    }

}
