//
// ooo-util - a place for OOO utilities
// Copyright (C) 2011 Three Rings Design, Inc., All Rights Reserved
// http://github.com/threerings/ooo-util/blob/master/LICENSE

package com.threerings.util;

import static com.threerings.util.Log.log;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.PropertyResourceBundle;
import java.util.ResourceBundle;

import com.samskivert.util.StringUtil;
import com.threerings.resource.ResourceManager;

/**
 * The message manager provides a thin wrapper around Java's built-in localization support,
 * supporting a policy of dividing up localization resources into logical units, all of the
 * translations for which are contained in a single messages file.
 *
 * <p> The message manager assumes that the locale remains constant for the duration of its
 * operation. If the locale were to change during the operation of the client, a call to
 * {@link #setLocale} should be made to inform the message manager of the new locale (which will
 * clear the message bundle cache).
 */
public class MessageManager
{
    private static final String PROPERTIES = "properties";

	/**
     * The name of the global resource bundle (which other bundles revert to if they can't locate
     * a message within themselves). It must be named <code>global.properties</code> and live at
     * the top of the bundle hierarchy.
     */
    public static final String GLOBAL_BUNDLE = "global";

    /**
     * Constructs a message manager with the supplied resource prefix and the default locale. The
     * prefix will be prepended to the path of all resource bundles prior to their resolution. For
     * example, if a prefix of <code>rsrc.messages</code> was provided and a message bundle with
     * the name <code>game.chess</code> was later requested, the message manager would attempt to
     * load a resource bundle with the path <code>rsrc.messages.game.chess</code> and would
     * eventually search for a file in the classpath with the path
     * <code>rsrc/messages/game/chess.properties</code>.
     *
     * <p> See the documentation for {@link ResourceBundle#getBundle(String,Locale,ClassLoader)}
     * for a more detailed explanation of how resource bundle paths are resolved.
     */
    public MessageManager (String resourcePrefix)
    {
    	this(resourcePrefix,null);
    }
    
    public MessageManager (String resourcePrefix,ResourceManager rsrcMgr)
    {
    	_rsrcMgr = rsrcMgr;
        // keep the prefix
        _prefix = resourcePrefix;

        // use the default locale
        _locale = Locale.getDefault();
        log.debug("Using locale: " + _locale + ".");

        // make sure the prefix ends with a dot
        if (!_prefix.endsWith(".")) {
            _prefix += ".";
        }

        // load up the global bundle
        _global = getBundle(GLOBAL_BUNDLE);
        
    }

    /**
     * Get the locale that is being used to translate messages. This may be useful if using
     * standard translations, for example new SimpleDateFormat("EEEE", getLocale()) to get the
     * name of a weekday that matches the language being used for all other client translations.
     */
    public Locale getLocale ()
    {
        return _locale;
    }

    /**
     * Sets the locale to the specified locale. Subsequent message bundles fetched via the message
     * manager will use the new locale. The message bundle cache will also be cleared.
     */
    public void setLocale (Locale locale)
    {
        setLocale(locale, false);
    }

    /**
     * Sets the locale to the specified locale. Subsequent message bundles fetched via the message
     * manager will use the new locale. The message bundle cache will also be cleared.
     *
     * @param updateGlobal set to true if you want the global bundle reloaded in the new locale
     */
    public void setLocale (Locale locale, boolean updateGlobal)
    {
        _locale = locale;
        _cache.clear();

        if (updateGlobal) {
            _global = getBundle(GLOBAL_BUNDLE);
        }
    }

    /**
     * Sets the appropriate resource prefix for where to find subsequent message bundles.
     */
    public void setPrefix (String resourcePrefix)
    {
        _prefix = resourcePrefix;

        // Need to reget the global bundle at the new prefix location.
        _global = getBundle(GLOBAL_BUNDLE);
    }

    /**
     * Allows a custom classloader to be configured for locating translation resources.
     */
    public void setClassLoader (ClassLoader loader)
    {
        _loader = loader;
    }

    /**
     * Fetches the message bundle for the specified path. If no bundle can be located with the
     * specified path, a special bundle is returned that returns the untranslated message
     * identifiers instead of an associated translation. This is done so that error code to handle
     * a failed bundle load need not be replicated wherever bundles are used. Instead an error
     * will be logged and the requesting service can continue to function in an impaired state.
     */
    public MessageBundle getBundle (String path)
    {
        // first look in the cache
        MessageBundle bundle = _cache.get(path);
        if (bundle != null) {
            return bundle;
        }

        // if it's not cached, we'll need to resolve it
        ResourceBundle rbundle = loadBundle(_prefix + path);

        // if the resource bundle contains a special resource, we'll interpret that as a derivation
        // of MessageBundle to instantiate for handling that class
        MessageBundle customBundle = null;
        if (rbundle != null) {
            String mbclass = null;
            try {
                mbclass = rbundle.getString(MBUNDLE_CLASS_KEY).trim();
                if (!StringUtil.isBlank(mbclass)) {
                    customBundle = (MessageBundle)Class.forName(mbclass).newInstance();
                }

            } catch (MissingResourceException mre) {
                // nothing to worry about

            } catch (Throwable t) {
                log.warning("Failure instantiating custom message bundle", "mbclass", mbclass,
                            "error", t);
            }
        }

        // initialize our message bundle, cache it and return it (if we couldn't resolve the
        // bundle, the message bundle will cope with its null resource bundle)
        bundle = createBundle(path, rbundle, customBundle);
        _cache.put(path, bundle);
        return bundle;
    }

    /**
     * Returns the bundle to use for the given path and resource bundle. If customBundle is
     * non-null, it's an instance of the bundle class specified by the bundle itself and should be
     * used as part of the created bundle.
     */
    protected MessageBundle createBundle (String path, ResourceBundle rbundle,
        MessageBundle customBundle)
    {
        // if there was no custom class, or we failed to instantiate the custom class, use a
        // standard message bundle
        if (customBundle == null) {
            customBundle = new MessageBundle();
        }
        initBundle(customBundle, path, rbundle);
        return customBundle;
    }

    /**
     * Initializes the given bundle with this manager and the given path and resource bundle.
     */
    protected void initBundle (MessageBundle bundle, String path, ResourceBundle rbundle)
    {
        bundle.init(this, path, rbundle, _global);
    }

    /**
     * Loads a bundle from the given path, or returns null if it can't be found.
     */
    protected ResourceBundle loadBundle (String path)
    {
    	if(_rsrcMgr != null){
    		InputStream in = null;
	    	try {
	    		in = getResource(path,_locale);
	    		if(in == null){
	    			log.warning("Unable to resolve resource bundle", "path", path, "locale", _locale);
					return null;
	    		}
				return new PropertyResourceBundle(in);
			} catch (IOException e) {
				 log.warning("Unable to resolve resource bundle", "path", path, "locale", _locale, e);
				 return null;
			}finally {
				if(in != null) {
					try {
						in.close();
					} catch (IOException e) {
						 log.warning("Unable to close inputstream", "path", path, e);
					}
				}
			}
    	
    	}else{
    		try {
                if (_loader != null) {
                    return ResourceBundle.getBundle(path, _locale, _loader);
                }
                return ResourceBundle.getBundle(path, _locale);
            } catch (MissingResourceException mre) {
                log.warning("Unable to resolve resource bundle", "path", path, "locale", _locale,
                    mre);
                return null;
            }
    	}
        
    }
    
    /**
     * Fetches the message bundle for the specified path. If no bundle can be located with the
     * specified path, a special bundle is returned that returns the untranslated message
     * identifiers instead of an associated translation. This is done so that error code to handle
     * a failed bundle load need not be replicated wherever bundles are used. Instead an error
     * will be logged and the requesting service can continue to function in an impaired state.
     */
    public MessageBundle getBundle (String path,String language)
    {
        // if it's not cached, we'll need to resolve it
        ResourceBundle rbundle = loadBundle(path,language);

        // if the resource bundle contains a special resource, we'll interpret that as a derivation
        // of MessageBundle to instantiate for handling that class
        MessageBundle customBundle = null;
        if (rbundle != null) {
            String mbclass = null;
            try {
                mbclass = rbundle.getString(MBUNDLE_CLASS_KEY).trim();
                if (!StringUtil.isBlank(mbclass)) {
                    customBundle = (MessageBundle)Class.forName(mbclass).newInstance();
                }

            } catch (MissingResourceException mre) {
                // nothing to worry about

            } catch (Throwable t) {
                log.warning("Failure instantiating custom message bundle", "mbclass", mbclass,
                            "error", t);
            }
        }

        // initialize our message bundle, cache it and return it (if we couldn't resolve the
        // bundle, the message bundle will cope with its null resource bundle)
        MessageBundle bundle = createBundle(path, rbundle, customBundle);
        //_cache.put(path, bundle);
        return bundle;
    }
    
    /**
     * Loads a bundle from the given path, or returns null if it can't be found.
     */
    public ResourceBundle loadBundle (String name,String language)
    {
    	String path = _prefix + name;
    	if(_rsrcMgr != null){
    		InputStream in = null;
	    	try {
	    		String resourceName = toResourceName(toBundleName(path, language, null, null, null), PROPERTIES);
				in = _rsrcMgr.getResource(resourceName);
				if (in == null) {
					resourceName = toResourceName(toBundleName(path, null, null, null, null), PROPERTIES);
					in = _rsrcMgr.getResource(resourceName);
				}
	    		if(in == null){
	    			log.warning("Unable to resolve resource bundle", "path", path, "language", language);
					return null;
	    		}
				return new PropertyResourceBundle(in);
			} catch (IOException e) {
				 log.warning("Unable to resolve resource bundle", "path", path, "language", language, e);
				 return null;
			}finally {
				if(in != null) {
					try {
						in.close();
					} catch (IOException e) {
						 log.warning("Unable to close inputstream", "path", path, e);
					}
				}
			}
    	
    	}else{
    		 java.util.Locale local = new  java.util.Locale(language,null);
 
    		try {
                if (_loader != null) {
                    return ResourceBundle.getBundle(path, local, _loader);
                }
                return ResourceBundle.getBundle(path, local);
            } catch (MissingResourceException mre) {
                log.warning("Unable to resolve resource bundle", "path", path, "locale", _locale,
                    mre);
                return null;
            }
    	}
        
    }

    protected InputStream getResource(String baseName,Locale locale) throws IOException{
    	
    	if (locale == Locale.ROOT) {
            return _rsrcMgr.getResource(baseName);
        }

        String language = locale.getLanguage();
        String country = locale.getCountry();
        
		String resourceName = toResourceName(toBundleName(baseName, language, null, country, null), PROPERTIES);
		InputStream in = null;
		try{
			in = _rsrcMgr.getResource(resourceName);
			if (in == null) {
				resourceName = toResourceName(toBundleName(baseName, language, null, null, null), PROPERTIES);
				in = _rsrcMgr.getResource(resourceName);
				if (in == null) {
					resourceName = toResourceName(toBundleName(baseName, null, null, null, null), PROPERTIES);
					in = _rsrcMgr.getResource(resourceName);
				}
			}
		}catch(IOException e){
			resourceName = toResourceName(toBundleName(baseName, language, null, null, null), PROPERTIES);
			try{
				in = _rsrcMgr.getResource(resourceName);
			}catch(IOException e1){
				resourceName = toResourceName(toBundleName(baseName, null, null, null, null), PROPERTIES);
				in = _rsrcMgr.getResource(resourceName);
			}
		}

		return in;
    }
    
    /**
     * Converts the given <code>baseName</code> and <code>locale</code>
     * to the bundle name. This method is called from the default
     * implementation of the {@link #newBundle(String, Locale, String,
     * ClassLoader, boolean) newBundle} and {@link #needsReload(String,
     * Locale, String, ClassLoader, ResourceBundle, long) needsReload}
     * methods.
     *
     * <p>This implementation returns the following value:
     * <pre>
     *     baseName + "_" + language + "_" + script + "_" + country + "_" + variant
     * </pre>
     * where <code>language</code>, <code>script</code>, <code>country</code>,
     * and <code>variant</code> are the language, script, country, and variant
     * values of <code>locale</code>, respectively. Final component values that
     * are empty Strings are omitted along with the preceding '_'.  When the
     * script is empty, the script value is ommitted along with the preceding '_'.
     * If all of the values are empty strings, then <code>baseName</code>
     * is returned.
     *
     * <p>For example, if <code>baseName</code> is
     * <code>"baseName"</code> and <code>locale</code> is
     * <code>Locale("ja",&nbsp;"",&nbsp;"XX")</code>, then
     * <code>"baseName_ja_&thinsp;_XX"</code> is returned. If the given
     * locale is <code>Locale("en")</code>, then
     * <code>"baseName_en"</code> is returned.
     *
     * <p>Overriding this method allows applications to use different
     * conventions in the organization and packaging of localized
     * resources.
     *
     * @param baseName
     *        the base name of the resource bundle, a fully
     *        qualified class name
     * @param locale
     *        the locale for which a resource bundle should be
     *        loaded
     * @return the bundle name for the resource bundle
     * @exception NullPointerException
     *        if <code>baseName</code> or <code>locale</code>
     *        is <code>null</code>
     */
    public String toBundleName(String baseName, String language,String script,String country,String variant) {

        if ((language == "" || language == null) && (country == "" || country == null) && (variant == "" || variant == null) ) {
            return baseName;
        }

        StringBuilder sb = new StringBuilder(baseName);
        sb.append('_');
        if (script != null && !script.equals("")) {
            if (variant != null && !variant.equals("")) {
                sb.append(language).append('_').append(script).append('_').append(country).append('_').append(variant);
            } else if (country != null && !country.equals("")) {
                sb.append(language).append('_').append(script).append('_').append(country);
            } else {
                sb.append(language).append('_').append(script);
            }
        } else {
            if (variant != null && !variant.equals("")) {
                sb.append(language).append('_').append(country).append('_').append(variant);
            } else if (country != null && !country.equals("")) {
                sb.append(language).append('_').append(country);
            } else {
                sb.append(language);
            }
        }
        return sb.toString();

    }
    
    /**
     * Converts the given <code>bundleName</code> to the form required
     * by the {@link ClassLoader#getResource ClassLoader.getResource}
     * method by replacing all occurrences of <code>'.'</code> in
     * <code>bundleName</code> with <code>'/'</code> and appending a
     * <code>'.'</code> and the given file <code>suffix</code>. For
     * example, if <code>bundleName</code> is
     * <code>"foo.bar.MyResources_ja_JP"</code> and <code>suffix</code>
     * is <code>"properties"</code>, then
     * <code>"foo/bar/MyResources_ja_JP.properties"</code> is returned.
     *
     * @param bundleName
     *        the bundle name
     * @param suffix
     *        the file type suffix
     * @return the converted resource name
     * @exception NullPointerException
     *         if <code>bundleName</code> or <code>suffix</code>
     *         is <code>null</code>
     */
    public final String toResourceName(String bundleName, String suffix) {
        StringBuilder sb = new StringBuilder(bundleName.length() + 1 + suffix.length());
        sb.append(bundleName.replace('.', '/')).append('.').append(suffix);
        return sb.toString();
    }
    
    /** The prefix we prepend to resource paths prior to loading. */
    protected String _prefix;

    /** The locale for which we're obtaining message bundles. */
    protected Locale _locale;

    /** A custom class loader that we use to load resource bundles. */
    protected ClassLoader _loader;

    /** A cache of instantiated message bundles. */
    protected HashMap<String, MessageBundle> _cache = new HashMap<String, MessageBundle>();

    /** Our top-level message bundle, from which others obtain messages if
     * they can't find them within themselves. */
    protected MessageBundle _global;

    private ResourceManager _rsrcMgr;
    
    /** A key that can contain the classname of a custom message bundle
     * class to be used to handle messages for a particular bundle. */
    protected static final String MBUNDLE_CLASS_KEY = "msgbundle_class";
}
