//
// Nenya library - tools for developing networked games
// Copyright (C) 2002-2012 Three Rings Design, Inc., All Rights Reserved
// http://code.google.com/p/nenya/
//
// This library is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License as published
// by the Free Software Foundation; either version 2.1 of the License, or
// (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package com.threerings.resource;

import static com.threerings.resource.Log.log;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLStreamHandler;
import java.security.Permission;

import com.samskivert.util.StringUtil;

/**
 * This class is not used directly, except by a registering ResourceManager so that we can load
 * data from the resource manager using URLs of the form
 * <code>resource://&lt;resourceSet&gt;/&lt;path&gt;</code>. ResourceSet may be the empty string
 * to load from the default resource sets.
 */
public class ResourceHandler extends URLStreamHandler
{
	public static String RESOURCE_PROTOCOL = "resource";

    public void setResourceManager(ResourceManager _rmgr){
    	this._rmgr = _rmgr;
    }
    
    @Override
    protected int hashCode (URL url)
    {
        return String.valueOf(url).hashCode();
    }

    @Override
    protected boolean equals (URL u1, URL u2)
    {
        return String.valueOf(u1).equals(String.valueOf(u2));
    }

    @Override
    protected URLConnection openConnection (URL url)
        throws IOException
    {
        return new URLConnection(url) {
            @Override
            public void connect ()
                throws IOException
            {
                // the host is the bundle name
                String bundle = this.url.getHost();
                // and we need to remove the leading '/' from path;
                String path = this.url.getPath().substring(1);
                try {
                    // if there are query parameters, we need special magic
                    String query = url.getQuery();
                    if (!StringUtil.isBlank(query)) {
                        _stream = getStream(bundle, path, query);
                    } else if (StringUtil.isBlank(bundle)) {
                        _stream = _rmgr.getResource(path);
                    } else {
                        _stream = _rmgr.getResource(bundle, path);
                    }
                    this.connected = true;

                } catch (IOException ioe) {
                    log.warning("Could not find resource",
                        "url", this.url, "error", ioe.getMessage());
                    throw ioe; // rethrow
                }
            }

            @Override
            public InputStream getInputStream ()
                throws IOException
            {
                if (!this.connected) {
                    connect();
                }
                return _stream;
            }

            @Override
            public Permission getPermission ()
                throws IOException
            {
                // We allow anything in the resource bundle to be loaded
                // without any permission restrictions.
                return null;
            }

            protected InputStream _stream;
        };
    }

    /**
     * Does some magic to allow a subset of an image to be extracted, reencoded as a PNG and then
     * spat back out to the Java content handler system for inclusion in internal documentation.
     */
    protected InputStream getStream (String bundle, String path, String query)
        throws IOException
    {

        log.warning("Requested sub-tile class="+this.getClass()+" is not support",
                "bundle", bundle, "path", path, "dims", query);
        return _rmgr.getResource(bundle, path);
    }

    protected ResourceManager _rmgr;
}
