package com.threerings.resource;

import static com.threerings.resource.Log.log;

import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.InputStream;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.Properties;

import javax.imageio.ImageIO;
import javax.imageio.stream.ImageInputStream;
import javax.imageio.stream.MemoryCacheImageInputStream;

import com.samskivert.io.StreamUtil;
import com.samskivert.net.PathUtil;

public abstract class AbstractResourceManager implements ResourceManager{
	/** Converts a path to a locale-specific path. */
    protected LocaleHandler _localeHandler;
    
    /**
     * Loads the configuration properties for our resource sets.
     */
    protected Properties loadConfig (String configPath)
        throws IOException
    {
        Properties config = new Properties();
        try {
            config.load(getResource(configPath));
        } catch (Exception e) {
            String errmsg = "Unable to load resource manager config, cpath=" + configPath + "]";
            log.warning(errmsg + ".", e);
            throw new IOException(errmsg);
        }
        return config;
    }
    
    /**
     * Registers a protocol handler with URL to handle <code>resource:</code> URLs. The URLs take
     * the form: <pre>resource://bundle_name/resource_path</pre> Resources from the default bundle
     * can be loaded via: <pre>resource:///resource_path</pre>
     */
    public void activateResourceProtocol ()
    {
        // set up a URL handler so that things can be loaded via urls with the 'resource' protocol
        try {
            AccessController.doPrivileged(new PrivilegedAction<Void>() {
                public Void run () {
                	ResourceHandler handler = null;
                	try {
						Class<ResourceHandler> clazz = (Class<ResourceHandler>) Class.forName(System.getProperty("resource.handler", "com.threerings.resource.SubImageResourceHandler"));
						handler = clazz.newInstance();
						handler.setResourceManager(AbstractResourceManager.this);
					} catch (Exception e) {
						log.error("ResourceHandler init error, use default ResourceHandler class to bind resource:// ",e);
						handler = new ResourceHandler();
					}
                	
                	handler.setResourceManager(AbstractResourceManager.this);
                	AttachableURLFactory.attachHandler(ResourceHandler.RESOURCE_PROTOCOL, handler);
                    return null;
                }
            });
        } catch (SecurityException se) {
            log.info("Running in sandbox. Unable to bind "+ResourceHandler.RESOURCE_PROTOCOL+":// handler.");
        }
    }

    /**
     * Configure a default LocaleHandler with the specified prefix.
     */
    public void setLocalePrefix (final String prefix)
    {
        setLocaleHandler(
            new LocaleHandler() {
                public String getLocalePath (String path) {
                	if(path != null) {
                		int index = path.lastIndexOf(".");
                		if(index >0) {
                			int slash = path.lastIndexOf("/");
                			if(slash< index) {
                				String before = path.substring(0, index);
                				String ext = path.substring(index+1); 
                				return before +"_"+prefix+"." + ext;
                			}
                		}
                	}
                    return PathUtil.appendPath(prefix, path);
                }
            });
    }

    /**
     * Configure a custom LocaleHandler.
     */
    public void setLocaleHandler (LocaleHandler localeHandler)
    {
        _localeHandler = localeHandler;
    }

    /**
     * Returns an InputStream from this manager's classloader for the given path.
     */
    protected InputStream getInputStreamFromClasspath (final String fullyQualifiedPath)
    {
    	return AbstractResourceManager.class.getResourceAsStream(fullyQualifiedPath);
    }

    /**
     * Transform the path into a locale-specific one, or return null.
     */
    protected String getLocalePath (String path)
    {
        return (_localeHandler == null) ? null : _localeHandler.getLocalePath(path);
    }
    
    /**
     * Transforms a regular resource path into a locale-specific path. The returned path
     * does not need to represent a valid resource. The ResourceManager will attempt to use
     * the locale-specific path, but if it fails will fall back to the generic path.
     */
    public interface LocaleHandler
    {
        /**
         * Return a locale-specific path, or null if the specified path cannot or need not
         * be transformed.
         */
        String getLocalePath (String path);
    }
    
    /**
     * Loads an image from the given input stream. Supports formats supported by {@link ImageIO}
     * as well as {@link FastImageIO} based on the useFastIO param.
     */
    public static BufferedImage loadImage (InputStream iis, boolean useFastIO)
        throws IOException
    {
        if (iis == null) {
            return null;
        } else if (useFastIO) {
            return FastImageIO.read(iis);
        }
        return ImageIO.read(iis);
    }

    /**
     * Loads an image from the supplied input stream. Supports formats supported by {@link ImageIO}
     * but not {@link FastImageIO}.
     */
    protected static BufferedImage loadImage (InputStream iis)
        throws IOException
    {
        BufferedImage image;

        if (iis instanceof ImageInputStream) {
            image = ImageIO.read(iis);

        } else {
            // if we don't already have an image input stream, create a memory cache image input
            // stream to avoid causing freakout if we're used in a sandbox because ImageIO
            // otherwise use FileCacheImageInputStream which tries to create a temp file
            MemoryCacheImageInputStream mciis = new MemoryCacheImageInputStream(iis);
            image = ImageIO.read(mciis);
            try {
                // this doesn't close the underlying stream
                mciis.close();
            } catch (IOException ioe) {
                // ImageInputStreamImpl.close() throws an IOException if it's already closed;
                // there's no way to find out if it's already closed or not, so we have to check
                // the exception message to determine if this is actually warning worthy
                if (!"closed".equals(ioe.getMessage())) {
                    log.warning("Failure closing image input '" + iis + "'.", ioe);
                }
            }
        }

        // finally close our input stream
        StreamUtil.close(iis);

        return image;
    }
}
