/*
 * Copyright 2008-2108 amoeba.meidusa.com 
 * 
 * 	This program is free software; you can redistribute it and/or modify it under the terms of 
 * the GNU AFFERO GENERAL PUBLIC LICENSE as published by the Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version. 
 * 
 * 	This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * See the GNU AFFERO GENERAL PUBLIC LICENSE for more details. 
 * 	You should have received a copy of the GNU AFFERO GENERAL PUBLIC LICENSE along with this program; 
 * if not, write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package com.meidusa.venus.client;

import java.io.IOException;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.pool.ObjectPool;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;

import com.meidusa.venus.annotations.Endpoint;
import com.meidusa.venus.annotations.PerformanceLevel;
import com.meidusa.venus.annotations.Service;
import com.meidusa.venus.annotations.util.AnnotationUtil;
import com.meidusa.venus.client.xml.bean.EndpointConfig;
import com.meidusa.venus.client.xml.bean.ServiceConfig;
import com.meidusa.venus.exception.CodedException;
import com.meidusa.venus.exception.DefaultVenusException;
import com.meidusa.venus.exception.InvalidParameterException;
import com.meidusa.venus.exception.VenusConfigException;
import com.meidusa.venus.exception.VenusExceptionFactory;
import com.meidusa.venus.io.packet.AbstractServicePacket;
import com.meidusa.venus.io.packet.AbstractServiceRequestPacket;
import com.meidusa.venus.io.packet.ErrorPacket;
import com.meidusa.venus.io.packet.OKPacket;
import com.meidusa.venus.io.packet.PacketConstant;
import com.meidusa.venus.io.packet.ServicePacketBuffer;
import com.meidusa.venus.io.packet.ServiceResponsePacket;
import com.meidusa.venus.io.packet.serialize.SerializeServiceRequestPacket;
import com.meidusa.venus.io.packet.serialize.SerializeServiceResponsePacket;
import com.meidusa.venus.io.serializer.Serializer;
import com.meidusa.venus.io.serializer.SerializerFactory;
import com.meidusa.venus.io.serializer.json.JsonSerializer;
import com.meidusa.venus.metainfo.EndpointParameter;
import com.meidusa.venus.notify.InvocationListener;
import com.meidusa.venus.notify.ReferenceInvocationListener;
import com.meidusa.venus.poolable.RequestLoadbalanceObjectPool;
import com.meidusa.venus.util.Utils;
import com.meidusa.venus.util.VenusAnnotationUtils;
import com.meidusa.fastjson.JSON;
import com.meidusa.toolkit.common.bean.util.InitialisationException;
import com.meidusa.toolkit.common.util.StringUtil;
import com.meidusa.toolkit.net.BIOConnection;
import com.meidusa.toolkit.net.Connection;

/**
 * 
 * @author Struct
 *
 */
public class RemotingInvocationHandler extends VenusInvocationHandler {
	private static Logger logger = Logger.getLogger(RemotingInvocationHandler.class);
	private static Logger performanceLogger = Logger.getLogger("venus.service.performance");
	private InvocationListenerContainer container;
	private ObjectPool bioConnPool;
	private ObjectPool nioConnPool;
	private static AtomicLong sequenceId = new AtomicLong(1);
	private VenusServiceFactory serviceFactory;
	private VenusExceptionFactory venusExceptionFactory;
	private boolean enableAsync = true;
	private byte serializeType = PacketConstant.CONTENT_TYPE_JSON;
	public boolean isEnableAsync() {
		return enableAsync;
	}

	public void setEnableAsync(boolean enableAsync) {
		this.enableAsync = enableAsync;
	}

	public VenusExceptionFactory getVenusExceptionFactory() {
		return venusExceptionFactory;
	}

	public void setVenusExceptionFactory(VenusExceptionFactory venusExceptionFactory) {
		this.venusExceptionFactory = venusExceptionFactory;
	}
	
	public void setServiceFactory(VenusServiceFactory serviceFactory) {
		this.serviceFactory = serviceFactory;
	}

	public short getSerializeType() {
		return serializeType;
	}

	public void setSerializeType(byte serializeType) {
		this.serializeType = serializeType;
	}

	public InvocationListenerContainer getContainer() {
		return container;
	}

	public void setContainer(InvocationListenerContainer container) {
		this.container = container;
	}

	public ObjectPool getBioConnPool() {
		return bioConnPool;
	}

	public void setBioConnPool(ObjectPool bioConnPool) {
		this.bioConnPool = bioConnPool;
	}

	public ObjectPool getNioConnPool() {
		return nioConnPool;
	}

	public void setNioConnPool(ObjectPool nioConnPool) {
		this.nioConnPool = nioConnPool;
	}

	protected Object invokeRemoteService(Service service,Endpoint endpoint, Method method, EndpointParameter[] params, Object[] args) throws Exception{
		boolean async = false;
		
		if(endpoint.async()){
			async = true;
		}
		
		AbstractServiceRequestPacket serviceRequestPacket = null;
		
		Serializer serializer = SerializerFactory.getSerializer(serializeType);
		serviceRequestPacket = new SerializeServiceRequestPacket(serializer,null);
		serviceRequestPacket.clientId = VenusConstant.VENUS_CLIENT_ID;
		serviceRequestPacket.clientRequestId = sequenceId.getAndIncrement();
		
		serviceRequestPacket.apiName = VenusAnnotationUtils.getApiname(method, service, endpoint);
		serviceRequestPacket.serviceVersion = service.version();
		serviceRequestPacket.parameterMap = new HashMap<String,Object>();
		if(params != null){
			for(int i=0;i<params.length;i++){
				if(args[i] instanceof InvocationListener){
					async = true;
					ReferenceInvocationListener listener = new ReferenceInvocationListener();
					ServicePacketBuffer buffer = new ServicePacketBuffer(16);
					buffer.writeLengthCodedString(args[i].getClass().getName(), "utf-8");
					buffer.writeInt(System.identityHashCode(args[i]));
					listener.setIdentityData(buffer.toByteBuffer().array());
					Type type = method.getGenericParameterTypes()[i];
					if (type instanceof ParameterizedType) {
						ParameterizedType genericType = ((ParameterizedType) type);
						container.putInvocationListener((InvocationListener)args[i],genericType.getActualTypeArguments()[0]);
					}else{
						throw new InvalidParameterException("invocationListener is not generic");
					}
					
					serviceRequestPacket.parameterMap.put(params[i].getParamName(),listener);
				}else{
					serviceRequestPacket.parameterMap.put(params[i].getParamName(), args[i]);
				}
				
			}
		}
		
		PerformanceLevel pLevel = AnnotationUtil.getAnnotation(method.getAnnotations(), PerformanceLevel.class);
		long start = System.currentTimeMillis();
		long borrowed = start;
		
		if(async){
			if(!this.isEnableAsync()){
				throw new VenusConfigException("service async call disabled");
			}
			
			Connection conn = null;
			try{
				
				if(nioConnPool instanceof RequestLoadbalanceObjectPool){
					conn = (Connection)((RequestLoadbalanceObjectPool)nioConnPool).borrowObject(serviceRequestPacket.parameterMap, endpoint);
				}else{
					conn = (Connection)nioConnPool.borrowObject();
				}
				borrowed = System.currentTimeMillis();
				conn.postMessage(serviceRequestPacket.toByteBuffer());
				
				return null;
			}finally{
				if(performanceLogger.isEnabledFor(Level.DEBUG)){
					long end = System.currentTimeMillis();
					long time = end - borrowed;
					StringBuffer buffer = new StringBuffer();
					buffer.append("[").append(borrowed - start).append(",")
						  .append(time).append("]ms api=")
						  .append(serviceRequestPacket.apiName);
					
					performanceLogger.log(Level.DEBUG, buffer.toString());
				}
				if(conn != null){
					nioConnPool.returnObject(conn);
				}
			}
		}else{
			BIOConnection conn = null;
			int soTimeout = 0;
			int oldTimeout = 0;
			boolean success = true;
			try{
				if(bioConnPool instanceof RequestLoadbalanceObjectPool){
					conn = (BIOConnection)((RequestLoadbalanceObjectPool)bioConnPool).borrowObject(serviceRequestPacket.parameterMap, endpoint);
				}else{
					conn = (BIOConnection)bioConnPool.borrowObject();
				}
				borrowed = System.currentTimeMillis();
				ServiceConfig config = this.serviceFactory.getServiceConfig(method.getDeclaringClass());
				
				oldTimeout = conn.getSoTimeout();
				if(config != null){
					EndpointConfig endpointConfig = config.getEndpointConfig(endpoint.name());
					if(endpointConfig != null){
						int eTimeOut = endpointConfig.getSoTimeout();
						if(eTimeOut > 0 ){
							soTimeout = eTimeOut;
						}
					}else{
						soTimeout = config.getSoTimeout();
					}
					
					if(soTimeout>0){
						conn.setSoTimeout(soTimeout);
					}
				}
				byte[] bts = null;
				try{
					conn.write(serviceRequestPacket.toByteBuffer().array());
					bts = conn.read();
				}catch(IOException e){
					try{
						conn.close();
					}catch(Exception e1){};
					success = false;
					if(e instanceof CodedException){
						throw e;
					}else{
						throw new DefaultVenusException(e.getMessage()+". remoteAddress="+conn.getRemoteAddress(), e);
					}
				}
				
				int type = AbstractServicePacket.getType(bts);
				switch(type){
					case PacketConstant.PACKET_TYPE_ERROR:
						ErrorPacket error = new ErrorPacket();
						error.init(bts);
						Exception e = venusExceptionFactory.getException(error.errorCode, error.message);
						if(e == null){
							throw new DefaultVenusException(error.errorCode,error.message);
						}else{
							if(error.additionalData != null){
								Map<String,Type> tmap = Utils.getBeanFieldType( e.getClass(), Exception.class);
								if(tmap != null && tmap.size() >0){
									Object obj = serializer.decode(error.additionalData, tmap);
									BeanUtils.copyProperties(e,obj);
								}
							}
							throw e;
						}
					case PacketConstant.PACKET_TYPE_OK:
						OKPacket ok = new OKPacket();
						ok.init(bts);
						return null;
					case PacketConstant.PACKET_TYPE_SERVICE_RESPONSE:
						ServiceResponsePacket response = null;
						
						response = new SerializeServiceResponsePacket(serializer,method.getGenericReturnType());
						
						response.init(bts);
						return response.result;
					default : {
						logger.warn("unknow response type="+type);
						success = false;
						return null;
					}
				}
			} catch(Exception e){
				success = false;
				if(e instanceof CodedException || e instanceof RuntimeException || venusExceptionFactory.getErrorCode(e.getClass()) != 0){
					throw e;
				}else{
					if(conn == null){
						throw new DefaultVenusException(e.getMessage(), e);
					}else{
						throw new DefaultVenusException(e.getMessage()+". remoteAddress="+conn.getRemoteAddress(), e);
					}
				}
			}finally{
				long end = System.currentTimeMillis();
				long time = end - borrowed;
				StringBuffer buffer = new StringBuffer();
				buffer.append("[").append(borrowed - start).append(",")
					  .append(time).append("]ms api=")
					  .append(serviceRequestPacket.apiName).append(", success=").append(success);
				
				if(conn != null){
					buffer.append(", remote=").append(conn.getRemoteAddress());
				}
				
				buffer.append(", clientID=").append(VenusConstant.VENUS_CLIENT_ID)
					  .append(", requestID=").append(serviceRequestPacket.clientRequestId);
				
				if(pLevel != null){
					
					if(pLevel.printParams()){
						buffer.append(", params=\n");
						buffer.append(JSON.toJSONString(serviceRequestPacket.parameterMap));
					}
					
					if(time > pLevel.error() &&  pLevel.error() > 0 ){
						if(performanceLogger.isEnabledFor(Level.ERROR)){
							performanceLogger.log(Level.ERROR, buffer.toString());
						}
					}else if(time > pLevel.warn() &&  pLevel.warn() > 0 ){
						if(performanceLogger.isEnabledFor(Level.WARN)){
							performanceLogger.log(Level.WARN, buffer.toString());
						}
					}else if(time > pLevel.info() &&  pLevel.info() > 0 ){
						if(performanceLogger.isEnabledFor(Level.INFO)){
							performanceLogger.log(Level.INFO, buffer.toString());
						}
					}else{
						if(performanceLogger.isEnabledFor(Level.DEBUG)){
							performanceLogger.log(Level.DEBUG, buffer.toString());
						}
					}
				}else{
					if(time >= 30 * 1000){
						if(performanceLogger.isEnabledFor(Level.ERROR)){
							performanceLogger.error(buffer.toString());
						}
					}else if(time >= 10 * 1000){
						if(performanceLogger.isEnabledFor(Level.WARN)){
							performanceLogger.warn(buffer.toString());
						}
					}else if(time >= 5 * 1000){
						if(performanceLogger.isInfoEnabled()){
							performanceLogger.info(buffer.toString());
						}
					}else{
						if(performanceLogger.isDebugEnabled()){
							performanceLogger.debug(buffer.toString());
						}
					}
				}
				
				if(conn != null){
					if(!conn.isClosed() && soTimeout >0 ){
						conn.setSoTimeout(oldTimeout);
					}
					bioConnPool.returnObject(conn);
				}
			}
		}
	}
	
	
	public void init() throws InitialisationException {
	    
	}
}
