/*
 * Decompiled with CFR 0.152.
 */
package org.robovm.debugger.utils.macho;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteOrder;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.BiConsumer;
import java.util.function.BiPredicate;
import java.util.function.Predicate;
import java.util.function.ToLongFunction;
import org.robovm.debugger.utils.Pair;
import org.robovm.debugger.utils.bytebuffer.CompositeDataBufferReader;
import org.robovm.debugger.utils.bytebuffer.CompositeDataBufferWriter;
import org.robovm.debugger.utils.bytebuffer.DataBufferReader;
import org.robovm.debugger.utils.bytebuffer.DataBufferReaderWriter;
import org.robovm.debugger.utils.bytebuffer.DataByteBufferWriter;
import org.robovm.debugger.utils.bytebuffer.NullDataBufferReader;
import org.robovm.debugger.utils.macho.MachOException;
import org.robovm.debugger.utils.macho.cmds.LinkeditDataCommand;
import org.robovm.debugger.utils.macho.cmds.SegmentCommand;
import org.robovm.debugger.utils.macho.cmds.SymtabCommand;
import org.robovm.debugger.utils.macho.structs.DyLdChainedFixups;
import org.robovm.debugger.utils.macho.structs.MachHeader;
import org.robovm.debugger.utils.macho.structs.Section;
import org.robovm.debugger.utils.macho.tools.ChainedFixup;
import org.robovm.debugger.utils.macho.tools.ExportedSymbolsParser;
import org.robovm.debugger.utils.macho.tools.RegionSquasher;

public class MachOLoader {
    private final DataBufferReader rootReader;
    private final DataBufferReaderWriter rootReaderWriter;
    private final MachHeader machOHeader;
    private final int machOCpuType;
    private final List<SegmentCommand> segments = new ArrayList<SegmentCommand>();
    private final Map<String, Long> symTable = new HashMap<String, Long>();

    public MachOLoader(File executable, int cpuType) throws MachOException {
        MappedByteBuffer bb;
        try {
            bb = new RandomAccessFile(executable, "rw").getChannel().map(FileChannel.MapMode.PRIVATE, 0L, executable.length());
        }
        catch (IOException e) {
            throw new MachOException("Failed to open mach-o file", e);
        }
        this.rootReaderWriter = new DataByteBufferWriter(bb);
        this.rootReader = this.rootReaderWriter;
        this.rootReader.setByteOrder(ByteOrder.LITTLE_ENDIAN);
        long magic = this.rootReader.readUnsignedInt32();
        if (magic == 3199925962L || magic == 3216703178L) {
            throw new MachOException("FAT mach-o files are not supported!");
        }
        MachHeader header = this.parseMachHeader(this.rootReader, magic);
        if (header.cputype() != cpuType) {
            throw new MachOException("CPU type " + Integer.toHexString(cpuType) + " is not present in mach-o file");
        }
        this.machOHeader = header;
        this.machOCpuType = cpuType;
        this.readCommandData(this.rootReader, this.machOHeader);
    }

    private MachHeader parseMachHeader(DataBufferReader reader, long magic) throws MachOException {
        if (magic != 4277009102L && magic != 4277009103L) {
            throw new MachOException("unexpected Mach header MAGIC 0x" + Long.toHexString(magic));
        }
        return new MachHeader(reader, magic == 4277009103L);
    }

    private void readCommandData(DataBufferReader reader, MachHeader header) {
        ArrayList<LinkeditDataCommand> dyldChainedFixups = null;
        HashMap<String, ExportedSymbolsParser.ResolvedSymbol> exportedSymbols = new HashMap<String, ExportedSymbolsParser.ResolvedSymbol>();
        long imageBaseAddress = 0L;
        int sectionIdx = 1;
        int idx = 0;
        while ((long)idx < header.ncmds()) {
            long pos = reader.position();
            int cmd = (int)reader.readUnsignedInt32();
            int cmdsize = (int)reader.readUnsignedInt32();
            if (cmd == 1 || cmd == 25) {
                SegmentCommand segCmd = new SegmentCommand(reader, cmd == 25, sectionIdx);
                if (header.is64b() != segCmd.is64b()) {
                    throw new RuntimeException("bits of LC_SEGMENT shall match MAGIC header");
                }
                this.segments.add(segCmd);
                sectionIdx += segCmd.sections().length;
                if ("__TEXT".equals(segCmd.segname())) {
                    imageBaseAddress = segCmd.vmaddr();
                }
            } else if (cmd == 2) {
                SymtabCommand symtabCommand = new SymtabCommand(reader);
                ExportedSymbolsParser.parseSymtabCommand(exportedSymbols, reader, symtabCommand, header.is64b());
            } else if (cmd == -2147483597) {
                LinkeditDataCommand dyldExportTrie = new LinkeditDataCommand(reader);
                ExportedSymbolsParser.parseExportTrie(exportedSymbols, reader.sliceAt(dyldExportTrie.getDataoff(), dyldExportTrie.getDatasize()));
            } else if (cmd == -2147483596) {
                LinkeditDataCommand linkedit = new LinkeditDataCommand(reader);
                if (dyldChainedFixups == null) {
                    dyldChainedFixups = new ArrayList<LinkeditDataCommand>();
                }
                dyldChainedFixups.add(linkedit);
            }
            reader.setPosition(pos + (long)cmdsize);
            ++idx;
        }
        if (dyldChainedFixups != null) {
            this.fixupAllChainedFixups(imageBaseAddress, dyldChainedFixups, exportedSymbols);
        }
        for (Map.Entry e : exportedSymbols.entrySet()) {
            this.symTable.put((String)e.getKey(), ((ExportedSymbolsParser.ResolvedSymbol)e.getValue()).target);
        }
    }

    private void fixupAllChainedFixups(long imageBaseAddress, List<LinkeditDataCommand> fixups, Map<String, ExportedSymbolsParser.ResolvedSymbol> exportedSymbols) {
        ArrayList<DataBufferReaderWriter> regions = new ArrayList<DataBufferReaderWriter>();
        for (SegmentCommand segment : this.segments) {
            if (segment.vmaddr() == 0L || segment.vmsize() <= 0L || "__LINKEDIT".equals(segment.segname())) continue;
            DataBufferReaderWriter writer = this.rootReaderWriter.sliceAt(segment.fileoff(), (int)segment.filesize(), segment.vmaddr(), this.isPatform64Bit());
            regions.add(writer);
        }
        CompositeDataBufferWriter compositeWriter = new CompositeDataBufferWriter(regions.toArray(new DataBufferReaderWriter[0]));
        long slide = 0L;
        for (LinkeditDataCommand linkedit : fixups) {
            DyLdChainedFixups.Header fixupsHeader = new DyLdChainedFixups.Header(this.rootReader.sliceAt(linkedit.getDataoff(), linkedit.getDatasize()), exportedSymbols);
            ChainedFixup mapper = new ChainedFixup(imageBaseAddress, compositeWriter);
            mapper.fixupAllChainedFixups(fixupsHeader.startsInSegment, slide, fixupsHeader.imports);
        }
    }

    public static int cpuTypeFromString(String cpuString) {
        switch (cpuString) {
            case "i386": 
            case "x86": {
                return 7;
            }
            case "x86_64": {
                return 0x1000007;
            }
            case "thumbv7": 
            case "armv7": {
                return 12;
            }
            case "arm64": {
                return 0x100000C;
            }
        }
        return -1;
    }

    public int cpuType() {
        return this.machOCpuType;
    }

    public boolean isPatform64Bit() {
        switch (this.machOCpuType) {
            case 0x1000007: 
            case 0x100000C: {
                return true;
            }
            case 7: 
            case 12: {
                return false;
            }
        }
        throw new RuntimeException("Unknown CPU type to get data bit width " + this.machOCpuType);
    }

    public long resolveSymbol(String symbolName) {
        Long symbAddr = this.symTable.get("_" + symbolName);
        if (symbAddr == null) {
            return -1L;
        }
        return symbAddr;
    }

    public DataBufferReader memoryReader() {
        ArrayList regions = new ArrayList();
        ArrayList<Pair> sections = new ArrayList<Pair>();
        for (SegmentCommand segment : this.segments) {
            if (segment.vmaddr() == 0L || segment.vmsize() <= 0L || "__LINKEDIT".equals(segment.segname())) continue;
            for (Section section : segment.sections()) {
                sections.add(new Pair<SegmentCommand, Section>(segment, section));
            }
        }
        Predicate<Pair> uninitialized = p -> {
            Section section = (Section)p.right;
            return "__DATA".equals(section.segname()) && ("__bss".equals(section.sectname()) || "__common".equals(section.sectname()));
        };
        ToLongFunction<Pair> rightMarginOnDisk = p -> Math.min(((Section)p.right).addr() + ((Section)p.right).size(), ((SegmentCommand)p.left).vmaddr() + ((SegmentCommand)p.left).filesize());
        BiPredicate<Pair, Pair> canSquash = (first, last) -> {
            boolean zeroSection = uninitialized.test((Pair)first);
            if (zeroSection != uninitialized.test((Pair)last)) {
                return false;
            }
            if (zeroSection) {
                return ((Section)last.right).addr() - (((Section)first.right).addr() + ((Section)first.right).size()) < 64L;
            }
            return ((Section)last.right).addr() - rightMarginOnDisk.applyAsLong((Pair)first) < 64L;
        };
        sections.sort(Comparator.comparingLong(p -> ((Section)p.right).addr()));
        BiConsumer<Pair, Pair> squasher = (p1, p2) -> {
            DataBufferReader reader;
            Section first = (Section)p1.right;
            Section last = (Section)p2.right;
            if (uninitialized.test((Pair)p1)) {
                long size = last.addr() + last.size() - first.addr();
                reader = new NullDataBufferReader(first.addr(), size, this.isPatform64Bit());
            } else {
                long size = rightMarginOnDisk.applyAsLong((Pair)p2) - first.addr();
                reader = this.rootReader.sliceAt(first.offset(), (int)size, first.addr(), this.isPatform64Bit());
            }
            regions.add(reader);
        };
        RegionSquasher.squash(sections, canSquash, squasher);
        return new CompositeDataBufferReader(regions.toArray(new DataBufferReader[0]));
    }

    public static void main(String[] argv) {
        try {
            MachOLoader loader = new MachOLoader(new File(argv[0]), MachOLoader.cpuTypeFromString(argv[1]));
            System.out.println("Segments:");
            System.out.printf("  %-30s | %10s | %16s | %16s%n", "", "size", "file_offs", "vm_addr");
            for (SegmentCommand segmentCommand : loader.segments) {
                System.out.printf("  %-30s | %10X | %16X | %16X%n", segmentCommand.segname(), segmentCommand.filesize(), segmentCommand.fileoff(), segmentCommand.vmaddr());
                int idx = 0;
                for (Section section : segmentCommand.sections()) {
                    System.out.printf("  %4d:%-25s | %10X | %16X | %16X%n", segmentCommand.firstSectionIdx() + idx, section.sectname(), section.size(), section.offset(), section.addr());
                    ++idx;
                }
            }
            System.out.printf("Symbols(%d):%n", loader.symTable.size());
            for (Map.Entry entry : loader.symTable.entrySet()) {
                System.out.println("  " + (String)entry.getKey());
            }
        }
        catch (MachOException e) {
            e.printStackTrace();
        }
    }
}

