package com.meidusa.venus.validate.validator.annotation;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.HashMap;

import com.meidusa.venus.validate.chain.ValidatorChain;
import com.meidusa.venus.validate.chain.VenusValidatorChain;
import com.meidusa.venus.validate.exception.ValidationRuntimeException;
import com.meidusa.venus.validate.validator.DateRangeFieldValidator;
import com.meidusa.venus.validate.validator.DoubleRangeFieldValidator;
import com.meidusa.venus.validate.validator.EmailValidator;
import com.meidusa.venus.validate.validator.ExpressionValidator;
import com.meidusa.venus.validate.validator.FieldExpressionValidator;
import com.meidusa.venus.validate.validator.FieldValidator;
import com.meidusa.venus.validate.validator.IntRangeFieldValidator;
import com.meidusa.venus.validate.validator.LongRangeFieldValidator;
import com.meidusa.venus.validate.validator.RegexFieldValidator;
import com.meidusa.venus.validate.validator.RequiredFieldValidator;
import com.meidusa.venus.validate.validator.RequiredStringValidator;
import com.meidusa.venus.validate.validator.StringLengthFieldValidator;
import com.meidusa.venus.validate.validator.URLValidator;
import com.meidusa.venus.validate.validator.VisitorFieldValidator;

public class AnnotationValidatorFactoryGroup {

	public static class DateRangeFactory implements
			AnnotationValidatorFactory<DateRange, DateRangeFieldValidator> {
		private HashMap<String, DateFormat> patternMap = new HashMap<String, DateFormat>();

		@Override
		public DateRangeFieldValidator createValidator(DateRange anno, String fieldName,
				Class<?> fieldClazz) {
			DateRangeFieldValidator validator = new DateRangeFieldValidator();

			DateFormat format = patternMap.get(anno.format());
			if (format == null) {
				synchronized (patternMap) {
					format = patternMap.get(anno.format());
					if (format == null) {
						format = new SimpleDateFormat(anno.format());
						patternMap.put(anno.format(), format);
					}
				}
			}

			try {
				validator.setMin(format.parse(anno.min()));
				validator.setMax(format.parse(anno.max()));
			} catch (ParseException e) {
				throw new ValidationRuntimeException("can't parse date, format should be set", e);
			}
			validator.setMessage(anno.message());
			validator.setFieldName(fieldName);
			return validator;
		}

	}

	public static class DoubleRangeFactory implements
			AnnotationValidatorFactory<DoubleRange, DoubleRangeFieldValidator> {

		@Override
		public DoubleRangeFieldValidator createValidator(DoubleRange anno, String fieldName,
				Class<?> fieldClazz) {
			DoubleRangeFieldValidator validator = new DoubleRangeFieldValidator();
			validator.setMaxExclusiveValue(anno.maxExclusive());
			validator.setMaxInclusiveValue(anno.maxInclusive());
			validator.setMinExclusiveValue(anno.minExclusive());
			validator.setMinInclusiveValue(anno.minInclusive());
			validator.setFromAnnotation(true);
			validator.setMessage(anno.message());
			validator.setFieldName(fieldName);
			return validator;
		}

	}

	public static class EmailFactory implements AnnotationValidatorFactory<Email, EmailValidator> {

		@Override
		public EmailValidator createValidator(Email anno, String fieldName, Class<?> fieldClazz) {
			EmailValidator validator = new EmailValidator();
			validator.setMessage(anno.message());
			validator.setTrim(anno.trim());
			validator.setCaseSensitive(anno.caseSensitive());
			validator.setFieldName(fieldName);
			return validator;
		}
	}

	public static class FieldExpressionFactory implements
			AnnotationValidatorFactory<Expression, FieldExpressionValidator> {

		@Override
		public FieldExpressionValidator createValidator(Expression anno, String fieldName,
				Class<?> fieldClazz) {
			FieldExpressionValidator validator = new FieldExpressionValidator();
			validator.setExpression(anno.value());
			validator.setMessage(anno.message());
			validator.setFieldName(fieldName);
			return validator;
		}

	}

	public static class IntRangeFactory implements
			AnnotationValidatorFactory<IntRange, IntRangeFieldValidator> {

		@Override
		public IntRangeFieldValidator createValidator(IntRange anno, String fieldName,
				Class<?> fieldClazz) {
			IntRangeFieldValidator validator = new IntRangeFieldValidator();
			validator.setMin(anno.min());
			validator.setMax(anno.max());
			validator.setMessage(anno.message());
			validator.setFieldName(fieldName);
			return validator;
		}
	}

	public static class LongRangeFactory implements
			AnnotationValidatorFactory<LongRange, LongRangeFieldValidator> {

		@Override
		public LongRangeFieldValidator createValidator(LongRange anno, String fieldName,
				Class<?> fieldClazz) {
			LongRangeFieldValidator validator = new LongRangeFieldValidator();
			validator.setMin(anno.min());
			validator.setMax(anno.max());
			validator.setMessage(anno.message());
			validator.setFieldName(fieldName);
			return validator;
		}
	}

	public static class RegexFactory implements
			AnnotationValidatorFactory<Regex, RegexFieldValidator> {

		@Override
		public RegexFieldValidator createValidator(Regex anno, String fieldName, Class<?> fieldClazz) {
			RegexFieldValidator validator = new RegexFieldValidator();
			validator.setExpression(anno.value());
			validator.setTrim(anno.trim());
			validator.setCaseSensitive(anno.caseSensitive());
			validator.setMessage(anno.message());
			validator.setFieldName(fieldName);
			return validator;
		}
	}

	public static class RequiredFactory implements
			AnnotationValidatorFactory<Required, RequiredFieldValidator> {

		@Override
		public RequiredFieldValidator createValidator(Required anno, String fieldName,
				Class<?> fieldClazz) {
			RequiredFieldValidator validator = new RequiredFieldValidator();
			validator.setMessage(anno.message());
			validator.setFieldName(fieldName);
			return validator;
		}
	}

	public static class RequiredStringFactory implements
			AnnotationValidatorFactory<RequiredString, RequiredStringValidator> {

		@Override
		public RequiredStringValidator createValidator(RequiredString anno, String fieldName,
				Class<?> fieldClazz) {
			RequiredStringValidator validator = new RequiredStringValidator();
			validator.setMessage(anno.message());
			validator.setFieldName(fieldName);
			return validator;
		}
	}

	public static class StringLengthFactory implements
			AnnotationValidatorFactory<StringLength, StringLengthFieldValidator> {

		@Override
		public StringLengthFieldValidator createValidator(StringLength anno, String fieldName,
				Class<?> fieldClazz) {
			StringLengthFieldValidator validator = new StringLengthFieldValidator();
			validator.setMinLength(anno.minLength());
			validator.setMaxLength(anno.maxLength());
			validator.setMessage(anno.message());
			validator.setFieldName(fieldName);
			return validator;
		}
	}

	public static class URLFactory implements AnnotationValidatorFactory<URL, URLValidator> {

		@Override
		public URLValidator createValidator(URL anno, String fieldName, Class<?> fieldClazz) {
			URLValidator validator = new URLValidator();
			validator.setMessage(anno.message());
			validator.setFieldName(fieldName);
			return validator;
		}
	}

	public static class VisitorFactory implements
			AnnotationValidatorFactory<Visitor, VisitorFieldValidator> {

		public void analyzeFields(Class<?> clazz, ValidatorChain chain) {
			do {
				for (Field field : clazz.getDeclaredFields()) {
					int modifiers = field.getModifiers();
					// fastbson will not process static and transient field
					if (Modifier.isStatic(modifiers) || Modifier.isTransient(modifiers))
						continue;
					// for public field, fastbson will get/set field directly
					Annotation[] annotations = field.getAnnotations();
					for (int i = 0; i < annotations.length; i++) {
						Annotation fieldAnnotation = annotations[i];
						Validator validatorAnnotation = fieldAnnotation.annotationType().getAnnotation(
								Validator.class);
						if (validatorAnnotation != null) {
							Class<? extends AnnotationValidatorFactory> factoryClass = validatorAnnotation
									.factory();
							AnnotationValidatorFactory factory;
							try {
								factory = factoryClass.newInstance();
							} catch (Exception e) {
								throw new ValidationRuntimeException(e);
							}
							com.meidusa.venus.validate.validator.Validator validator = factory
									.createValidator(fieldAnnotation, field.getName(),
											field.getClass());
							if (validator != null) {
								chain.addValidator(validator);
							}
						} else {
							throw new ValidationRuntimeException(
									"Validator annotation not specified for user-defined annotation");
						}

					}
				}
				// the while clause is to make sure to get the super class fiedl
				// as
				// well
			} while ((clazz = clazz.getSuperclass()) != Object.class);
		}

		@Override
		public VisitorFieldValidator createValidator(Visitor annotation, String fieldName,
				Class<?> fieldClazz) {
			VisitorFieldValidator validator = new VisitorFieldValidator();
			Visitor anno = (Visitor) annotation;
			validator.setMessage(anno.message());
			validator.setFieldName(fieldName);
			ValidatorChain chain = new VenusValidatorChain();
			Annotation[] classAnnotation = fieldClazz.getDeclaredAnnotations();
			if (classAnnotation != null) {
				for (int i = 0; i < classAnnotation.length; i++) {
					if (classAnnotation[i].getClass() == Expression.class) {
						ExpressionValidator exprValidator = new ExpressionValidator();
						exprValidator.setExpression(((Expression) classAnnotation[i]).value());
						exprValidator.setMessage(((Expression) classAnnotation[i]).message());
						chain.addValidator(validator);
					}
				}
			}
			this.analyzeFields(fieldClazz, chain);
			validator.setInternalValidatorChain(chain);
			return validator;
		}
	}

}
