package com.meidusa.toolkit.net.packet;

import java.io.UnsupportedEncodingException;
import java.lang.reflect.Constructor;
import java.nio.ByteBuffer;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang.builder.ToStringBuilder;

/**
 * @author struct
 */
@SuppressWarnings("unchecked")
public abstract class AbstractPacket<T extends AbstractPacketBuffer> implements
		Packet {
	private static final long serialVersionUID = 1L;
	private static Map<Class,Constructor> intTypeMap = new HashMap<Class,Constructor>();
	private static Map<Class,Constructor> byteTypeMap = new HashMap<Class,Constructor>();
	public void init(byte[] buffer) {
		T packetBuffer = constractorBuffer(buffer);
		packetBuffer.init();
		init(packetBuffer);
	}
	
	protected abstract void initHead(T buffer);

	/**
	 * 分析数据包(分析包头+数据区域,分析完包头以后应该将Buffer的postion设置到数据区)
	 */
	public void init(T buffer){
		initHead(buffer);
		readBody(buffer);
		afterInit(buffer);
	}

	protected abstract void readBody(T buffer);

	/**
	 * 做完初始化以后
	 */
	protected void afterInit(T buffer) {
	}

	public ByteBuffer toByteBuffer() {
		try {
			int bufferSize = calculatePacketSize();
			T packetBuffer = constractorBuffer(bufferSize);
			packetBuffer.init();
			return toBuffer(packetBuffer).toByteBuffer();
		} catch (UnsupportedEncodingException e) {
			return null;
		}
	}

	public byte[] toByteArray(){
		try {
			int bufferSize = calculatePacketSize();
			T packetBuffer = constractorBuffer(bufferSize);
			packetBuffer.init();
			toBuffer(packetBuffer);
			byte[] result =  new byte[packetBuffer.position];
			System.arraycopy(packetBuffer.data, 0, result, 0, packetBuffer.position);
			return result;
		} catch (UnsupportedEncodingException e) {
			return null;
		}
	}
	
	protected T constractorBuffer(int bufferSize) {
		T buffer = null;
		try {
			Constructor<T> constractor = intTypeMap.get(getPacketBufferClass());
			if(constractor == null){
				synchronized (intTypeMap) {
					constractor = intTypeMap.get(getPacketBufferClass());
					if(constractor == null){
						constractor = getPacketBufferClass().getConstructor(int.class);
						intTypeMap.put(getPacketBufferClass(), constractor);
					}
				}
			}
			
			buffer = constractor.newInstance(bufferSize);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return buffer;
	}

	/**
	 * <pre>
	 *  该方法调用了{@link #write2Buffer(PacketBuffer)} 写入到指定的buffer， 
	 *  并且调用了{@link #afterPacketWritten(PacketBuffer)}
	 * </pre>
	 */
	private T toBuffer(T buffer) throws UnsupportedEncodingException {
		write2Buffer(buffer);
		afterPacketWritten(buffer);
		return buffer;
	}

	/**
	 * 包含头的消息封装
	 */
	protected void write2Buffer(T buffer)
			throws UnsupportedEncodingException{
		writeHead(buffer);
		writeBody(buffer);
	}

	protected abstract void writeBody(T buffer) throws UnsupportedEncodingException ;
	protected abstract void writeHead(T buffer);

	/**
	 * <pre>
	 * 写完之后一定需要调用这个方法，buffer的指针位置指向末尾的下一个位置（包总长度位置）。
	 * 这儿一般是计算数据包总长度,或者其他需要数据包写完才能完成的数据
	 * </pre>
	 */
	protected abstract void afterPacketWritten(T buffer);

	/**
	 * 估算packet的大小，估算的太大浪费内存，估算的太小会影响性能
	 */
	protected abstract int calculatePacketSize();

	protected T constractorBuffer(byte[] buffer) {
		T packetbuffer = null;
		try {
			
			Constructor<T> constractor = byteTypeMap.get(getPacketBufferClass());
			if(constractor == null){
				synchronized (byteTypeMap) {
					constractor = byteTypeMap.get(getPacketBufferClass());
					if(constractor == null){
						constractor = getPacketBufferClass().getConstructor(byte[].class);
						byteTypeMap.put(getPacketBufferClass(), constractor);
					}
				}
			}
			
			packetbuffer = constractor.newInstance(buffer);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return packetbuffer;
	}

	protected abstract Class<T> getPacketBufferClass();

	public String toString() {
		return ToStringBuilder.reflectionToString(this);
	}

}
