/**
 * <pre>
 * 	This program is free software; you can redistribute it and/or modify it under the terms of 
 * the GNU AFFERO GENERAL PUBLIC LICENSE as published by the Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version. 
 * 
 * 	This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * See the GNU AFFERO GENERAL PUBLIC LICENSE for more details. 
 * 	You should have received a copy of the GNU AFFERO GENERAL PUBLIC LICENSE along with this program; 
 * if not, write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * </pre>
 */
package com.meidusa.toolkit.net;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.nio.channels.ClosedSelectorException;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.ServerSocketChannel;
import java.nio.channels.SocketChannel;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.meidusa.toolkit.net.factory.FrontendConnectionFactory;
import com.meidusa.toolkit.net.util.LoopingThread;
import com.meidusa.toolkit.util.IdGenerator;

public class ConnectionAcceptor extends LoopingThread {
    private static final Logger LOGGER = LoggerFactory.getLogger(ConnectionAcceptor.class);
    private static final IdGenerator ID_GENERATOR = new IdGenerator();

    private int port;
    private Selector selector;
    private ServerSocketChannel serverChannel;
    private FrontendConnectionFactory connectionFactory;
    protected ConnectionManager[] processors;
    
    /**
     * 每个ConnectionManager的executor线程数量,默认 CPU core 数量
     */
    private int executorSize = Runtime.getRuntime().availableProcessors();
    private int nextProcessor;
    private long acceptCount;

    public ConnectionAcceptor(){
    	
	}

    public ConnectionAcceptor(String name, int port, FrontendConnectionFactory factory) {
        super.setName(name);
        this.port = port;
        this.connectionFactory = factory;
    }
    
    public FrontendConnectionFactory getConnectionFactory() {
		return connectionFactory;
	}

	public void setConnectionFactory(FrontendConnectionFactory connectionFactory) {
		this.connectionFactory = connectionFactory;
	}

	public int getExecutorSize() {
		return executorSize;
	}

	public void setExecutorSize(int executorSize) {
		this.executorSize = executorSize;
	}

	public void setPort(int port) {
		this.port = port;
	}

	public int getPort() {
        return port;
    }

    public long getAcceptCount() {
        return acceptCount;
    }

    public void setProcessors(ConnectionManager[] processors) {
        this.processors = processors;
    }

    public void initProcessors() throws IOException{
    	if(processors == null){
        	processors = new ConnectionManager[Runtime.getRuntime().availableProcessors()];
        	for(int i=0;i<processors.length;i++){
        		processors[i] = new ConnectionManager(this.getName()+"-Manager-"+i,getExecutorSize());
        		processors[i].start();
        	}
        }
    }
    
    protected void willStart() {
    	try {
	        initProcessors();
	        this.selector = Selector.open();
	        this.serverChannel = ServerSocketChannel.open();
	        this.serverChannel.socket().bind(new InetSocketAddress(port));
	        /*Level level = LOGGER.getLevel();
	        LOGGER.setLevel(Level.INFO);*/
	        LOGGER.warn("Server listening on " +  this.serverChannel.socket().getLocalSocketAddress() + ".");
	        //LOGGER.setLevel(level);
	        
	        
	        this.serverChannel.configureBlocking(false);
	        this.serverChannel.register(selector, SelectionKey.OP_ACCEPT);
       
	        super.willStart();
    	} catch (IOException ioe) {
    		LOGGER.error("Failure listening to socket on port '" + port + "'.", ioe);
    		this._running = false;
            System.exit(-1);
        }
    }
    
    @Override
    public void iterate() throws Throwable {
    	++acceptCount;
        selector.select(1000L);
        Set<SelectionKey> keys = selector.selectedKeys();
        try {
            for (SelectionKey key : keys) {
                if (key.isValid() && key.isAcceptable()) {
                    accept();
                } else {
                    key.cancel();
                }
            }
        } finally {
            keys.clear();
        }
            
    }

    private void accept() {
        SocketChannel channel = null;
        try {
            channel = serverChannel.accept();
            channel.configureBlocking(false);
            FrontendConnection c = connectionFactory.make(channel);
            c.setAccepted(true);
            c.setId(ID_GENERATOR.nextId());
            ConnectionManager processor = nextProcessor();
            c.setProcessor(processor);
            processor.postRegister(c);
        } catch (Throwable e) {
            closeChannel(channel);
            LOGGER.warn(getName(), e);
        }
    }

    private ConnectionManager nextProcessor() {
        if (++nextProcessor == processors.length) {
            nextProcessor = 0;
        }
        return processors[nextProcessor];
    }

    private static void closeChannel(SocketChannel channel) {
        if (channel == null) {
            return;
        }
        Socket socket = channel.socket();
        if (socket != null) {
            try {
                socket.close();
            } catch (IOException e) {
            }
        }
        try {
            channel.close();
        } catch (IOException e) {
        }
    }

    protected void handleIterateFailure(Throwable e) {
    	if(e instanceof ClosedSelectorException){
    		super.handleIterateFailure(e);
    	}else if(e instanceof InterruptedException){
    		super.handleIterateFailure(e);
    	}else{
    		LOGGER.error(this.getName(), e);
    	}
    }

    public void finalize() throws Throwable{
    	/*Level level = LOGGER.getLevel();
        LOGGER.setLevel(Level.INFO);*/
        LOGGER.warn("Server port: " +  this.serverChannel.socket().getLocalSocketAddress() + " closed!");
        //LOGGER.setLevel(level);
    	super.finalize();
    }
}
