/**
 * <pre>
 * 	This program is free software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version. 
 * 
 * 	This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * See the GNU General Public License for more details. 
 * 	You should have received a copy of the GNU General Public License along with this program; 
 * if not, write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * </pre>
 */
package com.meidusa.toolkit.net.packet;

import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;

import com.meidusa.toolkit.common.util.CharsetCache;
import com.meidusa.toolkit.net.IllegalPacketException;
import com.meidusa.toolkit.net.util.StringUtil;

/**
 * ฺ ͡socketҿԸݰͷϢByteBuffer ֽͨѶ
 * 
 * @author <a href=mailto:piratebase@sina.com>Struct chen</a>
 */
public abstract class GenericIOPacketBuffer extends AbstractPacketBuffer {
	public abstract int getHeadSize();
	public GenericIOPacketBuffer(byte[] buf) {
		super(buf);
	}

	public GenericIOPacketBuffer(int size) {
		super(size);
	}

	final void dumpHeader() {
		for (int i = 0; i < getHeadSize(); i++) {
			String hexVal = Integer.toHexString(readByte(i) & 0xff);

			if (hexVal.length() == 1) {
				hexVal = "0" + hexVal; //$NON-NLS-1$
			}
			System.out.print(hexVal + " "); //$NON-NLS-1$
		}
	}



	public final byte[] getBytes(int len) {
		byte[] b = new byte[len];
		this.readBytes(b);
		return b;
	}

	public byte[] getBytes(int offset, int len) {
		byte[] dest = new byte[len];
		System.arraycopy(this.data, offset, dest, 0, len);
		return dest;
	}

	/**
	 * ʾҪȡݵĳ
	 * 
	 * @return
	 */
	public abstract int readFieldLength();

	public final int readInt() {
		int i = this.position;
		this.position = this.position + 4;
		return getIntB(this.data,i);
	}

	static private int makeInt(byte b3, byte b2, byte b1, byte b0) {
		return (int)((((b3 & 0xff) << 24) |
			      ((b2 & 0xff) << 16) |
			      ((b1 & 0xff) <<  8) |
			      ((b0 & 0xff) <<  0)));
	}
	
	static int getIntB(byte[] bb, int bi) {
		return makeInt(bb[bi + 0],
				bb[bi + 1],
				bb[bi + 2],
				bb[bi + 3]);
	}
	
	public final long readLong() {
		int i = this.position;
		this.position = this.position + 8;
		return getLongB(this.data,i);
	}
	
	static private long makeLong(byte b7, byte b6, byte b5, byte b4,
			 byte b3, byte b2, byte b1, byte b0)
	{
	return ((((long)b7 & 0xff) << 56) |
		(((long)b6 & 0xff) << 48) |
		(((long)b5 & 0xff) << 40) |
		(((long)b4 & 0xff) << 32) |
		(((long)b3 & 0xff) << 24) |
		(((long)b2 & 0xff) << 16) |
		(((long)b1 & 0xff) <<  8) |
		(((long)b0 & 0xff) <<  0));
	}
	
	static long getLongB(byte[] bb, int bi) {
	return makeLong(bb[bi + 0],
			bb[bi + 1],
			bb[bi + 2],
			bb[bi + 3],
			bb[bi + 4],
			bb[bi + 5],
			bb[bi + 6],
			bb[bi + 7]);
	}

	public int writeLengthCodedBytes(byte[] ab) {
    	ensureCapacity(ab.length+4);
    	this.writeInt(ab==null?0:ab.length);
    	if(ab != null && ab.length > 0){
    		return writeBytes(ab, 0, ab.length);
    	}else{
    		return 0;
    	}
    }
	
	public byte[] readLengthCodedBytes() {
		int count = readInt();
		if(count ==0){
			return new byte[0];
		}else{
			byte[] bts = new byte[count];
			if(count > this.remaining()){
				throw new IllegalPacketException("packet content size error: "+count +">" +this.remaining()+" remaining");
			}
			this.readBytes(bts, 0, count);
			return bts;
		}
    }
	
	public final String readLengthCodedString(Charset charset) {
		int fieldLength = (int) readFieldLength();
		
		if (fieldLength == 0) {
			return null;
		}
		
		if(fieldLength> this.remaining()){
			throw new IllegalPacketException("fieldLength error Buffer.Remaining="+remaining()+" ,but need size="+ fieldLength);
		}
		byte[] bytes = getBytes(fieldLength);
		if (charset != null) {
			return new String(bytes,charset);
		} else {
			return new String(bytes);
		}
	}
	
	public final String readLengthCodedString(String encoding) {
		Charset charset = CharsetCache.getCharset(encoding);
		return readLengthCodedString(charset);
	}

	public String toString() {
		return StringUtil.dumpAsHex(this.data,getPosition());
	}

	public String toSuperString() {
		return super.toString();
	}

	public abstract void writeFieldLength(int length);

	public final void writeInt(int i) {
		ensureCapacity(4);
		this.data[position++] = (byte)(i >> 24);
		this.data[position++] = (byte)(i >> 16);
		this.data[position++] = (byte)(i >> 8);
		this.data[position++] = (byte)(i >> 0);
	}

    public short readShort(){
    	int i = this.position;
    	this.position +=2;
    	return getShortB(this.data,i);
    }
    
    static private short makeShort(byte b1, byte b0) {
    	return (short)((b1 << 8) | (b0 & 0xff));
    }

   
    static short getShortB(byte[] bb, int bi) {
	return makeShort(bb[bi + 0],
					 bb[bi + 1]);
    }
    
    public void writeShort(short i) {
        ensureCapacity(2);
		this.data[position++] = (byte)(i >> 8);
		this.data[position++] = (byte)(i >> 0);
    }
    
	public final void writeLengthCodedString(String s, String encoding) {
		Charset charset = CharsetCache.getCharset(encoding);
		writeLengthCodedString(s,charset);
	}

	public final void writeLengthCodedString(String s, Charset charset) {
		if (s != null) {
			byte[] b;
			b = s.getBytes(charset);
			ensureCapacity(b.length + 9);
			this.writeFieldLength(b.length);
			this.writeBytes(b);
		} else {
			this.writeFieldLength(0);
		}
	}
	
	public final void writeLong(long i) {
		ensureCapacity(8);
		this.data[position++] = (byte)(i >> 56);
		this.data[position++] = (byte)(i >> 48);
		this.data[position++] = (byte)(i >> 40);
		this.data[position++] = (byte)(i >> 32);
		this.data[position++] = (byte)(i >> 24);
		this.data[position++] = (byte)(i >> 16);
		this.data[position++] = (byte)(i >> 8);
		this.data[position++] = (byte)(i >> 0);
	}

	// Write null-terminated string
	public final void writeString(String s) {
		try {
			writeString(s, null);
		} catch (UnsupportedEncodingException e) {
		}
	}

	public final void writeString(String s, String encoding)
			throws UnsupportedEncodingException {
		byte[] bytes = null;
		if (encoding == null) {
			bytes = s.getBytes();
		} else {
			bytes = s.getBytes(encoding);
		}
		ensureCapacity(bytes.length + 1 + 8);
		this.writeFieldLength(bytes.length);
		this.writeBytes(bytes);
	}

	public static void main(String[] args) {
		GenericIOPacketBuffer buffer = new GenericIOPacketBuffer(34) {
			@Override
			public int getHeadSize() {
				return 0;
			}
			@Override
			public int readFieldLength() {
				return 0;
			}
			@Override
			public void writeFieldLength(int length) {
			}
		};

		buffer.writeLong(44);
		// System.out.println(buffer.dump(34));
		ByteBuffer bytbuffer = ByteBuffer.allocate(32);
		bytbuffer.putLong(44);
		System.out.println(StringUtil.dumpAsHex(bytbuffer.array(), 32));
	}
}
