
/**
 * A collection of Atlassian JavaScript UI components.
 *
 * @module AJS
 * @requires jQuery
 */
if (typeof jQuery != "undefined") {

    /**
     * AJS contains utility methods, used by various components. It also provides the namespacing for all AUI components. 
     *
     * @class AJS
     * @requires jQuery
     */
    var AJS = (function () {
        var included = [];

        function escapeHtml_replacement(specialChar) {
            switch (specialChar) {
                case "<": return "&lt;";
                case ">": return "&gt;";
                case '&': return "&amp;";
                case "'": return "&#39;";
                case "`": return "&#96;";
                default : return "&quot;";
            }
        }
        var ESCAPE_HTML_SPECIAL_CHARS = /[&"'<>`]/g;

        var res = {
            /**
             * Version number to allow for rough backwards compatibility via conditionals
             * NOTE: Don't change. Generated by the Maven at build.
             * @property version
             */
            version: "5.1-PDL-OLD",

            /**
             * Parameters are loaded from the DOM on page load.
             * @property params
             */
            params: {},
            /**
            * Returns an HTMLElement reference.
            * @method $
            * @param {String | HTMLElement |Array} el Accepts a string to use as an ID for getting a DOM reference, an actual DOM reference, or an Array of IDs and/or HTMLElements.
            * @return {HTMLElement | Array} A DOM reference to an HTML element or an array of HTMLElements.
            */
            $: jQuery,

            /**
             * Logs the given object to the console.
             * @param obj object to log
             */
            log: function() {
                if (typeof console != "undefined" && console.log){
                    Function.prototype.apply.apply(console.log, [console, arguments]);
                }
            },
            /**
             * Placeholder function for i18n which should be translated by the JsI18nTransformer. This has been
             * added so that for whatever reason if its not translated, the scripts don't blow up.
             */
            I18n : {
                getText: function(key) {
                    return key;
                }
            },
            /**
             * Prevent further handling of an event. Returns false, which you should use as the return value of your event handler:
             *     return AJS.stopEvent(e);
             * @param e jQuery event
             */
            stopEvent: function(e) {
                e.stopPropagation();
                return false; // required for JWebUnit pop-up links to work properly
            },
            include: function (url) {
                if (!this.contains(included, url)) {
                    included.push(url);
                    var s = document.createElement("script");
                    s.src = url;
                    this.$("body").append(s);
                }
            },
            /**
            * Shortcut function to toggle class name of an element.
            * @method toggleClassName
            * @param {String | HTMLElement} element The HTMLElement or an ID to toggle class name on.
            * @param {String} className The class name to remove or add.
            */
            toggleClassName: function (element, className) {
                if (!(element = this.$(element))) {
                    return;
                }
                element.toggleClass(className);
            },
            /**
             * Shortcut function adds or removes "hidden" classname to an element based on a passed boolean.
             * @method setVisible
             * @param {String | HTMLElement} element The HTMLElement or an ID to show or hide.
             * @param {boolean} show true to show, false to hide
             */
            setVisible: function (element, show) {
                if (!(element = this.$(element))) {
                    return;
                }
                var $ = this.$; // aliased for use inside function below
                $(element).each(function () {
                    var isHidden = $(this).hasClass("hidden");
                    if (isHidden && show) {
                        $(this).removeClass("hidden");
                    }
                    else if (!isHidden && !show) {
                        $(this).addClass("hidden");
                    }
                });
            },
            /**
             * Shortcut function adds or removes "current" classname to an element based on a passed boolean.
             * @param {String | HTMLElement} element The HTMLElement or an ID to show or hide.
             * @param {boolean} show true to add "current" class, false to remove
             */
            setCurrent: function (element, current) {
                if (!(element = this.$(element))) {
                    return;
                }

                if (current)
                    element.addClass("current");
                else
                    element.removeClass("current");
            },
            /**
             * Shortcut function to see if passed element is currently visible on screen.
             * @method isVisible
             * @param {String | HTMLElement} element The HTMLElement or an jQuery selector to check.
             */
            isVisible: function (element) {
                return !this.$(element).hasClass("hidden");
            },

            /**
             * Shortcut function to see if passed element is truncated/clipped, eg. with text-overflow: ellipsis
             * @method isClipped
             * @param {String | HTMLElement} element The HTMLElement or an jQuery selector to check.
             */
            isClipped: function (el) {
                el = AJS.$(el);
                return ( el.prop("scrollWidth") > el.prop("clientWidth") );            
            },
            
            /**
            * Find parameters in the DOM and store them in the ajs.params object.
            */
            populateParameters: function() {
                var ajs = this;
                this.$(".parameters input").each(function () {
                    var value = this.value,
                        id = this.title || this.id;
                    if (ajs.$(this).hasClass("list")) {
                        if (ajs.params[id]) {
                            ajs.params[id].push(value);
                        } else {
                            ajs.params[id] = [value];
                        }
                    } else {
                        ajs.params[id] = (value.match(/^(tru|fals)e$/i) ? value.toLowerCase() == "true" : value);
                    }
                });
            },
            
            /**
            * Adds functions to the list of methods to be run on initialisation. Wraps
            * error handling around the provided function so its failure won't prevent
            * other init functions running.
            * @method toInit
            * @param {Function} func Function to be call on initialisation.
            * @return AJS object.
            */
            toInit: function (func) {
                var ajs = this;
                this.$(function () {
                    try {
                        func.apply(this, arguments);
                    } catch(ex) {
                        ajs.log("Failed to run init function: " + ex + "\n" + func.toString());
                    }
                });
                return this;
            },

            /**
            * Finds the index of an element in the array.
            * @method indexOf
            * @param item Array element which will be searched.
            * @param fromIndex (optional) the index from which the item will be searched. Negative values will search from the
            * end of the array.
            * @return a zero based index of the element.
            */
            indexOf: function (array, item, fromIndex) {
                var length = array.length;
                if (fromIndex == null) {
                  fromIndex = 0;
                } else {
                    if (fromIndex < 0) {
                      fromIndex = Math.max(0, length + fromIndex);
                    }
                }
                for (var i = fromIndex; i < length; i++) {
                  if (array[i] === item) return i;
                }
                return -1;
            },
            /**
            * Looks for an element inside the array.
            * @method contains
            * @param item Array element which will be searched.
            * @return {Boolean} Is element in array.
            */
            contains: function (array, item) {
                return this.indexOf(array, item) > -1;
            },
            /**
            * Replaces tokens in a string with arguments, similar to Java's MessageFormat.
            * Tokens are in the form {0}, {1}, {2}, etc.
            * @method format
            * @param message the message to replace tokens in
            * @param arg (optional) replacement value for token {0}, with subsequent arguments being {1}, etc.
            * @return {String} the message with the tokens replaced
            * @usage AJS.format("This is a {0} test", "simple");
            */
            format: function (message) {
                var token = /^((?:(?:[^']*'){2})*?[^']*?)\{(\d+)\}/, // founds numbers in curly braces that are not surrounded by apostrophes
                    apos = /'(?!')/g; // founds "'", bot not "''"
                // we are caching RegExps, so will not spend time on recreating them on each call
                AJS.format = function (message) {
                    var args = arguments,
                        res = "",
                        match = message.match(token);
                    while (match) {
                         message = message.substring(match[0].length);
                         res += match[1].replace(apos, "") + (args.length > ++match[2] ? args[match[2]] : "");
                         match = message.match(token);
                    }
                    return res += message.replace(apos, "");
                };
                return AJS.format.apply(AJS, arguments);
            },
            /**
            * Includes firebug lite for debugging in IE. Especially in IE.
            * @method firebug
            * @usage Type in addressbar "javascript:alert(AJS.firebug());"
            */
            firebug: function () {
                var script = this.$(document.createElement("script"));
                script.attr("src", "http://getfirebug.com/releases/lite/1.2/firebug-lite-compressed.js");
                this.$("head").append(script);
                (function () {
                    if (window.firebug) {
                        firebug.init();
                    } else {
                        setTimeout(arguments.callee, 0);
                    }
                })();
            },
            /**
             * Clones the element specified by the selector and removes the id attribute
             * @param selector a jQuery selector
             */
            clone : function(selector) {
                return AJS.$(selector).clone().removeAttr("id");
            },
            /**
            * Compare two strings in alphanumeric way
            * @method alphanum
            * @param {String} a first string to compare
            * @param {String} b second string to compare
            * @return {Number(-1|0|1)} -1 if a < b, 0 if a = b, 1 if a > b
            * @usage a.sort(AJS.alphanum)
            */
            alphanum: function (a, b) {
                a = (a + "").toLowerCase();
                b = (b + "").toLowerCase();
                var chunks = /(\d+|\D+)/g,
                    am = a.match(chunks),
                    bm = b.match(chunks),
                    len = Math.max(am.length, bm.length);
                for (var i = 0; i < len; i++) {
                    if (i == am.length) {
                        return -1;
                    }
                    if (i == bm.length) {
                        return 1;
                    }
                    var ad = parseInt(am[i], 10),
                        bd = parseInt(bm[i], 10);
                    if (ad == am[i] && bd == bm[i] && ad != bd) {
                        return (ad - bd) / Math.abs(ad - bd);
                    }
                    if ((ad != am[i] || bd != bm[i]) && am[i] != bm[i]) {
                        return am[i] < bm[i] ? -1 : 1;
                    }
                }
                return 0;
            },
            /**
             * Covers screen with semitransparent DIV
             * @param useShim
             */
            dim: function (useShim) {
                if (!AJS.dim.$dim) {
                    AJS.dim.$dim = AJS("div").addClass("aui-blanket");
                    if (AJS.$.browser.msie) {
                        AJS.dim.$dim.css({width: "200%", height: Math.max(AJS.$(document).height(), AJS.$(window).height()) + "px"});
                    }
                    AJS.$("body").append(AJS.dim.$dim);

                    AJS.hasFlash = false;
                    var matchHostname = /^[^:]*:\/*[^/]*|/;
                    var hostname = matchHostname.exec(location.href)[0];

                    // Even if we do not want to use a shim, we are going to override it in the case of flash being on the page.
                    // Flash will sit ontop of our blanket unless wmode is set to opaque in the object/embed tag...
                    if (AJS.$.browser.msie && typeof AJS.hasFlash === "undefined" && useShim === false) {
                        AJS.$("object, embed, iframe").each(function () {
                            if (this.nodeName.toLowerCase() === "iframe") {
                                // Don't attempt to access $(this).contents() unless the iframe has
                                // the same hostname as window.location. If not, be pessimistic and
                                // assume the document contains flash objects.
                                if (matchHostname.exec(this.src)[0] === hostname && AJS.$(this).contents().find("object, embed").length === 0) {
                                    return true; // continue loop
                                }
                            }
                            AJS.hasFlash = true;
                            return false; // break loop
                        });
                    }

                    // Add IFrame shim
                    if (AJS.$.browser.msie && (useShim !== false || AJS.hasFlash)) {
                        AJS.dim.shim = AJS.$('<iframe frameBorder="0" class="aui-blanket-shim" src="javascript:false;"/>');
                        AJS.dim.shim.css({height: Math.max(AJS.$(document).height(), AJS.$(window).height()) + "px"});
                        AJS.$("body").append(AJS.dim.shim);
                    }

                    // IE needs the overflow on the HTML element so scrollbars are hidden
                    if (AJS.$.browser.msie && parseInt(AJS.$.browser.version,10) < 8) {
                        AJS.dim.cachedOverflow = AJS.$("html").css("overflow");
                        AJS.$("html").css("overflow", "hidden");
                    } else {
                        AJS.dim.cachedOverflow = AJS.$("body").css("overflow");
                        AJS.$("body").css("overflow", "hidden");
                    }
                }
            },
            /**
             * Removes semitransparent DIV
             * @see AJS.dim
             */
            undim: function() {
                if (AJS.dim.$dim) {
                    AJS.dim.$dim.remove();
                    AJS.dim.$dim = null;
                    if (AJS.dim.shim) {
                        AJS.dim.shim.remove();
                    }

                    // IE needs the overflow on the HTML element so scrollbars are hidden
                    if (AJS.$.browser.msie && parseInt(AJS.$.browser.version,10) < 8) {
                        AJS.$("html").css("overflow",  AJS.dim.cachedOverflow);
                    } else {
                        AJS.$("body").css("overflow",  AJS.dim.cachedOverflow);
                    }

                    // Safari bug workaround
                    if (AJS.$.browser.safari) {
                        var top = AJS.$(window).scrollTop();
                        AJS.$(window).scrollTop(10 + 5 * (top == 10)).scrollTop(top);
                    }
                }
            },
            onTextResize: function (f) {
                if (typeof f == "function") {
                    if (AJS.onTextResize["on-text-resize"]) {
                        AJS.onTextResize["on-text-resize"].push(function (emsize) {
                            f(emsize);
                        });
                    } else {
                        var em = AJS("div");
                        em.css({
                            width: "1em",
                            height: "1em",
                            position: "absolute",
                            top: "-9999em",
                            left: "-9999em"
                        });
                        this.$("body").append(em);
                        em.size = em.width();
                        setInterval(function () {
                            if (em.size != em.width()) {
                                em.size = em.width();
                                for (var i = 0, ii = AJS.onTextResize["on-text-resize"].length; i < ii; i++) {
                                    AJS.onTextResize["on-text-resize"][i](em.size);
                                };
                            }
                        }, 0);
                        AJS.onTextResize.em = em;
                        AJS.onTextResize["on-text-resize"] = [function (emsize) {
                            f(emsize);
                        }];
                    }
                }
            },
            unbindTextResize: function (f) {
                for (var i = 0, ii = AJS.onTextResize["on-text-resize"].length; i < ii; i++) {
                    if (AJS.onTextResize["on-text-resize"][i] == f) {
                        return AJS.onTextResize["on-text-resize"].splice(i, 1);
                    }
                }
            },
            /**
             * Similar to Javascript's in-built escape() function, but where the built-in escape()
             * might encode unicode charaters as %uHHHH, this function will leave them as-is.
             *
             * NOTE: this function does not do html-escaping, see AJS.escapeHtml()
             */
            escape: function (string) {
                return escape(string).replace(/%u\w{4}/gi, function (w) {
                    return unescape(w);
                });
            },
            /**
             * Sanitise a string for use with innerHTML or as an attribute.
             *
             * @param {String} str
             */
            escapeHtml: function (str) {
                return str.replace(ESCAPE_HTML_SPECIAL_CHARS, escapeHtml_replacement);
            },

            /**
             * Filters a list of entries by a passed search term.
             *
             * Options :
             *   - "keywordsField" - name of entry field containing keywords, default "keywords"
             *   - "ignoreForCamelCase" - ignore search case for camel case, e.g. CB matches Code Block *and* Code block
             *   - "matchBoundary" - match words only at boundary, e.g. link matches "linking" but not "hyperlinks"
             *   - "splitRegex" - regex to split search words, instead of on whitespace
             *
             * @param entries an index array of objects with a "keywords" property
             * @param search one or more words to search on, which may include camel-casing.
             * @param options - optional - specifiy to override default behaviour
             */
            filterBySearch : function(entries, search, options) {
                if (search == "") return [];   // search for nothing, get nothing - up to calling code to handle.

                var $ = this.$;
                var keywordsField = (options && options.keywordsField) || "keywords";
                var camelCaseFlags = (options && options.ignoreForCamelCase) ? "i" : "";
                var boundaryFlag  = (options && options.matchBoundary) ? "\\b" : "";
                var splitRegex = (options && options.splitRegex) || (/\s+/);

                // each word in the input is considered a distinct filter that has to match a keyword in the record
                var filterWords = search.split(splitRegex);
                var filters = [];
                $.each(filterWords, function () {
                  var subfilters = [new RegExp(boundaryFlag + this, "i")]; // anchor on word boundaries
                  if (/^([A-Z][a-z]*){2,}$/.test(this)) { // split camel-case into separate words
                      var camelRegexStr = this.replace(/([A-Z][a-z]*)/g, "\\b$1[^,]*");
                      subfilters.push(new RegExp(camelRegexStr, camelCaseFlags));
                  }
                  filters.push(subfilters);
                });
                var result = [];
                $.each(entries, function () {
                    for (var i = 0; i < filters.length; i++) {
                        var somethingMatches = false;
                        for (var j = 0; j < filters[i].length; j++) {
                            if (filters[i][j].test(this[keywordsField])) {
                                somethingMatches = true;
                                break;
                            }
                        }
                        if (!somethingMatches) return;
                    }
                    result.push(this);
                });
                return result;
            },
            
            drawLogo : function(opts) {
                options = {};
                options = AJS.$.extend(opts, options);
                var scale = options.scaleFactor || 1,
                    fill = options.fill || "#fff",
                    stroke = options.stroke || "#000",
                    width = 400 * scale,
                    height = 40 * scale;
                    strokeWidth = options.strokeWidth || 1;
                if(AJS.$(".aui-logo").size()==0){
                   AJS.$("body").append("<div id='aui-logo' class='aui-logo'><div>"); 
                }
                strokeWidth = options.strokeWidth || 1,
                containerID = options.containerID || ".aui-logo";
                var logoCanvas = Raphael(containerID, width + 50*scale, height + 100*scale);
                var logo = logoCanvas.path("M 0,0 c 3.5433333,-4.7243333 7.0866667,-9.4486667 10.63,-14.173 -14.173,0 -28.346,0 -42.519,0 C -35.432667,-9.4486667 -38.976333,-4.7243333 -42.52,0 -28.346667,0 -14.173333,0 0,0 z m 277.031,28.346 c -14.17367,0 -28.34733,0 -42.521,0 C 245.14,14.173 255.77,0 266.4,-14.173 c -14.17267,0 -28.34533,0 -42.518,0 C 213.25167,0 202.62133,14.173 191.991,28.346 c -14.17333,0 -28.34667,0 -42.52,0 14.17333,-18.8976667 28.34667,-37.7953333 42.52,-56.693 -7.08667,-9.448667 -14.17333,-18.897333 -21.26,-28.346 -14.173,0 -28.346,0 -42.519,0 7.08667,9.448667 14.17333,18.897333 21.26,28.346 -14.17333,18.8976667 -28.34667,37.7953333 -42.52,56.693 -14.173333,0 -28.346667,0 -42.52,0 10.63,-14.173 21.26,-28.346 31.89,-42.519 -14.390333,0 -28.780667,0 -43.171,0 C 42.520733,1.330715e-4 31.889933,14.174867 21.26,28.347 c -42.520624,6.24e-4 -85.039187,-8.13e-4 -127.559,-0.001 11.220667,-14.961 22.441333,-29.922 33.662,-44.883 -6.496,-8.661 -12.992,-17.322 -19.488,-25.983 5.905333,0 11.810667,0 17.716,0 -10.63,-14.173333 -21.26,-28.346667 -31.89,-42.52 14.173333,0 28.346667,0 42.52,0 10.63,14.173333 21.26,28.346667 31.89,42.52 14.173333,0 28.3466667,0 42.52,0 -10.63,-14.173333 -21.26,-28.346667 -31.89,-42.52 14.1733333,0 28.3466667,0 42.52,0 10.63,14.173333 21.26,28.346667 31.89,42.52 14.390333,0 28.780667,0 43.171,0 -10.63,-14.173333 -21.26,-28.346667 -31.89,-42.52 42.51967,0 85.03933,0 127.559,0 10.63033,14.173333 21.26067,28.346667 31.891,42.52 14.17267,0 28.34533,0 42.518,0 -10.63,-14.173333 -21.26,-28.346667 -31.89,-42.52 14.17367,0 28.34733,0 42.521,0 14.17333,18.897667 28.34667,37.795333 42.52,56.693 -14.17333,18.8976667 -28.34667,37.7953333 -42.52,56.693 z");
                console.log(height);
                logo.scale(scale, -scale, 0, 0);
                logo.translate(120 * scale, height);
                logo.attr("fill", fill);
                logo.attr("stroke", stroke);
                logo.attr("stroke-width", strokeWidth);
                                
                
            }
        };
        if (typeof AJS != "undefined") {
            for (var i in AJS) {
                res[i] = AJS[i];
            }
        }
        /**
        * Creates DOM object
        * @method AJS
        * @param {String} element tag name
        * @return {jQuery object}
        * @usage var a = AJS("div");
        */
        var result = function () {
            var res = null;
            if (arguments.length && typeof arguments[0] == "string") {
                res = arguments.callee.$(document.createElement(arguments[0]));
                if (arguments.length == 2) {
                    res.html(arguments[1]);
                }
            }
            return res;
        };
        for (var i in res) {
            result[i] = res[i];
        }
        return result;
    })();

    AJS.$(function () {AJS.populateParameters();});
}

if (typeof console == "undefined") {
    console = {
        messages: [],
        log: function (text) {
            this.messages.push(text);
        },
        show: function () {
            alert(this.messages.join("\n"));
            this.messages = [];
        }
    };
}
else {
    // Firebug console - show not required to do anything.
    console.show = function(){};
}

// Setting Traditional to handle our default param serialisation
// See http://api.jquery.com/jQuery.param/ for more
AJS.$.ajaxSettings.traditional = true;

