define('feature/issues/fields/all-fields', ['zepto', 'backbone'], function ($, Backbone) {
    // Fields we don't want to render generically. Fields we don't care about or fields that have special treatment.
    // Maps are faster to access.
    var blacklist = {
        /* Things we don't care about */
        aggregateprogress: true,
        aggregatetimeestimate: true,
        aggregatetimeoriginalestimate: true,
        aggregatetimespent: true,
        progress: true,
        workratio: true,
        timeestimate: true,
        timeoriginalestimate: true,
        timespent: true,
        /* Things we are trying to hide explicitly */
        assignee: true,
        comment: true,
        description: true,
        priority: true,
        resolution: true,
        status: true,
        summary: true,
        timetracking: true,
        worklog: true,
        /* Things that will be handled in a special way */
        attachment: true,
        issuelinks: true,
        parent: true,
        votes: true,
        watches: true
    };
    var viewRender = {
        group: AtlassianMobile.Templates.JIRA.Issue.drawALabelField,
        issuetype: AtlassianMobile.Templates.JIRA.Issue.drawConstantField,
        number: AtlassianMobile.Templates.JIRA.Issue.drawStringField,
        priority: AtlassianMobile.Templates.JIRA.Issue.drawConstantField,
        project: AtlassianMobile.Templates.JIRA.Issue.drawProjectField,
        resolution: AtlassianMobile.Templates.JIRA.Issue.drawResolutionField,
        securitylevel: AtlassianMobile.Templates.JIRA.Issue.drawSecurityField,
        status: AtlassianMobile.Templates.JIRA.Issue.drawConstantField,
        string: AtlassianMobile.Templates.JIRA.Issue.drawStringField,
        version: AtlassianMobile.Templates.JIRA.Issue.drawALabelField,
        user: AtlassianMobile.Templates.JIRA.Issue.drawUserField
    };
    var arrayRenderer = {
        attachment: AtlassianMobile.Templates.JIRA.Issue.drawAttachmentField,
        component: AtlassianMobile.Templates.JIRA.Issue.drawObjectListField,
        group: AtlassianMobile.Templates.JIRA.Issue.drawObjectListField,
        version: AtlassianMobile.Templates.JIRA.Issue.drawObjectListField,
        user: AtlassianMobile.Templates.JIRA.Issue.drawMultiUserField,
        issuelinks: renderIssueLinks    //can't call the template straight because we need to group the issue links first.

    };

    var arrayStringRenderer = {
        cascadingselect: AtlassianMobile.Templates.JIRA.Issue.drawCasSelField,
        labels: AtlassianMobile.Templates.JIRA.Issue.drawStringListField,
        multicheckboxes: AtlassianMobile.Templates.JIRA.Issue.drawSelectListField,
        multiselect: AtlassianMobile.Templates.JIRA.Issue.drawSelectListField,
        string: AtlassianMobile.Templates.JIRA.Issue.drawStringListField
    };

    //custom function for rendering issue links, groups the issues by issue link type before calling the soy.
    function renderIssueLinks(object){
        var processedIssueLinks = {},
            renderedOutput = "";
        for(i in object.value){
            var thisType;
            if(object.value[i].inwardIssue){
                thisType = object.value[i].type.inward;
            } else {
                thisType = object.value[i].type.outward
            }
            if(!processedIssueLinks[thisType]){
                processedIssueLinks[thisType] = [];
            }
            processedIssueLinks[thisType].push(object.value[i]);
        }
        return  AtlassianMobile.Templates.JIRA.Issue.drawIssueLinksField({value: processedIssueLinks})
    }

    return Backbone.View.extend({
        className: 'all-fields',

        _updateData: function () {
            this.issue = this.model.toJSON();
        },

        initialize: function (options) {
            this._updateData();
            this.listenTo(this.model, 'change:fields', function () {
                this._updateData();
                this.render();
            });
            this.show = false;
        },

        setShow: function (value) {
            this.show = !!value;
            this.$el.toggleClass('expanded', this.show);
        },

        getRenderer: function(key) {
            var renderer;
            var schema = this.issue.schema[key];
            if (!schema) {
                return renderer;
            }
            var type = schema.type;
            // Arrays need to check the item
            if (type === 'array') {
                var item = this.issue.schema[key].items;
                // Array with string needs to check further
                if (item === 'string') {
                    var custom = this.issue.schema[key].custom;
                    if (custom) {
                        var type = custom.split(":")[1];
                        renderer = arrayStringRenderer[type];
                    } else {
                        renderer = arrayStringRenderer[item];
                    }
                } else {
                    // Array renderer
                    renderer = arrayRenderer[item];
                }
            } else {
                // Single value renderer
                renderer = viewRender[type];
            }
            return renderer;
        },

        isSystemOrCustomField: function (key) {
            var schema = this.issue.schema[key];
            if (!schema) {
                return false;
            }
            // If it is a system field
            if (schema.system) {
                return true;
            }
            // If it is a known system CFT
            var customFieldKey = schema.custom;
            if (customFieldKey) {
                var prefix = customFieldKey.split(":")[0];
                if (prefix === "com.atlassian.jira.plugin.system.customfieldtypes") {
                    return true;
                }
            }

            return false;
        },

        render: function () {
            this.$el.empty();
            // Only show if we were told to
            if (!this.show) {
                return this;
            }

            /* 
             * Fields we want to render as a generic list 
             */
            // If it has a name, we can display it
            for (var i = 0, ii = this.issue.screen.length; i < ii; i++) {
                var key = this.issue.screen[i].id;
                // Make sure it isn't in the blacklist
                if (!blacklist[key] && this.isSystemOrCustomField(key)) {
                    // Check to see if the rendered value was returned by the server
                    var renderedValue = this.issue.renderedFields[key];
                    if (!renderedValue || typeof renderedValue != "string") {
                        // If the server didn't render it or it wasn't a string, then try and find a renderer that can draw it
                        var renderer = this.getRenderer(key);

                        // Render with the renderer we found if we found it
                        if (renderer) {
                            if (!renderedValue) {
                                renderedValue = renderer({
                                    value: this.issue.fields[key]
                                });
                            } else {
                                renderedValue = renderer({
                                    value: renderedValue
                                });
                            }
                        }
                    }

                    // Draw the field now with the rendered value
                    renderField(this, this.issue.names[key], renderedValue, false);
                }
            }

            /*
             *  fields we want to render in a special way
             */
             // Votes
            if(this.issue.names.votes) {
                var renderedVotes = AtlassianMobile.Templates.JIRA.Issue.drawVotesField({
                    value: this.issue.fields.votes
                });
                renderField(this, this.issue.names['votes'], renderedVotes, false);
            }
            // Watchers
            if(this.issue.names.watches) {
                var renderedWatches = AtlassianMobile.Templates.JIRA.Issue.drawWatchesField({
                    value: this.issue.fields.watches
                });
                renderField(this, this.issue.names['watches'], renderedWatches, false);
            }
             // Attachments
            if(this.issue.names.attachment) {
                var renderedattachment = AtlassianMobile.Templates.JIRA.Issue.drawAttachmentField({
                    value: this.issue.renderedFields.attachment
                });
                renderField(this, AJS.I18n.getText('jira.mobile.fields.attachments'), renderedattachment, true);
            }
            // Issue links
            if(this.issue.names.issuelinks) {
                var renderedIssueLinks = renderIssueLinks({
                    value: this.issue.fields.issuelinks
                });
                renderField(this, this.issue.names['issuelinks'], renderedIssueLinks, true);
            }
            //SUBTASKS
            if(this.issue.names.subtasks) {
                var renderedSubtasks = AtlassianMobile.Templates.JIRA.Issue.drawSubTasksField({
                    value: this.issue.fields.subtasks
                });
                renderField(this, this.issue.names['subtasks'], renderedSubtasks, true);
            }

            return this;

            function renderField(context, name, html, block) {
                context.$el.append(AtlassianMobile.Templates.JIRA.Issue.drawField({
                    name: name,
                    valueHtml: html,
                    block: block
                }));
            }
        }
    });
});