define('feature/issues/comments/comments-view', ['zepto', 'backbone', 'feature/issues/comments/single-comment-model', 'feature/issues/comments/single-comment-view'], function ($, Backbone, SingleCommentModel, SingleCommentView) {
    var COLLAPSE_COMMENT_THRESHOLD = 3,
        EXPAND_COMMENT_THRESHOLD = 10;

    return Backbone.View.extend({
        className: 'issue-comments no-flicker',

        events: {
            'click .collapsed-comments': 'expandComments'
        },

        _updateData: function () {
            // If we already have comment information saved, then use that
            if (this.model.commentModel) {
                this.comments = this.model.commentModel.comments;
                this.collapsedComments = this.model.commentModel.collapsedComments;
            } else {
                var data = this.model.get('renderedFields').comment;
                var total = data.total;
                var commentArray = data.comments;
                this.comments = [];
                this.collapsedComments = 0;
                if (total > 0) {
                    if (this._checkCommentId(commentArray, total, this.commentId)) {
                        this._processWithSelectedComment(commentArray, total, this.commentId);
                    } else {
                        this._processCommentDataNormally(commentArray, total, this.commentId);
                    }
                }
                this.model.commentModel = {
                    comments: this.comments,
                    collapsedComments: this.collapsedComments
                }
            }
        },

        _checkCommentId: function (commentArray, total, commentId) {
            // We don't check the last one because it will always be expanded
            for (var i = 0; i < total - 1; i++) {
                if (commentArray[i].id === commentId) {
                    return true;
                }
            }
            // False if the comment Id isn't in the list or it's the last one
            return false;
        },

        _processCommentDataNormally: function (commentArray, total, commentId) {
            /* If there are more than x comments in total, collapse comments into one blob */
            if (total > COLLAPSE_COMMENT_THRESHOLD) {
                this.collapsedComments = total - 1
            /* Otherwise, draw 'summaries' of all the comments */
            } else {
                for (var i = 0; i < total - 1; i++) {
                    var model = new SingleCommentModel(commentArray[i]);
                    this.comments.push(new SingleCommentView({
                        model: model
                    }));
                }
            }
            /* Always have the latest comment expanded */
            var model = new SingleCommentModel(commentArray[total - 1]);
            model.expand = true;
            model.selected = commentId == model.attributes.id;
            this.comments.push(new SingleCommentView({
                model: model
            }));
        },

        _processWithSelectedComment: function (commentArray, total, commentId) {
            for (var i = 0; i < total; i++) {
                var model = new SingleCommentModel(commentArray[i]);
                model.expand = commentId == model.attributes.id;
                model.selected = commentId == model.attributes.id;
                this.comments.push(new SingleCommentView({
                    model: model
                }));
            }
        },

        initialize: function (options) {
            this.commentId = options.commentId;
            this._updateData();
            this.listenTo(this.model, 'change:renderedFields', function () {
                this._updateData();
                this.render();
            });
            this.listenTo(this.model, 'change:comments', function() {
                this._updateData();
                this.render();
            });
        },

        render: function () {
            this.$el.html(AtlassianMobile.Templates.JIRA.Issue.Comment.drawCommentsHeader({
                count: this.collapsedComments
            }));
            for (var i = 0, ii = this.comments.length; i < ii; i++) {
                this.$el.append(this.comments[i].render().$el);
            }
            return this;
        },

        expandComments: function() {
            var data = this.model.get('renderedFields').comment;
            var total = data.total,
                commentArray = data.comments,
                tempArray = [],
                /* if we have less than x comments, then we want to expand them, otherwise we just show summaries */
                doExpand = total < EXPAND_COMMENT_THRESHOLD;

            for (var i = 0; i < total - 1; i++) {
                var model = new SingleCommentModel(commentArray[i]);
                tempArray.push(new SingleCommentView({
                    model: model,
                    expand: doExpand
                }));
            }
            /* Join the temp array to what we already have so the order is correct */
            this.comments = tempArray.concat(this.comments);
            this.collapsedComments = 0;
            this.render();
        }
    });
});