define('util/image-viewer', ['zepto', 'fastclick'], function ($, Fastclick) {


    /**
     * ImageViewer
     * @param $img An optionally Zepto wrapped reference to the `img` element.
     * @return {*}
     * @constructor
     */
    var ImageViewer = function ($img) {

        // If constructor called without `new`.
        if (!(this instanceof ImageViewer)) {
            return new ImageViewer($img);
        }

        // Rewrap the object with Zepto.
        $img = $($img);

        if (!$img.is("img")) {
            throw new TypeError("First argument must be an `img` element.");
        }

        $img.click(_.bind(this.enlarge, this));

        $(window).bind("scroll", _.bind(function () {
            this.prevTouches = {};
        }, this));

        new FastClick($img[0]);

        this.$img = $img;
        this.$body = $("body");

        // This holds the maximum dimensions that the image
        // may be expanded to. It is set initially to the
        // thumbnail's dimensions, and then, if possible, set
        // to the full size image's dimensions.
        var thumbImage = new Image;

        $(thumbImage).bind("load", _.bind(function () {
            this.actualImageDimensions = {width: thumbImage.width, height: thumbImage.height};
            //alert(this.actualImageDimensions.width + " " + this.actualImageDimensions.height)
        }, this));

        thumbImage.src = $img.attr("src");

        // A prefix for all `class` attributes.
        this.classNamespace = "image-viewer-";

    }

    ImageViewer.prototype = {

        /**
         * Downloads the linked full size image if the initial image
         * is just a thumbnail and assigns the new path to the expanded
         * image.
         */
        _getFullSizeImage: function () {

            var fullSizeImagePath = this.$img.data("image-src");

            if (!fullSizeImagePath) {
                return;
            }

            var fullSizeImage = new Image;
            fullSizeImage.src = fullSizeImagePath;

            var callback = _.bind(function () {
                this.$clonedImg.attr("src", fullSizeImagePath);
                this.actualImageDimensions = {width: fullSizeImage.width, height: fullSizeImage.height};
            }, this);

            if (fullSizeImage.complete) {
                // If the image is cached, immediately invoke callback.
                callback();
            } else {
                // Otherwise, wait for it to load.
                $(fullSizeImage).bind("load", callback);
            }
        },

        _saveTouchCoords: function (event) {
            var prevTouches = this.prevTouches;
            _.each(event.targetTouches, function(touch) {
                if (!touch.pageX || !touch.pageY) {
                    return;
                }

                prevTouches.x[touch.identifier] = touch.pageX;
                prevTouches.y[touch.identifier] = touch.pageY;
            });
        },

        _touchstart: function (event) {

            // Only interested if this event bubbled from the image.
            if (event.target != this.$clonedImg[0]) {
                return;
            }

            this._saveTouchCoords(event);
        },

        _touchmove: function (event) {
            event.stopPropagation();
            event.preventDefault();

            // Only interested if this event bubbled from the image
            // and it's not currently snapping.
            if (event.target != this.$clonedImg[0]) {
                return;
            }

            var offset = this.$clonedImg.offset(),
                targetTouchesLength = event.targetTouches.length;

            // *If* they're not zooming, *else* they are panning.
            if (targetTouchesLength == 2) {
                this.isZooming = true;

                var relativeScale = event.scale;

                this.relativeScale = relativeScale;

                // Don't let the user resize the image down too small.
                if (offset.width < 100) {
                    return;
                }

                this.$clonedImg.css({
                    "-webkit-transform": "scale(" + relativeScale + ")"
                });

                this.zoomFingers = _.pluck(event.changedTouches, "identifier");


            } else if (targetTouchesLength == 1 && !this.isZooming) {

                var changedTouch = event.changedTouches[0];

                // If this finger was zoomin', don't do any pannin'
                if (this.zoomFingers.indexOf(changedTouch.identifier) > -1) {
                    return;
                }

                // Don't drag things around if the user hasn't zoomed in.
                if (offset.width <= $(window).width()) {
                    return;
                }

                var newY = offset.top + changedTouch.pageY - this.prevTouches.y[changedTouch.identifier] - this.$body[0].scrollTop,
                    newX =  offset.left + changedTouch.pageX - this.prevTouches.x[changedTouch.identifier];

                this.$clonedImg.css({
                    left: newX,
                    top: newY
                });

                this.isPanning = true;

            }

            this._saveTouchCoords(event);
        },

        _touchend: function (event) {
            // Only interested if this event bubbled from the image.
            if (event.target != this.$clonedImg[0]) {
                return;
            }

            var offset = this.$clonedImg.offset(),
                absoluteScale = offset.width / $(window).width();

            if (this.isZooming) {

                !event.targetTouches.length && (this.isZooming = false);

                if (offset.width > this.actualImageDimensions.width || offset.height > this.actualImageDimensions.height) {

                    alert(this.actualImageDimensions.width)

                    this.$container
                        .one("webkitTransitionEnd", _.bind(function () {

                        this.$container.removeClass(this._getClassName("ANIMATING"));

                        setTimeout(function (self) {

                            offset = self.$clonedImg.offset();

                            self.$clonedImg.css(
                                _.extend(offset,
                                    {
                                        "-webkit-transform": "scale(1)",
                                        top: offset.top - self.$body[0].scrollTop
                                    }
                                )
                            );
                        }, 0, this);


                        this.atMaxZoom = true;

                    }, this))
                        .addClass(this._getClassName("ANIMATING"));

                    var newScale = this.relativeScale * (this.actualImageDimensions.width / offset.width);

                    // The CSS to animate to.
                    this.$clonedImg.css(
                        {
                            "-webkit-transform": "scale(" + (this.atMaxZoom ? 1 : newScale) + ")"
                        }
                    );



                } else if (absoluteScale != 1) {
                    // At the end of the zoom, the element's dimensions have been scaled but the
                    // `event.scale` is always relative to the element's size when it begins
                    // scaling. By resetting the dimensions to their transformed dimensions,
                    // we get the relative zooming and it makes other dimension calculations easier.
                    this.$clonedImg.css(
                        _.extend(offset, {
                            "-webkit-transform": "scale(1)",
                            top: offset.top - this.$body[0].scrollTop
                        }));

                    this.atMaxZoom = false;

                    this.snapTo(offset);

                }

                // If the user zooms the image smaller than the viewport, pop it back.
                if (absoluteScale < 1) {
                    this.expandCloned(offset);
                    this.snapTo(offset);
                    this.isZooming = false;
                    this.atMaxZoom = false;
                }

            } else if (this.isPanning) {

                this.snapTo(offset);
                this._saveTouchCoords(event);

                this.isPanning = false;
            }

        },

        /**
         *
         * A method for getting a namespaced class name.
         *
         * @param key
         * @return {String} className
         */

        _getClassName: function(key) {
            var classes = {
                ANIMATING: "animating",
                EXPANDING: "expanding",
                SNAP_BACK: "snap-back"
            };

            return classes.hasOwnProperty(key) ? this.classNamespace + classes[key] : "";
        },

        _resetProperties: function () {
            this.isExpanded = this.isZooming = this.isPanning = this.atMaxZoom = false;
            this.prevTouches = {x: {}, y: {}};
            this.zoomFingers = [];
            this.expandedDefaultDimensions = null;
        },

        /**
         *
         * Shifts the image to a defined offset.
         *
         * @param offset An object with width, height, top and left properties.
         */

        expandCloned: function (offset) {

            var firstTime = !this.expandedDefaultDimensions;

            this.expandedDefaultDimensions = this.expandedDefaultDimensions || this.$clonedImg
                .css("visibility", "hidden")
                .show()
                .width($(window).width())
                .offset();

            if (firstTime) {
                // Set cloned image to initial size of original.
                this.$clonedImg
                    .css(_.extend(offset,
                    {
                        top: offset.top - this.$body[0].scrollTop
                    }
                ))
                    .css("visibility", "visible");
            }

            setTimeout(_.bind(function() {

                var top = ($(window).height() - this.expandedDefaultDimensions.height) / 2;

                this.$clonedImg.css({
                    width: $(window).width(),
                    height: "auto",
                    top: top,
                    left: 0,
                    right: 0
                });

                this.$container.addClass(this._getClassName("ANIMATING"));

                this.$clonedImg.one("webkitTransitionEnd", _.bind(function() {
                    this.$container
                        .removeClass(this._getClassName("ANIMATING"));

                    this.isExpanded = true;

                }, this));

            }, this), 0);
        },

        snapTo: function (offset) {

            if (!this.isPanning) {
                return;
            }

            var newCss = {};

            if (offset.left > 0) {
                newCss.left = 0;
            } else if ($(window).width() + Math.abs(offset.left) > offset.width) {
                newCss.left = -offset.width + $(window).width();
            }

            var imageBeyondViewportTop = offset.top < 0,
                imageBeyondViewportBottom = offset.height + offset.top > $(window).height(),
                imageBeyondViewport = offset.height > $(window).height(),
                imageContainedByViewport = imageBeyondViewportBottom && imageBeyondViewportTop;

            // This can't be an `else if` (from the `left` handling) or you won't get diagonal snap.
            if (!imageContainedByViewport) {

                if (imageBeyondViewport && imageBeyondViewportBottom) {
                    newCss.top = 0;
                } else if (imageBeyondViewport && imageBeyondViewportTop) {
                    newCss.top = $(window).height() - offset.height;
                } else {
                    newCss.top = ($(window).height() - offset.height) / 2;
                }

            }

            if (_.keys(newCss).length) {
                var className = this._getClassName("ANIMATING");
                this.$container
                    .one("webkitTransitionEnd", function() {
                        $(this).removeClass(className);
                    })
                    .addClass(className);

                this.$clonedImg.css(newCss);
            }
        },

        /**
         *
         * Enlarges the original image into its own space
         * ready for resizing and panning.
         *
         */

        enlarge: function () {

            if (this.isExpanded) {
                return;
            }

            // Reset properties.
            this._resetProperties();

            this.$container = $(Confluence.Templates.Mobile.ImageViewer.container({ src: this.$img.attr("src") }));
            this.$clonedImg = this.$container.find("img");

            this._getFullSizeImage();

            // Hide original image.
            this.$img.css("visibility", "hidden");

            // When rendering the cloned container and adding the class,
            // the browser will optimise the CSS to be applied by ignoring
            // the "initial" CSS. Wrapping the addClass with a setTimeout
            // ensures that the initial CSS is applied so it provides the
            // base for the animation.
            setTimeout(function ($cloned, className) {
                $cloned.addClass(className);
            }, 0, this.$container, this._getClassName("EXPANDING"));

            this.$body.append(this.$container);

            // Close button.
            this.$container.find("button").click(_.bind(this.shrink, this));

            this.expandCloned(this.$img.offset());

            this.$container
                // Don't want any dragging to occur.
                .bind("dragstart", function (event) { event.preventDefault(); })
                .bind("touchstart", _.bind(this._touchstart, this))
                .bind("touchmove", _.bind(this._touchmove, this))
                .bind("touchend", _.bind(this._touchend, this));

            $(window).bind("orientationchange", _.bind(function() {
                this._resetProperties();
                this.expandCloned.call(this, this.$clonedImg.offset());
            }, this));

            new FastClick(this.$container[0]);

        },

        /**
         *
         * Shrinks the expanded image back into its
         * original state.
         *
         */

        shrink: function () {

            if (!this.isExpanded) {
                return;
            }

            this.$clonedImg.one("webkitTransitionEnd", _.bind(function () {
                this.destroy();
                this.isExpanded = false;
            }, this));

            this.$container
                .addClass(this._getClassName("ANIMATING"))
                .removeClass(this._getClassName("EXPANDING"));

            // This time out prevents a "snap to"
            // from interfering with this transition.
            setTimeout(_.bind(function () {
                var offset = this.$img.offset();
                this.$clonedImg.css(
                    _.extend(
                        offset,
                        {
                            // We need to minus the scroll top because it's
                            // significant in this context.
                            top: offset.top - this.$body[0].scrollTop,
                            height: "auto"
                        }
                    )
                );

            }, this), 0);
        },
        /**
         * When navigating away from the page, the ImageViewer should be destroyed.
         */
        destroy: function () {
            this.$img.css("visibility", "visible");
            this.$container.remove();
        }
    };

    $.fn.imageViewer = function () {
        return this.each(function () {
            ImageViewer(this);
        });
    }

});