package com.aliyun.openservices.ons.api.impl.rocketmq;

import com.alibaba.rocketmq.client.exception.MQBrokerException;
import com.alibaba.rocketmq.client.exception.MQClientException;
import com.alibaba.rocketmq.client.log.ClientLogger;
import com.alibaba.rocketmq.client.producer.DefaultMQProducer;
import com.alibaba.rocketmq.common.protocol.ResponseCode;
import com.alibaba.rocketmq.remoting.exception.RemotingConnectException;
import com.alibaba.rocketmq.remoting.exception.RemotingTimeoutException;
import com.aliyun.openservices.ons.api.Message;
import com.aliyun.openservices.ons.api.Producer;
import com.aliyun.openservices.ons.api.PropertyKeyConst;
import com.aliyun.openservices.ons.api.SendResult;
import com.aliyun.openservices.ons.api.exception.ONSClientException;
import com.taobao.metaq.client.eagleeye.SendMessageHookImpl;
import org.slf4j.Logger;

import java.util.Properties;
import java.util.concurrent.atomic.AtomicBoolean;

import static com.alibaba.rocketmq.common.UtilAll.getPid;


public class ProducerImpl extends ONSClientAbstract implements Producer {
    private final static Logger log = ClientLogger.getLog();
    private final DefaultMQProducer defaultMQProducer;

    private final AtomicBoolean started = new AtomicBoolean(false);
    private final AtomicBoolean closed  = new AtomicBoolean(false);

    public ProducerImpl(final Properties properties) {
        super(properties);
        this.defaultMQProducer = new DefaultMQProducer(new ClientRPCHook(sessionCredentials));

        String producerGroup =
                properties.getProperty(PropertyKeyConst.ProducerId, "__ONS_PRODUCER_DEFAULT_GROUP");
        this.defaultMQProducer.setProducerGroup(producerGroup);

        if (properties.containsKey(PropertyKeyConst.SendMsgTimeoutMillis)) {
            this.defaultMQProducer.setSendMsgTimeout(Integer.valueOf(properties.get(
                    PropertyKeyConst.SendMsgTimeoutMillis).toString()));
        } else {
            this.defaultMQProducer.setSendMsgTimeout(5000);
        }

        this.defaultMQProducer.setInstanceName(this.buildIntanceName());
        this.defaultMQProducer.setNamesrvAddr(this.getNameServerAddr());
        // 消息最大大小128K
        this.defaultMQProducer.setMaxMessageSize(1024 * 128);

        // 为Consumer增加EagleEye打点日志
        try {
            this.defaultMQProducer.getDefaultMQProducerImpl().registerSendMessageHook(
                    new SendMessageHookImpl());
        } catch (Throwable e) {
        }
    }


    @Override
    public void start() {
        try {
            if (this.started.compareAndSet(false, true)) {
                this.defaultMQProducer.start();
            }
        } catch (Exception e) {
            throw new ONSClientException(e.getMessage(), e);
        }
    }


    @Override
    public void shutdown() {
        if (this.closed.compareAndSet(false, true)) {
            this.defaultMQProducer.shutdown();
        }
    }


    private void checkONSServiceState() {
        switch (this.defaultMQProducer.getDefaultMQProducerImpl().getServiceState()) {
            case CREATE_JUST:
                throw new ONSClientException(FAQ.errorMessage(String
                        .format("You do not have start the producer[" + getPid() + "], %s", this.defaultMQProducer
                                .getDefaultMQProducerImpl().getServiceState()), FAQ.SERVICE_STATE_WRONG));
            case SHUTDOWN_ALREADY:
                throw new ONSClientException(FAQ.errorMessage(String.format(
                        "Your producer has been shut down, %s", this.defaultMQProducer.getDefaultMQProducerImpl()
                                .getServiceState()), FAQ.SERVICE_STATE_WRONG));
            case START_FAILED:
                throw new ONSClientException(FAQ.errorMessage(String.format(
                        "When you start your service throws an exception, %s", this.defaultMQProducer
                                .getDefaultMQProducerImpl().getServiceState()), FAQ.SERVICE_STATE_WRONG));
            case RUNNING:
                break;
            default:
                break;
        }
    }


    private void checkProducerException(Exception e, Message message) {
        if (e instanceof MQClientException) {
            //
            if (e.getCause() != null) {
                // 无法连接Broker
                if (e.getCause() instanceof RemotingConnectException) {
                    throw new ONSClientException(FAQ.errorMessage(
                            String.format("Connect broker failed, Topic: %s", message.getTopic()),
                            FAQ.CONNECT_BROKER_FAILED));
                }
                // 发送消息超时
                else if (e.getCause() instanceof RemotingTimeoutException) {
                    throw new ONSClientException(FAQ.errorMessage(String.format(
                                    "Send message to broker timeout, %dms, Topic: %s",
                                    this.defaultMQProducer.getSendMsgTimeout(), message.getTopic()),
                            FAQ.SEND_MSG_TO_BROKER_TIMEOUT));
                }
                // Broker返回异常
                else if (e.getCause() instanceof MQBrokerException) {
                    MQBrokerException excep = (MQBrokerException) e.getCause();
                    throw new ONSClientException(FAQ.errorMessage(
                            String.format("Receive a broker exception, Topic: %s, %s", message.getTopic(),
                                    excep.getErrorMessage()), FAQ.BROKER_RESPONSE_EXCEPTION));
                }
            }
            // 纯客户端异常
            else {
                MQClientException excep = (MQClientException) e;
                if (-1 == excep.getResponseCode()) {
                    throw new ONSClientException(FAQ.errorMessage(
                            String.format("Topic does not exist, Topic: %s, %s", message.getTopic(),
                                    excep.getErrorMessage()), FAQ.TOPIC_ROUTE_NOT_EXIST));
                } else if (ResponseCode.MESSAGE_ILLEGAL == excep.getResponseCode()) {
                    throw new ONSClientException(FAQ.errorMessage(String.format(
                            "ONS Client check message exception, Topic: %s, %s", message.getTopic(),
                            excep.getErrorMessage()), FAQ.CLIENT_CHECK_MSG_EXCEPTION));
                }
            }
        }

        throw new ONSClientException("defaultMQProducer send exception", e);
    }


    @Override
    public SendResult send(Message message) {
        this.checkONSServiceState();

        com.alibaba.rocketmq.common.message.Message msgRMQ = ONSUtil.msgConvert(message);

        try {
            com.alibaba.rocketmq.client.producer.SendResult sendResultRMQ =
                    this.defaultMQProducer.send(msgRMQ);

            SendResult sendResult = new SendResult();
            sendResult.setMessageId(sendResultRMQ.getMsgId());
            return sendResult;
        } catch (Exception e) {
            log.error(String.format("Send message Exception, %s", message), e);
            this.checkProducerException(e, message);
            return null;
        }
    }


    @Override
    public void sendOneway(Message message) {
        this.checkONSServiceState();

        com.alibaba.rocketmq.common.message.Message msgRMQ = ONSUtil.msgConvert(message);
        try {
            this.defaultMQProducer.sendOneway(msgRMQ);
        } catch (Exception e) {
            log.error(String.format("Send message oneway Exception, %s", message), e);
            this.checkProducerException(e, message);
        }
    }

    @Override
    public boolean isStarted() {
        return started.get();
    }

    @Override
    public boolean isClosed() {
        return closed.get();
    }
}
