package com.aliyun.openservices.ons.api.impl.rocketmq;

import com.alibaba.rocketmq.client.consumer.DefaultMQPushConsumer;
import com.alibaba.rocketmq.client.consumer.listener.ConsumeOrderlyContext;
import com.alibaba.rocketmq.client.consumer.listener.ConsumeOrderlyStatus;
import com.alibaba.rocketmq.client.consumer.listener.MessageListenerOrderly;
import com.alibaba.rocketmq.client.exception.MQClientException;
import com.alibaba.rocketmq.common.message.MessageExt;
import com.aliyun.openservices.ons.api.Message;
import com.aliyun.openservices.ons.api.PropertyKeyConst;
import com.aliyun.openservices.ons.api.exception.ONSClientException;
import com.aliyun.openservices.ons.api.order.ConsumeOrderContext;
import com.aliyun.openservices.ons.api.order.MessageOrderListener;
import com.aliyun.openservices.ons.api.order.OrderAction;
import com.aliyun.openservices.ons.api.order.OrderConsumer;

import java.util.List;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;


public class OrderConsumerImpl extends ONSClientAbstract implements OrderConsumer {
    private final DefaultMQPushConsumer defaultMQPushConsumer;
    private final ConcurrentHashMap<String/* Topic */, MessageOrderListener> subscribeTable =
            new ConcurrentHashMap<String, MessageOrderListener>();

    private final AtomicBoolean started = new AtomicBoolean(false);
    private final AtomicBoolean closed = new AtomicBoolean(false);

    public OrderConsumerImpl(final Properties properties) {
        super(properties);
        this.defaultMQPushConsumer = new DefaultMQPushConsumer(new ClientRPCHook(sessionCredentials));

        String consumerGroup = properties.getProperty(PropertyKeyConst.ConsumerId);
        if (null == consumerGroup) {
            throw new ONSClientException("\'ConsumerGroup\' property is null");
        }

        this.defaultMQPushConsumer.setConsumerGroup(consumerGroup);
        this.defaultMQPushConsumer.setInstanceName(this.buildIntanceName());
        this.defaultMQPushConsumer.setNamesrvAddr(this.getNameServerAddr());

        if (properties.containsKey(PropertyKeyConst.ConsumeThreadNums)) {
            this.defaultMQPushConsumer.setConsumeThreadMin(Integer.valueOf(properties.get(
                    PropertyKeyConst.ConsumeThreadNums).toString()));

            this.defaultMQPushConsumer.setConsumeThreadMax(Integer.valueOf(properties.get(
                    PropertyKeyConst.ConsumeThreadNums).toString()));
        }
    }


    @Override
    public void start() {
        this.defaultMQPushConsumer.registerMessageListener(new MessageListenerOrderlyImpl());
        try {
            if (started.compareAndSet(false, true)) {
                this.defaultMQPushConsumer.start();
            }
        } catch (Exception e) {
            throw new ONSClientException(e.getMessage(), e);
        }
    }


    @Override
    public void shutdown() {
        if(closed.compareAndSet(false,true)){
            this.defaultMQPushConsumer.shutdown();
        }
    }

    class MessageListenerOrderlyImpl implements MessageListenerOrderly {

        @Override
        public ConsumeOrderlyStatus consumeMessage(List<MessageExt> arg0, ConsumeOrderlyContext arg1) {
            MessageExt msgRMQ = arg0.get(0);
            Message msg = ONSUtil.msgConvert(msgRMQ);
            msg.setMsgID(msgRMQ.getMsgId());

            MessageOrderListener listener = OrderConsumerImpl.this.subscribeTable.get(msg.getTopic());
            if (null == listener) {
                throw new ONSClientException("MessageOrderListener is null");
            }

            final ConsumeOrderContext context = new ConsumeOrderContext();
            OrderAction action = listener.consume(msg, context);
            if (action != null) {
                switch (action) {
                    case ConsumeSuccessfully:
                        return ConsumeOrderlyStatus.SUCCESS;
                    case SuspendMoment:
                        return ConsumeOrderlyStatus.SUSPEND_CURRENT_QUEUE_A_MOMENT;
                    default:
                        break;
                }
            }

            return ConsumeOrderlyStatus.SUSPEND_CURRENT_QUEUE_A_MOMENT;
        }
    }


    @Override
    public void subscribe(String topic, String subExpression, MessageOrderListener listener) {
        if (null == topic) {
            throw new ONSClientException("topic is null");
        }

        if (null == listener) {
            throw new ONSClientException("listener is null");
        }

        try {
            this.subscribeTable.put(topic, listener);
            this.defaultMQPushConsumer.subscribe(topic, subExpression);
        } catch (MQClientException e) {
            throw new ONSClientException("defaultMQPushConsumer subscribe exception", e);
        }
    }

    @Override
    public boolean isStarted() {
        return started.get();
    }

    @Override
    public boolean isClosed() {
        return closed.get();
    }
}
