/*
 * Copyright 2008-2108 amoeba.meidusa.com 
 * 
 * 	This program is free software; you can redistribute it and/or modify it under the terms of 
 * the GNU AFFERO GENERAL PUBLIC LICENSE as published by the Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version. 
 * 
 * 	This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * See the GNU AFFERO GENERAL PUBLIC LICENSE for more details. 
 * 	You should have received a copy of the GNU AFFERO GENERAL PUBLIC LICENSE along with this program; 
 * if not, write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package com.meidusa.venus.client;

import java.beans.PropertyDescriptor;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.beanutils.BeanUtilsBean;
import org.apache.commons.beanutils.ConvertUtilsBean;
import org.apache.commons.beanutils.PropertyUtilsBean;
import org.apache.commons.digester.Digester;
import org.apache.commons.digester.RuleSet;
import org.apache.commons.digester.xmlrules.FromXmlRuleSet;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.BeanFactoryAware;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.config.AutowireCapableBeanFactory;
import org.springframework.beans.factory.config.BeanFactoryPostProcessor;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.util.ResourceUtils;

import com.meidusa.venus.annotations.Endpoint;
import com.meidusa.venus.client.net.VenusBIOConnectionFactory;
import com.meidusa.venus.client.net.VenusNIOConnectionFactory;
import com.meidusa.venus.client.xml.bean.FactoryConfig;
import com.meidusa.venus.client.xml.bean.PoolConfig;
import com.meidusa.venus.client.xml.bean.Remote;
import com.meidusa.venus.client.xml.bean.ServiceConfig;
import com.meidusa.venus.client.xml.bean.VenusClient;
import com.meidusa.venus.digester.DigesterRuleParser;
import com.meidusa.venus.exception.CodedException;
import com.meidusa.venus.exception.VenusExceptionFactory;
import com.meidusa.venus.exception.XmlVenusExceptionFactory;
import com.meidusa.venus.poolable.MultipleLoadBalanceObjectPool;
import com.meidusa.venus.poolable.ObjectPool;
import com.meidusa.venus.poolable.PoolableObjectPool;
import com.meidusa.venus.util.FileWatchdog;
import com.meidusa.toolkit.common.bean.BeanContext;
import com.meidusa.toolkit.common.bean.BeanContextBean;
import com.meidusa.toolkit.common.bean.config.ConfigUtil;
import com.meidusa.toolkit.common.bean.config.ConfigurationException;
import com.meidusa.toolkit.common.bean.util.InitialisationException;
import com.meidusa.toolkit.common.util.Tuple;
import com.meidusa.toolkit.net.ConnectionManager;
import com.meidusa.toolkit.net.util.StringUtil;

public class VenusServiceFactory implements ServiceFactory,BeanFactoryAware,InitializingBean,BeanFactoryPostProcessor {
	private static Logger logger = Logger.getLogger(ServiceFactory.class);
	private Map<Class<?>,Tuple<Object,RemotingInvocationHandler>> servicesMap = new HashMap<Class<?>,Tuple<Object,RemotingInvocationHandler>>(); 
	private BeanFactory beanFactory;
	private ConnectionManager connManager;
	private String[] configFiles;
	private BeanContext beanContext;
	private boolean enableAsync = true;
	private boolean shutdown = false;
	private Map<String,Tuple<ObjectPool,ObjectPool>> poolMap = new HashMap<String,Tuple<ObjectPool,ObjectPool>>();
	private List<ObjectPool> realPools = new ArrayList<ObjectPool>();
	private InvocationListenerContainer container = new InvocationListenerContainer();
	private VenusNIOMessageHandler handler = new VenusNIOMessageHandler();
	private VenusExceptionFactory venusExceptionFactory;
	private Map<Class<?>,ServiceConfig> serviceConfig = new HashMap<Class<?>,ServiceConfig>();
	private int asyncExecutorSize = 10;
	private boolean needPing = false;
	private Timer reloadTimer = new Timer();
	private boolean enableReload = true;
	
	public boolean isEnableReload() {
		return enableReload;
	}

	public void setEnableReload(boolean enableReload) {
		this.enableReload = enableReload;
	}

	public boolean isNeedPing() {
		return needPing;
	}

	public void setNeedPing(boolean needPing) {
		this.needPing = needPing;
	}

	public boolean isEnableAsync() {
		return enableAsync;
	}

	public void setEnableAsync(boolean enableAsync) {
		this.enableAsync = enableAsync;
	}

	public int getAsyncExecutorSize() {
		return asyncExecutorSize;
	}

	public void setAsyncExecutorSize(int asyncExecutorSize) {
		this.asyncExecutorSize = asyncExecutorSize;
	}

	public VenusExceptionFactory getVenusExceptionFactory() {
		return venusExceptionFactory;
	}

	public void setVenusExceptionFactory(VenusExceptionFactory venusExceptionFactory) {
		this.venusExceptionFactory = venusExceptionFactory;
	}

	public String[] getConfigFiles() {
		return configFiles;
	}

	public void setConfigFiles(String[] configFiles) {
		this.configFiles = configFiles;
	}

	
	@SuppressWarnings("unchecked")
	public <T> T getService(Class<T> t) {
		if(shutdown){
			throw new IllegalStateException("service factory has been shutdown");
		}
		Tuple<Object,RemotingInvocationHandler> object = servicesMap.get(t);
		
		return (T)object.left;
	}

	public void setBeanFactory(BeanFactory beanFactory) throws BeansException {
		this.beanFactory = beanFactory; 
	}


	@SuppressWarnings("unchecked")
	public void afterPropertiesSet() throws Exception {
		Level level = logger.getLevel();
		logger.setLevel(Level.INFO);
		logger.info("current Venus Client id="+VenusConstant.VENUS_CLIENT_ID);
		logger.setLevel(level);
		if(venusExceptionFactory == null){
			XmlVenusExceptionFactory xmlVenusExceptionFactory = new XmlVenusExceptionFactory();
			xmlVenusExceptionFactory.setConfigFiles(new String[]{"classpath:com/meidusa/venus/exception/VenusSystemException.xml"});
			xmlVenusExceptionFactory.init();
			this.venusExceptionFactory = xmlVenusExceptionFactory;
		}
		
		handler.setVenusExceptionFactory(venusExceptionFactory);
		if(enableAsync && connManager == null){
			ExecutorService excutors = Executors.newFixedThreadPool(this.getAsyncExecutorSize());
			try {
				connManager = new ConnectionManager();
			} catch (IOException e) {
				throw new InitialisationException(e);
			}
			
		    connManager.setExecutor(excutors);
		    connManager.start();
		}
		
		beanContext = new BeanContext(){
			public Object getBean(String beanName) {
				if(beanFactory != null){
					return beanFactory.getBean(beanName);
				}else{
					return null;
				}
			}
			
			public Object createBean(Class clazz) throws Exception {
				if(beanFactory instanceof AutowireCapableBeanFactory){
					AutowireCapableBeanFactory factory = (AutowireCapableBeanFactory)beanFactory;
					return factory.autowire(clazz, AutowireCapableBeanFactory.AUTOWIRE_BY_NAME, false);
				}
				return null;
			}
		};
		BeanContextBean.getInstance().setBeanContext(beanContext);
		BeanUtilsBean.setInstance(new BeanUtilsBean(new ConvertUtilsBean(),new PropertyUtilsBean()){

			public void setProperty(Object bean, String name, Object value)throws IllegalAccessException, InvocationTargetException{
				if(value instanceof String){
					PropertyDescriptor descriptor = null;
		            try {
		                descriptor =
		                    getPropertyUtils().getPropertyDescriptor(bean, name);
		                if (descriptor == null) {
		                    return; // Skip this property setter
		                }else{
		                	if(descriptor.getPropertyType().isEnum()){
		                		Class<Enum> clazz = (Class<Enum>)descriptor.getPropertyType();
		                		value = Enum.valueOf(clazz,(String)value);
		                	}else{
		                		Object temp = null;
		                		try{
		                			temp = ConfigUtil.filter((String) value, beanContext);
		                		}catch(Exception e){
		                		}
		    					if(temp == null){
		    						temp = ConfigUtil.filter((String) value);
		    					}
		    					value = temp;
		                	}
		                }
		            } catch (NoSuchMethodException e) {
		                return; // Skip this property setter
		            }
				}
				super.setProperty(bean, name, value);
			}
			
		});

		handler.setContainer(this.container);
		reloadConfiguration();
		File[] files = new File[this.configFiles.length];
		for(int i=0;i<this.configFiles.length;i++){
			files[i]=ResourceUtils.getFile(configFiles[i]);
		}
		if(enableReload){
			VenusFileWatchdog dog = new VenusFileWatchdog(files);
			dog.setDelay(1000*10);
			dog.start();
		}
	}
	
	class VenusFileWatchdog extends FileWatchdog{

		protected VenusFileWatchdog(File... file) {
			super(file);
		}

		@Override
		protected void doOnChange() {
			try {
				VenusServiceFactory.this.reloadConfiguration();
			} catch (Exception e) {
				VenusServiceFactory.logger.error("reload configuration error",e);
			}
		}
		
	}

	private synchronized void  reloadConfiguration() throws Exception{
		Map<Class<?>,Tuple<Object,RemotingInvocationHandler>> servicesMap = new HashMap<Class<?>,Tuple<Object,RemotingInvocationHandler>>(); 
		Map<String,Tuple<ObjectPool,ObjectPool>> poolMap = new HashMap<String,Tuple<ObjectPool,ObjectPool>>();
		Map<Class<?>,ServiceConfig> serviceConfig = new HashMap<Class<?>,ServiceConfig>();
		
		final List<ObjectPool> realPools = new ArrayList<ObjectPool>();
		try{
			loadConfiguration(poolMap,servicesMap,serviceConfig,realPools);
		}catch(Exception e){
			reloadTimer.schedule(new ClosePoolTask(realPools),1000 * 30);
			throw e;
		}
		this.poolMap = poolMap;
		
		for(Map.Entry<Class<?>, Tuple<Object,RemotingInvocationHandler>> entry : servicesMap.entrySet()){
			Class<?> key = entry.getKey();
			Tuple<Object,RemotingInvocationHandler> source = entry.getValue();
			Tuple<Object,RemotingInvocationHandler> target = this.servicesMap.get(key);
			if(target != null){
				target.right.setBioConnPool(source.getRight().getBioConnPool());
				target.right.setNioConnPool(source.getRight().getNioConnPool());
				target.right.setSerializeType((byte)target.right.getSerializeType());
			}else{
				this.servicesMap.put(key, source);
			}
		}
		this.serviceConfig = serviceConfig;
		final List<ObjectPool> oldPools = this.realPools;
		this.realPools = realPools;
		reloadTimer.schedule(new ClosePoolTask(oldPools),1000 * 30);
		
	}
	
	class ClosePoolTask extends TimerTask {
		List<ObjectPool> pools;
		public ClosePoolTask(List<ObjectPool> pools){
			this.pools = pools;
		}
		
		@Override
		public void run() {
			for(ObjectPool pool : pools){
				try {
					pool.close();
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}
	}
	
	private void loadConfiguration(Map<String,Tuple<ObjectPool,ObjectPool>> poolMap,Map<Class<?>,Tuple<Object,RemotingInvocationHandler>> servicesMap,Map<Class<?>,ServiceConfig> serviceConfig,List<ObjectPool> realPools) throws Exception{
		VenusClient all = new VenusClient();
		for (String configFile : configFiles) {
			configFile = (String)ConfigUtil.filter(configFile);
			RuleSet ruleSet = new FromXmlRuleSet(this.getClass().getResource("venusClientRule.xml"),new DigesterRuleParser());
	        Digester digester = new Digester();
	        digester.setValidating(false);
	        digester.addRuleSet(ruleSet);
	        
			try {
				InputStream is = ResourceUtils.getURL(configFile.trim()).openStream();
				VenusClient venus = (VenusClient) digester.parse(is);
				for(Map.Entry<Class<?>, ServiceConfig> entry: venus.getServiceMap().entrySet()){
					ServiceConfig config = entry.getValue();
					if(config.getType() == null){
						throw new ConfigurationException("Service type can not be null:"+configFile);
					}
				}
				all.getRemoteMap().putAll(venus.getRemoteMap());
				all.getServiceMap().putAll(venus.getServiceMap());
			} catch (Exception e) {
				throw new ConfigurationException("can not parser xml:"+configFile,e);
			}
		}
		
		//ʼ remoteҴPool
		for(Map.Entry<String, Remote> entry : all.getRemoteMap().entrySet()){
			RemoteContainer container = createRemoteContainer(entry.getValue(),realPools);
			Tuple<ObjectPool,ObjectPool> tuple = new Tuple<ObjectPool,ObjectPool>();
			tuple.left = container.getBioPool();
			tuple.right = container.getNioPool();
			poolMap.put(entry.getKey(), tuple);
		}
		
		for(Map.Entry<Class<?>, ServiceConfig> entry: all.getServiceMap().entrySet()){
			ServiceConfig config = entry.getValue();
			
			Remote remote = all.getRemoteMap().get(config.getRemote());
			Tuple<ObjectPool,ObjectPool> tuple = null;
			if(!StringUtil.isEmpty(config.getRemote())){
				tuple = poolMap.get(config.getRemote());
				if(tuple == null){
					throw new ConfigurationException("remote="+config.getRemote() +" not found!!");
				}
			}else{
				String ipAddress = config.getIpAddressList();
				tuple = poolMap.get(ipAddress);
				if(ipAddress != null && tuple == null ){
					RemoteContainer container = createRemoteContainer(ipAddress,realPools);
					tuple = new Tuple<ObjectPool,ObjectPool>();
					tuple.left = container.getBioPool();
					tuple.right = container.getNioPool();
					poolMap.put(ipAddress, tuple);
				}
			}
			
			if(tuple != null ){
				RemotingInvocationHandler invocationHandler = new RemotingInvocationHandler();
				invocationHandler.setBioConnPool(tuple.left);
				invocationHandler.setNioConnPool(tuple.right);
				invocationHandler.setServiceFactory(this);
				invocationHandler.setVenusExceptionFactory(this.getVenusExceptionFactory());
				if(remote!= null && remote.getAuthenticator() != null){
					invocationHandler.setSerializeType(remote.getAuthenticator().getSerializeType());
				}
				
				invocationHandler.setContainer(this.container);
				
				Object object = Proxy.newProxyInstance(this.getClass().getClassLoader(), new Class[]{config.getType()}, invocationHandler);
				
				
				for(Method method : config.getType().getMethods()){
					Endpoint endpoint =	method.getAnnotation(Endpoint.class);
					if(endpoint != null){
						Class[] eclazz = method.getExceptionTypes();
						for(Class clazz : eclazz){
							if(venusExceptionFactory != null && CodedException.class.isAssignableFrom(clazz)){
								venusExceptionFactory.addException(clazz);
							}
						}
					}
				}
				
				serviceConfig.put(config.getType(), config);
				servicesMap.put(config.getType(), new Tuple<Object,RemotingInvocationHandler>(object,invocationHandler));
			}else{
				if(config.getInstance() != null){
					servicesMap.put(config.getType(), new Tuple<Object,RemotingInvocationHandler>(config.getInstance(),null));
				}else{
					throw new ConfigurationException("Service instance or ipAddressList or remote can not be null:"+config.getType());
				}
			}
			
		}
	}
	
	private RemoteContainer createRemoteContainer(String ipAddress,List<ObjectPool> realPools) throws Exception{
		RemoteContainer container = new RemoteContainer();
		
		if(!StringUtil.isEmpty(ipAddress)){
			String ipList[] = StringUtil.split(ipAddress ,", ");
			PoolableObjectPool bioPools[] = new PoolableObjectPool[ipList.length];
			PoolableObjectPool nioPools[] = new PoolableObjectPool[ipList.length];
			for(int i=0;i<ipList.length;i++){
				nioPools[i] = new PoolableObjectPool();
				bioPools[i] = new PoolableObjectPool();


				VenusNIOConnectionFactory nioFactory = new VenusNIOConnectionFactory();
				
				//bio
				VenusBIOConnectionFactory bioFactory = new VenusBIOConnectionFactory();

				bioFactory.setNeedPing(needPing);
				
				String temp[] = StringUtil.split(ipList[i],":");
				if(temp.length>1){
					nioFactory.setIpAddress(temp[0]);
					nioFactory.setPort(Integer.valueOf(temp[1]));
					
					bioFactory.setIpAddress(temp[0]);
					bioFactory.setPort(Integer.valueOf(temp[1]));
				}else{
					nioFactory.setIpAddress(temp[0]);
					nioFactory.setPort(16800);
					
					bioFactory.setIpAddress(temp[0]);
					bioFactory.setPort(16800);
				}
				
				if(this.isEnableAsync()){
					nioFactory.setConnectionManager(this.connManager);
					nioFactory.setMessageHandler(handler);
					nioFactory.init();
					nioPools[i].setFactory(nioFactory);
					nioPools[i].setName("n-connPool-"+nioFactory.getIpAddress());
					nioPools[i].init();
					realPools.add(nioPools[i]);
				}
				bioPools[i].setName("b-connPool-"+nioFactory.getIpAddress());
				bioPools[i].setFactory(bioFactory);
				bioPools[i].init();
				realPools.add(bioPools[i]);
				
			}
			
			if(ipList.length >1){
				MultipleLoadBalanceObjectPool bioPool = new MultipleLoadBalanceObjectPool(MultipleLoadBalanceObjectPool.LOADBALANCING_ROUNDROBIN,bioPools);
				MultipleLoadBalanceObjectPool nioPool = new MultipleLoadBalanceObjectPool(MultipleLoadBalanceObjectPool.LOADBALANCING_ROUNDROBIN,nioPools);
				bioPool.setName("b-multi-connPool"+ipAddress);
				nioPool.setName("n-multi-connPool"+ipAddress);
				
				bioPool.init();
				
				nioPool.init();
				realPools.add(bioPool);
				realPools.add(nioPool);
				container.setBioPool(bioPool);
				container.setNioPool(nioPool);
			}else{
				container.setBioPool(bioPools[0]);
				container.setNioPool(nioPools[0]);
			}
		}else{
			throw new IllegalArgumentException(" ipaddress cannot be null");
		}
		
		return container;
	}
	
	
	
	
	
	private RemoteContainer createRemoteContainer(Remote remote,List<ObjectPool> realPools) throws Exception{
		RemoteContainer container = new RemoteContainer();
		FactoryConfig factoryConfig = remote.getFactory();
		PoolConfig poolConfig = remote.getPool();
		String ipAddress = factoryConfig.getIpAddressList();
		if(!StringUtil.isEmpty(ipAddress)){
			String ipList[] = StringUtil.split(ipAddress ,", ");
			PoolableObjectPool bioPools[] = new PoolableObjectPool[ipList.length];
			PoolableObjectPool nioPools[] = new PoolableObjectPool[ipList.length];
			for(int i=0;i<ipList.length;i++){
				nioPools[i] = new com.meidusa.venus.poolable.PoolableObjectPool();
				bioPools[i] = new com.meidusa.venus.poolable.PoolableObjectPool();
				if(poolConfig != null){
					BeanUtils.copyProperties(nioPools[i], poolConfig);
					BeanUtils.copyProperties(bioPools[i], poolConfig);
				}
				
				//nio
				VenusNIOConnectionFactory nioFactory = new VenusNIOConnectionFactory();
				if(remote.getAuthenticator() != null){
					nioFactory.setAuthenticator(remote.getAuthenticator());
				}
				
				//bio
				VenusBIOConnectionFactory bioFactory = new VenusBIOConnectionFactory();
				if(remote!= null && remote.getAuthenticator() != null){
					bioFactory.setAuthenticator(remote.getAuthenticator());
				}
				
				bioFactory.setNeedPing(needPing);
				if(factoryConfig != null){
					
					BeanUtils.copyProperties(nioFactory, factoryConfig);
					BeanUtils.copyProperties(bioFactory, factoryConfig);
				}
				String temp[] = StringUtil.split(ipList[i],":");
				if(temp.length>1){
					nioFactory.setIpAddress(temp[0]);
					nioFactory.setPort(Integer.valueOf(temp[1]));
					
					bioFactory.setIpAddress(temp[0]);
					bioFactory.setPort(Integer.valueOf(temp[1]));
				}else{
					nioFactory.setIpAddress(temp[0]);
					nioFactory.setPort(16800);
					
					bioFactory.setIpAddress(temp[0]);
					bioFactory.setPort(16800);
				}
				
				if(this.isEnableAsync()){
					nioFactory.setConnectionManager(this.connManager);
					nioFactory.setMessageHandler(handler);
					nioFactory.init();
					nioPools[i].setFactory(nioFactory);
					nioPools[i].setName("n-connPool-"+nioFactory.getIpAddress());
					nioPools[i].init();
					realPools.add(nioPools[i]);
				}
				bioPools[i].setName("b-connPool-"+bioFactory.getIpAddress());
				bioPools[i].setFactory(bioFactory);
				bioPools[i].init();
				realPools.add(bioPools[i]);
			}
			
			if(ipList.length >1){
				MultipleLoadBalanceObjectPool bioPool = new MultipleLoadBalanceObjectPool(MultipleLoadBalanceObjectPool.LOADBALANCING_ROUNDROBIN,bioPools);
				MultipleLoadBalanceObjectPool nioPool = new MultipleLoadBalanceObjectPool(MultipleLoadBalanceObjectPool.LOADBALANCING_ROUNDROBIN,nioPools);
				
				bioPool.setName("b-multi-connPool"+remote.getName());
				nioPool.setName("n-multi-connPool"+remote.getName());
				
				container.setBioPool(bioPool);
				container.setNioPool(nioPool);
				bioPool.init();
				nioPool.init();
				realPools.add(bioPool);
				realPools.add(nioPool);
			}else{
				container.setBioPool(bioPools[0]);
				container.setNioPool(nioPools[0]);
			}
			container.setRemote(remote);
		}else{
			throw new IllegalArgumentException("remtoe="+remote.getName()+", ipaddress cannot be null");
		}
		
		return container;
	}
	
	public ServiceConfig getServiceConfig(Class<?> type){
		return serviceConfig.get(type);
	}
	public void postProcessBeanFactory(ConfigurableListableBeanFactory beanFactory) throws BeansException {
		
		//register to resolvable dependency container
		if(beanFactory instanceof ConfigurableListableBeanFactory){
			ConfigurableListableBeanFactory cbf = (ConfigurableListableBeanFactory)beanFactory;
			for(Map.Entry<Class<?>, Tuple<Object,RemotingInvocationHandler>> entry : servicesMap.entrySet()){
				cbf.registerResolvableDependency(entry.getKey(), entry.getValue().left);
			}
		}
	}

	public synchronized void destroy() {
		shutdown = true;
		if(connManager != null){
			if(connManager.isAlive()){
				connManager.shutdown();
			}
		}
	}

/*	@Override
	public <T> T getService(Class<T> t, String address) {
		
		return null;
	}*/

}
