/*
 * Copyright 2008-2108 amoeba.meidusa.com 
 * 
 * 	This program is free software; you can redistribute it and/or modify it under the terms of 
 * the GNU AFFERO GENERAL PUBLIC LICENSE as published by the Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version. 
 * 
 * 	This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * See the GNU AFFERO GENERAL PUBLIC LICENSE for more details. 
 * 	You should have received a copy of the GNU AFFERO GENERAL PUBLIC LICENSE along with this program; 
 * if not, write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package com.meidusa.venus.client.net;

import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.util.Date;

import org.apache.log4j.Logger;

import com.meidusa.venus.client.authenticate.Authenticator;
import com.meidusa.venus.client.authenticate.DummyAuthenticator;
import com.meidusa.venus.io.Status;
import com.meidusa.venus.io.VenusPacketMetaData;
import com.meidusa.venus.io.packet.AbstractServicePacket;
import com.meidusa.venus.io.packet.AuthenPacket;
import com.meidusa.venus.io.packet.ErrorPacket;
import com.meidusa.venus.io.packet.HandshakePacket;
import com.meidusa.venus.io.packet.PacketConstant;
import com.meidusa.venus.io.packet.PingPacket;
import com.meidusa.venus.poolable.ObjectPool;
import com.meidusa.venus.poolable.PoolableObject;
import com.meidusa.toolkit.net.AuthingableConnection;
import com.meidusa.toolkit.net.Connection;
import com.meidusa.toolkit.net.io.FramedInputStream;
import com.meidusa.toolkit.net.io.FramingOutputStream;
import com.meidusa.toolkit.net.io.PacketInputStream;
import com.meidusa.toolkit.net.io.PacketOutputStream;

/**
 * 
 * @author Struct
 *
 */
public class VenusNIOConnection extends AuthingableConnection implements PoolableObject{
	
    private static Logger logger = Logger.getLogger(VenusNIOConnection.class);
    private Authenticator<HandshakePacket,AuthenPacket> authenticator = new DummyAuthenticator();
    private long lastPingTime = System.currentTimeMillis();
    private long lastPongTime = System.currentTimeMillis();
    private long pingInterval = 10 * 1000;
    private static VenusPacketMetaData PACKET_META_DATA = new VenusPacketMetaData();
    private Status status = Status.WAITE_HANDSHAKE;
    private short serializeType;
    private ObjectPool objectPool;
    private boolean active;
    
    public VenusNIOConnection(SocketChannel channel, long createStamp) {
        super(channel, createStamp);
        _outQueue.setMaxSize(10);
    }


    public short getSerializeType() {
		return serializeType;
	}

	@Override
    protected PacketInputStream createPacketInputStream() {
        return new FramedInputStream(PACKET_META_DATA, true);
    }

    @Override
    protected PacketOutputStream createPacketOutputStream() {
        return new FramingOutputStream(PACKET_META_DATA, true);
    }

    public void postMessage(byte[] msg) {
    	ByteBuffer buff = ByteBuffer.wrap(msg);
    	this.postMessage(buff);
    }
    
    public boolean needPing(long now) {
        return (((now - lastPingTime)>pingInterval) 
        && (now - lastMessageSent > pingInterval));
    }

    protected void messageProcess(byte[] msg) {
    	
    	if(msg == null || msg.length < PacketConstant.SERVICE_HEADER_SIZE){
    		if(logger.isInfoEnabled()){
    			logger.info("conn ID="+this.getId()+" , received packet size error, size="+(msg == null?0: msg.length));
    		}
    		this.postClose(new IllegalArgumentException("packet size error size="+(msg == null?0: msg.length)));
    		return;
    	}
    	
        int type = AbstractServicePacket.getType(msg);
        if (type == PacketConstant.PACKET_TYPE_PONG) {
            lastPongTime = System.currentTimeMillis();
            if (logger.isDebugEnabled()) {
                logger.debug("receive pong packet from " + this.getId());
            }
        }else{
        	super.messageProcess(msg);
        }
    }

    public void ping(long now) {
        postMessage(new PingPacket().toByteBuffer());
        lastPingTime = System.currentTimeMillis();
        if (logger.isDebugEnabled()) {
            logger.debug("send ping packet to " + this.getId());
        }
    }

    public boolean checkIdle(long now) {
        boolean idle = super.checkIdle(now);
        if (!idle) {
            if (lastPingTime - lastPongTime > 2 * pingInterval) {
                logger.warn("receive pong packet timeout, id=" + this.getId() + ",lastPingTime=" + new Date(lastPingTime) + ",lastPongTime =" + new Date(lastPongTime));
                return true;
            }
        }
        return false;

    }

	public ObjectPool getObjectPool() {
		return objectPool;
	}

	public boolean isActive() {
		return active;
	}

	public boolean isRemovedFromPool() {
		return objectPool == null;
	}

	public void setActive(boolean isactive) {
		this.active = isactive;
	}

	public void setObjectPool(ObjectPool pool) {
		this.objectPool = pool;
	}

	public void handleMessage(Connection conn, byte[] message) {
		if(AbstractServicePacket.getType(message) == PacketConstant.PACKET_TYPE_ERROR){
			setAuthenticated(false);
			ErrorPacket error = new ErrorPacket();
			error.init(message);
			logger.error("handShake with host="+this.getId()+", errorCode="+error.errorCode+" ,message="+error.message+",hashCode="+this.hashCode());
			return;
		}
		
		if(status == Status.WAITE_HANDSHAKE){
			
			if(logger.isDebugEnabled()){
				logger.debug("1. handShake with "+this.getId()+",hashCode="+this.hashCode());
			}
			
			HandshakePacket handpacket = new HandshakePacket();
			handpacket.init(message);
			
			AuthenPacket authen = getAuthenticator().createAuthenPacket(handpacket);
			this.serializeType = authen.shakeSerializeType;
			status = Status.AUTHING;
			if(logger.isDebugEnabled()){
				logger.debug("2. authing packet sent to server:"+this.getId()+",hashCode="+this.hashCode());
			}
			
			this.postMessage(authen.toByteBuffer());
		}else if(status == Status.AUTHING){
			
			if(AbstractServicePacket.getType(message) == PacketConstant.PACKET_TYPE_OK){
				if(logger.isInfoEnabled()){
					logger.info("3. authing success from server:"+this.getId()+",hashCode="+this.hashCode());
				}
				
				setAuthenticated(true);
				return;
			}else{
				if(logger.isInfoEnabled()){
					logger.info("3. authing fail from server:"+this.getId()+",hashCode="+this.hashCode());
				}
				
				setAuthenticated(false);
				return;
			}
		}
	}


	public Authenticator<HandshakePacket, AuthenPacket> getAuthenticator() {
		return authenticator;
	}


	public void setAuthenticator(
			Authenticator<HandshakePacket, AuthenPacket> authenticator) {
		this.authenticator = authenticator;
	}


	@Override
	public boolean checkValid() {
		return !this.isClosed() && this.isAuthenticated();
	}
	
}
