package com.meidusa.toolkit.plugins.autoconfig;

import com.meidusa.toolkit.common.util.ConfigUtil;
import com.meidusa.toolkit.plugins.autoconfig.descriptor.ConfigDescriptor;
import com.meidusa.toolkit.plugins.autoconfig.entry.ConfigEntry;
import com.meidusa.toolkit.plugins.autoconfig.entry.ConfigEntryFactory;
import com.meidusa.toolkit.plugins.autoconfig.entry.ConfigEntryFactoryImpl;
import com.meidusa.toolkit.plugins.autoconfig.wizard.text.ConfigWizardLoader;
import com.meidusa.toolkit.plugins.autoconfig.util.FileUtil;
import com.meidusa.toolkit.plugins.autoconfig.util.PatternSet;
import com.meidusa.toolkit.plugins.autoconfig.util.StringUtil;

import java.io.BufferedReader;
import java.io.File;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;

public class ConfigRuntimeImpl implements ConfigRuntime {
    private BufferedReader     in;
    private PrintWriter        out;
    private PrintWriter        err;
    private String             charset;
    private String             mode;
    private String             interactiveMode;
    private PatternSet         descriptorPatterns;
    private PatternSet         packagePatterns;
    private String[]           dests;
    private File[]             destFiles;
    private List<File>         dependencyFiles;
    private File               userPropertiesFile;
    private String             userPropertiesFileCharset;
    private Map                userProperties;
    private boolean            verbose;
    private File               tempdir;
    private String projectName;
    private Properties properties;
    public void setProjectPath(File projectPath) {
		this.projectPath = projectPath;
	}

	private File projectPath;
    public Properties getProperties() {
		return properties;
	}

	public void setProperties(Properties properties) {
		this.properties = properties;
	}
	public String getProjectName() {
		return projectName;
	}

	public void setProjectName(String projectName) {
		this.projectName = projectName;
	}

	private ConfigEntryFactory configEntryFactory = new ConfigEntryFactoryImpl(this);

    public ConfigRuntimeImpl() {
        this(System.in, System.out, System.err, null);
    }

    public ConfigRuntimeImpl(InputStream inputStream, OutputStream outStream,
                             OutputStream errStream, String charset) {
        this.interactiveMode = ConfigConstant.INTERACTIVE_AUTO;
        this.charset         = StringUtil.isEmpty(charset) ? ConfigConstant.DEFAULT_CHARSET
                                                           : charset;

        try {
            in  = new BufferedReader(new InputStreamReader(inputStream, this.charset));
            out = new PrintWriter(new OutputStreamWriter(outStream, this.charset), true);
            err = new PrintWriter(new OutputStreamWriter(errStream, this.charset), true);
        } catch (UnsupportedEncodingException e) {
            throw new ConfigException(e);
        }
    }

    public BufferedReader getIn() {
        return in;
    }

    public PrintWriter getOut() {
        return out;
    }

    public PrintWriter getErr() {
        return err;
    }

    public String getCharset() {
        return charset;
    }

    public PatternSet getDescriptorPatterns() {
        return descriptorPatterns;
    }

    public PatternSet getPackagePatterns() {
        return packagePatterns;
    }

    public String getInteractiveMode() {
        return interactiveMode;
    }

    public String getMode() {
        return mode;
    }

    public File[] getDestFiles() {
        return destFiles;
    }

    public File getUserPropertiesFile() {
        return userPropertiesFile;
    }

    public String getUserPropertiesFileCharset() {
        return userPropertiesFileCharset;
    }

    public boolean isVerbose() {
        return verbose;
    }

    public ConfigEntryFactory getConfigEntryFactory() {
        return configEntryFactory;
    }

    public void setDescriptorPatterns(String includes, String excludes) {
        this.descriptorPatterns = new PatternSet(includes, excludes);
    }

    public void setDescriptorPatterns(String[] includes, String[] excludes) {
        this.descriptorPatterns = new PatternSet(includes, excludes);
    }

    public void setPackagePatterns(String includes, String excludes) {
        this.packagePatterns = new PatternSet(includes, excludes);
    }

    public void setPackagePatterns(String[] includes, String[] excludes) {
        this.packagePatterns = new PatternSet(includes, excludes);
    }

    public void setInteractiveMode(String mode) {
        if (ConfigConstant.INTERACTIVE_AUTO.equals(mode)
                    || ConfigConstant.INTERACTIVE_ON.equals(mode)
                    || ConfigConstant.INTERACTIVE_OFF.equals(mode)) {
            this.interactiveMode = mode;
        }
    }

    public void setGuiMode() {
        mode = ConfigConstant.MODE_GUI;
    }

    public void setTextMode() {
        mode = ConfigConstant.MODE_TEXT;
    }

    public void setDests(String[] dests) {
        this.dests = dests;
    }

    public void setUserPropertiesFile(String userPropertiesFile, String charset) {
        String basedir = new File("").getAbsolutePath();

        this.userPropertiesFile        = new File(FileUtil.getPathBasedOn(basedir,
                                                                          userPropertiesFile));
        this.userPropertiesFileCharset = charset;
        
        File directory = this.userPropertiesFile.getParentFile();
        if(!directory.exists()){
        	
        	info("directory " + directory.getAbsolutePath() + " does not exists.");
        	info("directory " + directory.getAbsolutePath() + " is created.");
        	if(!directory.mkdirs()){
        		info("can't make directory.");
        	}
        }
        info("set defined properties: " + this.userPropertiesFile.getAbsolutePath() + "\n");
    }

    public void setVerbose() {
        this.verbose = true;
    }

    
    
    


	private void init() {
        // tempdir
        if (tempdir == null) {
            tempdir = new File("");
        }

        tempdir = tempdir.getAbsoluteFile();

        // dests
        dests = StringUtil.trimStringArray(dests);
        if (dests.length > 0) {
            destFiles = new File[dests.length];

            for (int i = 0; i < dests.length; i++) {
                destFiles[i] = new File(ConfigUtil.filter(dests[i], properties)).getAbsoluteFile();
            }
        } else {
        	if(this.getProjectPath() == null){
        		destFiles = new File[] { new File("").getAbsoluteFile() };
        	}else{
        		destFiles = new File[] { this.getProjectPath() };
        	}
        }

        // user properties file
        if (userPropertiesFile == null) {
            userPropertiesFile = new File(System.getProperty("user.home"), (getProjectName() == null?"maven":getProjectName())+".properties");
        }

        userPropertiesFile = userPropertiesFile.getAbsoluteFile();

        info("User-defined properties: " + userPropertiesFile.getAbsolutePath() + "\n");
    }

    public void debug(String message) {
        if (verbose) {
            getOut().println(message);
        }
    }

    public void info(String message) {
        getOut().println(message);
    }

    public void warn(String message) {
        getOut().println(message);
    }

    public void error(String message) {
        error(message, null);
    }

    public void error(Throwable cause) {
        error(null, cause);
    }

    public void error(String message, Throwable cause) {
        if (StringUtil.isBlank(message) && (cause != null)) {
            message = "ERROR: " + cause.getMessage();
        }

        getErr().println(message);

        if (verbose) {
            cause.printStackTrace(getErr());
            getErr().println();
        }
    }

    public void start() {
        start(null);
    }

    public void start(ConfigDescriptor inlineDescriptor) {
        init();

        if (inlineDescriptor == null) {
            List entries = new ArrayList(destFiles.length);

            for (int i = 0; i < destFiles.length; i++) {
                File        destFile = destFiles[i];
                ConfigEntry entry    = getConfigEntryFactory().create(new ConfigResource(destFile));

                entry.scan();

                if (!entry.isEmpty()) {
                    entries.add(entry);
                }
            }
            
            for(int i = 0; i < dependencyFiles.size(); i++) {
            	File dependencyFile = dependencyFiles.get(i);
                ConfigEntry entry    = getConfigEntryFactory().create(new ConfigResource(dependencyFile)); 
                entry.scan();

                if (!entry.isEmpty()) {
                    entries.add(entry);
                }
            }

            if (entries.isEmpty()) {
                info("Nothing to configure");
                return;
            }

            ConfigWizardLoader wizard = new ConfigWizardLoader(this, entries);

            wizard.loadAndStart();

            for (Iterator i = entries.iterator(); i.hasNext();) {
                ConfigEntry entry = (ConfigEntry) i.next();

                entry.generate();
            }
        } else {
            ConfigWizardLoader wizard = new ConfigWizardLoader(this, inlineDescriptor);

            wizard.loadAndStart();
        }
    }

	public File getProjectPath() {
		return projectPath;
	}

	public List<File> getDependencyFiles() {
		return dependencyFiles;
	}

	public void setDependencyFiles(List<File> dependencyFiles) {
		this.dependencyFiles = dependencyFiles;
	}
	
	
}
