/**
 * <pre>
 * 	This program is free software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version. 
 * 
 * 	This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * See the GNU General Public License for more details. 
 * 	You should have received a copy of the GNU General Public License along with this program; 
 * if not, write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * </pre>
 */
package com.meidusa.toolkit.net;

import java.io.EOFException;
import java.io.IOException;
import java.net.InetAddress;
import java.nio.ByteBuffer;
import java.nio.channels.CancelledKeyException;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.SocketChannel;
import java.nio.channels.WritableByteChannel;
import java.util.Date;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;

import com.meidusa.toolkit.net.io.PacketInputStream;
import com.meidusa.toolkit.net.io.PacketOutputStream;
import com.meidusa.toolkit.net.io.Sessionable;
import com.meidusa.toolkit.net.packet.Packet;
import com.meidusa.toolkit.net.packet.PacketFactory;
import com.meidusa.toolkit.net.util.ByteUtil;
import com.meidusa.toolkit.net.util.MemoryQueue;
import com.meidusa.toolkit.net.util.StringUtil;

/**
 * @author <a href=mailto:piratebase@sina.com>Struct chen</a>
 */
@SuppressWarnings("unchecked")
public abstract class Connection implements NetEventHandler {

    private static Logger       logger        = Logger.getLogger(Connection.class);
    private static Logger       packetDumpLogger        = Logger.getLogger("packetDump");
    private String id;
    public static final long    PING_INTERVAL = 90 * 1000L;
    protected static final long LATENCY_GRACE = 30 * 1000L;
    protected long pingIterval = PING_INTERVAL;
	protected ConnectionManager _cmgr;
    protected SelectionKey      _selkey;
    protected SocketChannel     _channel;
    protected long              _lastEvent;
    private long _createTime = System.currentTimeMillis();
    private MessageHandler<Connection>    _handler;
    
    protected final Object writeLock = new Object();
    protected final Lock        closeLock     = new ReentrantLock(false);
    protected final Lock        postCloseLock = new ReentrantLock(false);
    protected AtomicLong queueSize = new AtomicLong();
    private ExecutorService executor;
    public ExecutorService getExecutor() {
		return executor;
	}

	public void setExecutor(ExecutorService executor) {
		this.executor = executor;
	}

	protected boolean           closePosted   = false;
    private PacketInputStream   _fin;
    protected long lastMessageSent = System.currentTimeMillis();
    private PacketOutputStream  _fout;

    protected MemoryQueue<ByteBuffer> _outQueue     = new MemoryQueue<ByteBuffer>();
    private boolean             socketClosed  = false;
    protected String            host;
    protected int               port;
    private ReadableByteChannel _rChannel;
    private WritableByteChannel _wChannel;

    public Connection(SocketChannel channel, long createStamp){
        _channel = channel;
        if(channel instanceof ReadableByteChannel){
        	_rChannel = (ReadableByteChannel)channel;
        }
        if(channel instanceof WritableByteChannel){
        	_wChannel = (WritableByteChannel)channel;
        }
        
        host = channel.socket().getInetAddress().getHostAddress();
		port = channel.socket().getPort();
		
        this.setId(host + ":" + port);
        _lastEvent = createStamp;
        this._createTime = createStamp;
    }
    
    public PacketFactory<? extends Packet> getPacketFactory() {
        return packetFactory;
    }

    public void setPacketFactory(PacketFactory<? extends Packet> packetFactory) {
        this.packetFactory = packetFactory;
    }

    protected PacketFactory<? extends Packet> packetFactory;
    
    /**
     *  host+":"+port
     * @return
     */
    public String getId(){
    	return id;
    }
    
    public void setId(String id){
    	this.id = id;
    }
    
    public long getPingIterval() {
		return pingIterval;
	}

	public void setPingIterval(long pingIterval) {
		this.pingIterval = pingIterval;
	}



    /**
     * when connection registed to ConnectionManager, {@link #init()} will invoked.
     * 
     * @see <code> {@link ConnectionManager#registerConnection(Connection, int)}</code>
     */
    protected void init() {
    	if(_outQueue.size()>0){
    		_selkey.interestOps(_selkey.interestOps() | SelectionKey.OP_WRITE);
        }
    }

    public void setOutQueueMaxSize(int maxSize){
    	this._outQueue.setMaxSize(maxSize);
    }
    public void setConnectionManager(ConnectionManager cmgr) {
        this._cmgr = cmgr;
    }

    /**
     *  SocketChannel ص SelectionKey
     */
    public void setSelectionKey(SelectionKey selkey) {
        this._selkey = selkey;
    }

    /**
     * SocketChannel ص Selection Key
     */
    public SelectionKey getSelectionKey() {
        return _selkey;
    }

    /**
     * һSocketChannel
     * 
     * @return
     */
    public SocketChannel getChannel() {
        return _channel;
    }

    
	public void setMessageHandler(MessageHandler handler) {
        _handler = handler;
    }

    public MessageHandler getMessageHandler() {
        return _handler;
    }

    protected void inheritStreams(Connection other) {
        _fin = other._fin;
        _fout = other._fout;
    }

    /**
     * ж  Ƿر
     * 
     * @return
     */
    public boolean isClosed() {
        closeLock.lock();
        try {
            return socketClosed || (this.getChannel() == null || !this.getChannel().isOpen());
        } finally {
            closeLock.unlock();
        }

    }

    /**
     * رյǰӣҴConnectionManagerɾӡ
     */
    protected void close(Exception exception) {
        closeLock.lock();
        try {
            // we shouldn't be closed twice
            if (isClosed()) {
                return;
            }
            socketClosed = true;
        } finally {
            closeLock.unlock();
        }

        _outQueue.clear();
        if (this.getMessageHandler() instanceof Sessionable) {
            Sessionable session = (Sessionable) this.getMessageHandler();
            logger.error(this + ",closeSocket,and endSession,handler=" + session);
            session.endSession();
        }

        if (_selkey != null) {
            _selkey.attach(null);
            Selector selector = _selkey.selector();
            _selkey.cancel();
            // wake up again to trigger thread death
            selector.wakeup();

            _selkey = null;
        }
        if(logger.isInfoEnabled()){
        	logger.info("Closing channel " + this + ",id="+id+", manager="+this._cmgr.getName());
        }
        try {
            _channel.close();
        } catch (IOException ioe) {
            logger.warn("Error closing connection [conn=" + this + ",id="+id+", manager="+this._cmgr.getName()+", error=" + ioe + "].");
        }

        if (exception != null) {
            _cmgr.connectionFailed(this, exception);
        } else {
            _cmgr.connectionClosed(this);
        }
    }

    /**
     * POST-->Queue->close->(_cmgr.connectionClosed( notify observer))-->closeSocket()
     * ṩãֻǵݽرոӵ󡣾رսConnection Manager
     */
    public void postClose(Exception exception) {
        if (closePosted) {
            return;
        }
        
        
        postCloseLock.lock();
        try {
            
            this._cmgr.closeConnection(this, exception);
        } finally {
            postCloseLock.unlock();
        }
    }

    /**
     * Ӵݻ쳣Ҫرӵµôη
     * 
     * @param ioe
     */
    public void handleFailure(Exception ioe) {
        // Ѿر
        if (isClosed()) {
            logger.warn("Failure reported on closed connection " + this + ".", ioe);
            return;
        }
        postClose(ioe);
    }

    public int handleEvent(long when) {
        int bytesInTotle = 0;
        try {
            if (_fin == null) {
                _fin = createPacketInputStream();
            }

            while (_channel != null && _channel.isOpen() && _fin.readPacket(_rChannel)) {
                int bytesIn = 0;
                // ¼һηʱ
                _lastEvent = when;
                /**
                 * õFramedInputStream ֽ
                 */
                bytesIn = _fin.available();
                bytesInTotle += bytesIn;
                byte[] msg = new byte[bytesIn];
                _fin.read(msg);
                messageProcess(msg);
            }
        } catch (EOFException eofe) {
            // close down the socket gracefully
            handleFailure(eofe);
        } catch (IOException ioe) {
            // don't log a warning for the ever-popular "the client dropped the
            // connection" failure
            String msg = ioe.getMessage();

            if (msg == null || msg.indexOf("reset by peer") == -1) {
                logger.info("Error reading message from socket [channel=" + StringUtil.safeToString(_channel) + ", error=" + ioe + "].", ioe);
            }
            // deal with the failure
            handleFailure(ioe);
        }

        return bytesInTotle;
    }

    protected void messageProcess(final byte[] msg) {
    	if(closePosted || isClosed()){
    		return;
    	}
    	if(executor != null){
    	  queueSize.incrementAndGet();
    	  Runnable task = new Runnable() {
    		  long start = System.currentTimeMillis();
              public void run() {
                  try {
                  	 Connection.this.getMessageHandler().handleMessage(Connection.this, msg);
                  } finally {
                  	queueSize.decrementAndGet();
                  	logInvokeTime(start,msg);
                  }
              }
          };
          executor.execute(task);
        }else{
        	long start = System.currentTimeMillis();
        	try{
        		Connection.this.getMessageHandler().handleMessage(this, msg);
        	}finally{
        		logInvokeTime(start,msg);
        	}
        }
    }
    
    protected void logInvokeTime(long start,byte[] msg){
    	long cost = (System.currentTimeMillis()-start);
      	if(cost <=11500){
          	if(logger.isDebugEnabled()){
          		StringBuffer buffer = new StringBuffer();
          		buffer.append(Connection.this.getId());
          		buffer.append(" request Completed, time=");
          		buffer.append(cost);
          		if(packetDumpLogger.isDebugEnabled() && msg != null){
          			buffer.append("\r\n").append("--------------------").append("\r\n");
          			buffer.append(ByteUtil.toHex(msg, 0, msg.length));
          			buffer.append("\r\n").append("--------------------").append("\r\n");
          		}
          		
          		logger.debug(buffer.toString());
          	}
      	}else{
      		if(logger.isEnabledFor(Level.WARN)){
      			StringBuffer buffer = new StringBuffer();
          		buffer.append(Connection.this.getId());
          		buffer.append(" request Completed, time=");
          		buffer.append(cost);
          		if(packetDumpLogger.isDebugEnabled() && msg != null){
          			buffer.append("\r\n").append("--------------------").append("\r\n");
          			buffer.append(ByteUtil.toHex(msg, 0, msg.length));
          			buffer.append("\r\n").append("--------------------").append("\r\n");
          		}
      			logger.warn(buffer.toString());
      		}
      		
      	}
    }

    public long getQueueSize(){
    	return queueSize.get();
    }
    
    
    /**
     * write buffer in non-blocking mode , return true if there no remaining buffer and out queue is empty
     * @param buff
     * @throws IOException
     */
    public boolean doWrite() throws IOException {
    	if (closePosted || socketClosed) {
            return false;
        }
    	SelectionKey key = getSelectionKey();
    	
        if(key == null){
    		return false;
    	}
        if (!key.isValid()) {
            handleFailure(new java.nio.channels.CancelledKeyException());
            return false;
        }
        
        synchronized (writeLock) {
        	boolean isCompleted = true;
            ByteBuffer buffer = null;
            int message = 0;
            while ((buffer = _outQueue.getNonBlocking()) != null) {
            	if(_wChannel != null){
            		int wrote = _wChannel.write(buffer);
	                if (buffer.remaining() > 0) {
	                	_outQueue.prepend(buffer);
	                	isCompleted = false;
	                	break;
	                } else {
	                    // buffer.clear();
	                    message++;
	                }
            	}
            }
    		if(isCompleted){
    			synchronized (_outQueue) {
    				if(_outQueue.size()>0){
	                    key.interestOps(key.interestOps() | SelectionKey.OP_WRITE);
        				return false;
        			}else{
        				key.interestOps(key.interestOps() & ~SelectionKey.OP_WRITE);
        				return true;
        			}
				}
    		}else{
    			key.interestOps(key.interestOps() | SelectionKey.OP_WRITE);
    			return false;
    		}
        }
    }
    
    public synchronized void postMessage(byte[] msg) {
        PacketOutputStream _framer = getPacketOutputStream();
        _framer.resetPacket();
        try {
            _framer.write(msg);
            ByteBuffer buffer = _framer.returnPacketBuffer();
            postMessage(buffer);
        } catch (IOException e) {
            this._cmgr.connectionFailed(this, e);
        }
    }

    public void postMessage(ByteBuffer msg) {
    	/*if(_outQueue.size()>0 && (this.getSelectionKey().interestOps() & SelectionKey.OP_WRITE) == 0){
    		if(_cmgr != null){
            	_cmgr.invokeConnectionWriteMessage(this);
            }
    	}*/
    	lastMessageSent = System.currentTimeMillis();
        _outQueue.append(msg);
        try{
        	this.doWrite();
        } catch (IOException ioe) {
            handleFailure(ioe);
        } catch(CancelledKeyException ce){
        	handleFailure(ce);
        }
        /*if(_cmgr != null){
        	_cmgr.invokeConnectionWriteMessage(this);
        }*/
    }
    
    


    public boolean checkIdle(long now) {
        long idleMillis = now - _lastEvent;
        if (idleMillis < this.getPingIterval() + LATENCY_GRACE) {
            return false;
        }
        if (isClosed()) {
            return true;
        }
        return true;
    }
    
    
    /**
     * һյĿӵݰʱ
     * @return
     */
    public long getLastEventTime(){
    	return _lastEvent;
    }
    
    /**
     * ǷҪping
     * @return
     */
    public boolean needPing(long now){
    	return false;
    }
    
    protected abstract PacketInputStream createPacketInputStream();

    protected abstract PacketOutputStream createPacketOutputStream();

    protected PacketOutputStream getPacketOutputStream() {
        if (_fout == null) {
            _fout = createPacketOutputStream();
            _fout.initHeader();
        }
        return this._fout;

    }

    protected PacketInputStream getPacketInputStream() {
        if (_fin == null) {
            _fin = createPacketInputStream();
        }
        return this._fin;
    }

	public ConnectionManager getConnectionManager() {
		return this._cmgr;
	}
	
    public InetAddress getInetAddress() {
        return (_channel == null) ? null : _channel.socket().getInetAddress();
    }
    
    public String toString(){
    	StringBuffer buffer = new StringBuffer();
    	buffer.append("class=").append(this.getClass()).append(",socket=").append(getId())
    	.append(",outQueue=").append(_outQueue.size()).append(",createTime=").append(new Date(_createTime));
    	return buffer.toString();
    }
}
