/**
 * <pre>
 * 	This program is free software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version. 
 * 
 * 	This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * See the GNU General Public License for more details. 
 * 	You should have received a copy of the GNU General Public License along with this program; 
 * if not, write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * </pre>
 */
package com.meidusa.toolkit.net;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.nio.channels.SelectableChannel;
import java.nio.channels.SelectionKey;
import java.nio.channels.ServerSocketChannel;
import java.nio.channels.SocketChannel;
import java.util.concurrent.atomic.AtomicLong;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;


/**
 * ָһ˿,һserverSocket. ServerSocketConnection
 * 
 * @author <a href=mailto:piratebase@sina.com>Struct chen</a>
 */
public class ServerableConnectionManager extends AuthingableConnectionManager {
	class AuthingConnectionObserver implements ConnectionObserver{

		public void connectionClosed(Connection conn) {
			
		}

		public void connectionEstablished(Connection connection) {
			if(connection instanceof AuthingableConnection){
            	((AuthingableConnection)connection).setAuthenticateProvider(ServerableConnectionManager.this.getAuthenticateProvider());
            	((AuthingableConnection)connection).beforeAuthing();
            }
		}

		public void connectionFailed(Connection conn, Exception fault) {
			
		}
		
	}
    protected static Logger       log = Logger.getLogger(ServerableConnectionManager.class);
    private ConnectionObserver connObserver = new AuthingConnectionObserver(); 
    protected int                 port;
    protected ServerSocketChannel ssocket;
    protected String              ipAddress;
    protected ConnectionFactory   connFactory;
    private ConnectionManager managers[];
    private int subManagerSize = Runtime.getRuntime().availableProcessors();
    private AtomicLong counter = new AtomicLong();
    private int backlog = 128;
    public int getBacklog() {
		return backlog;
	}

	public void setBacklog(int maxBacklogQueue) {
		this.backlog = maxBacklogQueue;
	}
	
	public int getSubManagerSize() {
		return subManagerSize;
	}

	public void setSubManagerSize(int subManagerSize) {
		this.subManagerSize = subManagerSize;
	}

	public ServerableConnectionManager() throws IOException{
    }

    public ServerableConnectionManager(String name, int port) throws IOException{
        super(name);
        this.port = port;
    }

    public ServerableConnectionManager(String name, String ipAddress, int port) throws IOException{
        this(name,port);
        this.ipAddress = ipAddress;
    }

    public void setConnectionFactory(ConnectionFactory connFactory) {
        this.connFactory = connFactory;
        if (connFactory instanceof AbstractConnectionFactory) {
            AbstractConnectionFactory afactory = (AbstractConnectionFactory) connFactory;
            if (afactory.getConnectionManager() == null) {
                afactory.setConnectionManager(this);
            }
        }
    }

    protected void initSubManager(ConnectionManager manager){
    	manager.setExecutor(this.getExecutor());
    	manager.setDaemon(true);
    	manager._observers = this._observers;
    	manager.setIdleCheckTime(this.getIdleCheckTime());
    	manager.setMaxQueueSizePerClient(this.getMaxQueueSizePerClient());
    }
    
    protected ConnectionManager newInstanceSubManager(int index) throws IOException{
    	return new ConnectionManager(this.getName()+"-"+index);
    }
    
    // documentation inherited
    protected void willStart() {
    	this.addConnectionObserver(connObserver);
    	managers = new ConnectionManager[subManagerSize>0?subManagerSize:Runtime.getRuntime().availableProcessors()];
    	for(int i=0;i<managers.length;i++){
    		try {
    			managers[i] = newInstanceSubManager(i);
    			initSubManager(managers[i]);
				managers[i].start();
			} catch (IOException ioe) {
				log.error("create sub manager error", ioe);
				throw new RuntimeException(ioe);
			}
    	}
    	
        super.willStart();
        try {
            // create a listening socket and add it to the select set
            ssocket = ServerSocketChannel.open();
            ssocket.configureBlocking(false);

            InetSocketAddress isa = null;
            if (ipAddress != null) {
                isa = new InetSocketAddress(ipAddress, port);
            } else {
                isa = new InetSocketAddress(port);
            }
            Level level = log.getLevel();
            log.setLevel(Level.INFO);
            
            ssocket.socket().bind(isa,backlog);
            	registerServerChannel(ssocket);
            log.info("Server listening on " + isa + ".");
            log.setLevel(level);

        } catch (IOException ioe) {
            log.error("Failure listening to socket on port '" + port + "'.", ioe);
            System.exit(-1);
        }
    }
    
    protected void registerServerChannel(final ServerSocketChannel listener) throws IOException {

        // register this listening socket and map its select key to a net event
        // handler that will
        // accept new connections
        NetEventHandler serverNetEvent = new NetEventHandler() {

            private SelectionKey key;

            public int handleEvent(long when) {
            	Connection conn = null;
            	do{
            		conn = acceptConnection(listener);
            	}while(conn != null);
                return 0;
            }

            public boolean checkIdle(long now) {
                return false; // we're never idle
            }

            public SelectionKey getSelectionKey() {
                return key;
            }

            public void setSelectionKey(SelectionKey key) {
                this.key = key;
            }

            public boolean doWrite() {
                return true;
            }

			public boolean needPing(long now) {
				return false;
			}

			public void ping(long now) {
				
			}

			public long getQueueSize() {
				return -1;
			}
        };
        SelectionKey sk = listener.register(_selector, SelectionKey.OP_ACCEPT, serverNetEvent);
        serverNetEvent.setSelectionKey(sk);
        postRegisterNetEventHandler(serverNetEvent, SelectionKey.OP_ACCEPT);
    }

    protected Connection acceptConnection(ServerSocketChannel listener) {
        SocketChannel channel = null;
        try {
            channel = listener.accept();
            if (channel == null) {
        		//log.info("Psych! Got ACCEPT_READY, but no connection.");
                return null;
            }
            if (!(channel instanceof SelectableChannel)) {
                try {
                    log.warn("Provided with un-selectable socket as result of accept(), can't " + "cope [channel=" + channel + "].");
                } catch (Error err) {
                    log.warn("Un-selectable channel also couldn't be printed.");
                }
                // stick a fork in the socket
                channel.socket().close();
                return null;
            }
            Connection connection = connFactory.createConnection(channel, System.currentTimeMillis());
            /*if(connection instanceof AuthingableConnection){
            	((AuthingableConnection)connection).setAuthenticator(this.getAuthenticator());
            	((AuthingableConnection)connection).beforeAuthing();
            }*/
            this.postRegisterNetEventHandler(connection,SelectionKey.OP_READ);
            return connection;
        } catch (Exception e) {
            if (channel != null) {
                try {
                    channel.socket().close();
                } catch (IOException ioe) {
                    log.warn("Failed closing aborted connection: " + ioe);
                }
            }
            return null;
        }
    }

    public void postRegisterNetEventHandler(NetEventHandler handler, int key) {
    	if(handler instanceof Connection){
    		long count = counter.incrementAndGet();
    		managers[(int)count% managers.length].postRegisterNetEventHandler(handler, key);
    	}else{
    		super.postRegisterNetEventHandler(handler, key);
    	}
    }
    
    public void closeAll() {
        super.closeAll();
        try {
            ssocket.close();
        } catch (IOException e) {
        }
    }

    public int getPort() {
        return port;
    }

    public void setPort(int port) {
        this.port = port;
    }

    public String getIpAddress() {
        return ipAddress;
    }

    public void setIpAddress(String ipAddress) {
        this.ipAddress = ipAddress;
    }

    public synchronized void shutdown(){
    	super.shutdown();
    	try {
			ssocket.close();
		} catch (IOException e) {
		}
    	
    	if(managers != null){
    		for(ConnectionManager conM : managers){
    			conM.shutdown();
    		}
    	}
    }
}
